package ru.yandex.solomon.coremon.meta.db;

import java.util.concurrent.atomic.AtomicIntegerFieldUpdater;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.util.concurrent.AtomicRandomFieldUpdater;
import ru.yandex.stockpile.client.StockpileClient;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.shard.StockpileMetricId;
import ru.yandex.stockpile.client.shard.StockpileShardId;

/**
 * @author Sergey Polovko
 */
final class StockpileMetricIdProviderStatsAware implements StockpileMetricIdProvider {

    private static final AtomicRandomFieldUpdater<StockpileMetricIdProviderStatsAware> rndUpdater =
        AtomicRandomFieldUpdater.newUpdater(StockpileMetricIdProviderStatsAware.class, "rnd");

    private static final AtomicIntegerFieldUpdater<StockpileMetricIdProviderStatsAware> shardIdUpdater =
        AtomicIntegerFieldUpdater.newUpdater(StockpileMetricIdProviderStatsAware.class, "shardId");

    private final ShardIdToAtomicInteger stats;
    private final StockpileClient stockpileClient;
    private final int maxCount;
    private volatile long rnd;
    private volatile int shardId;

    StockpileMetricIdProviderStatsAware(int numId, String host, int maxCount, ShardIdToAtomicInteger stats, StockpileClient stockpileClient) {
        this.stats = stats;
        this.stockpileClient = stockpileClient;
        this.maxCount = maxCount;
        rndUpdater.setSeed(this, longHashCode(numId, host));
        this.shardId = StockpileShardId.INVALID_SHARD_ID;
    }

    private int nextShardId() {
        int shardCount = stockpileClient.getTotalShardsCount();
        if (shardCount == 0) {
            throw new IllegalStateException("Unknown stockpile shard count");
        }
        return 1 + rndUpdater.nextInt(this, shardCount);
    }

    private int actualShardId() {
        final int shardId = this.shardId;
        if (shardId == StockpileShardId.INVALID_SHARD_ID) {
            // init shard count
            shardIdUpdater.compareAndSet(this, shardId, nextShardId());
            return this.shardId;
        }
        return shardId;
    }

    private static long longHashCode(int a, String b) {
        long h = a;
        for (int i = 0; i < b.length(); i++) {
            h = 31 * h + b.charAt(i);
        }
        return h;
    }

    @Override
    public int shardId(Labels labels) {
        final int shardId = actualShardId();

        final int count = stats.incrementAndGet(shardId);
        if (count >= maxCount) {
            // there too many metrics were stored in that stockpile shard,
            // so we have to find another one
            shardIdUpdater.compareAndSet(this, shardId, nextShardId());
        }

        // anyway use this shard to avoid long cycles of finding better one
        return shardId;
    }

    @Override
    public StockpileMetricId metricId(Labels labels) {
        return new StockpileMetricId(shardId(labels), StockpileLocalId.random());
    }

    @Override
    public String toString() {
        return "StatsAware{shardId='" + shardId + '\'' + ", rnd=" + rnd + '}';
    }
}
