package ru.yandex.solomon.coremon.meta.db.memory;

import java.util.Collection;
import java.util.Comparator;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentNavigableMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.annotations.VisibleForTesting;

import ru.yandex.monlib.metrics.labels.LabelAllocator;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.CoremonMetricArray;
import ru.yandex.solomon.coremon.meta.FileCoremonMetric;
import ru.yandex.solomon.coremon.meta.db.DeletedMetricsDao;
import ru.yandex.solomon.coremon.meta.db.ydb.LabelListSortedSerialize;

import static com.google.common.base.Preconditions.checkArgument;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
@VisibleForTesting
public class InMemoryDeletedMetricsDao implements DeletedMetricsDao {

    private final ConcurrentNavigableMap<Pk, CoremonMetric> table = new ConcurrentSkipListMap<>();
    public volatile Supplier<CompletableFuture<?>> beforeSupplier;

    @Override
    public CompletableFuture<Void> createSchema() {
        return async(() -> null);
    }

    @Override
    public CompletableFuture<Long> count(String operationId, int numId) {
        return async(() -> {
            var t = table.subMap(
                Pk.of(operationId, numId, null),
                true,
                Pk.of(operationId, numId + 1, null),
                false);

            return (long) t.size();
        });
    }

    @Override
    public CompletableFuture<Void> find(
        String operationId,
        int numId,
        int limit,
        @Nullable Labels lastKey,
        CoremonMetricArray buffer,
        LabelAllocator labelAllocator)
    {
        return async(() -> {
            var t = table.subMap(
                Pk.of(operationId, numId, lastKey),
                false,
                Pk.of(operationId, numId + 1, null),
                false);

            t.values().stream()
                .limit(limit)
                .forEach(buffer::add);

            return null;
        });
    }

    @Override
    public CompletableFuture<Void> bulkUpsert(
        String operationId,
        int numId,
        CoremonMetricArray metrics)
    {
        return async(() -> {
            checkArgument(!metrics.isEmpty(), "metrics for bulkUpsert should not be empty");

            metrics.stream()
                .map(FileCoremonMetric::new)
                .forEach(m -> table.put(Pk.of(operationId, numId, m.getLabels()), m));

            return null;
        });
    }

    @Override
    public CompletableFuture<Void> delete(
        String operationId,
        int numId,
        Collection<Labels> keys)
    {
        return async(() -> {
            for (var key : keys) {
                table.remove(Pk.of(operationId, numId, key));
            }
            return null;
        });
    }

    @Override
    public CompletableFuture<Long> deleteBatch(String operationId, int numId) {
        return async(() -> {
            var t = table.subMap(
                Pk.of(operationId, numId, null),
                true,
                Pk.of(operationId, numId + 1, null),
                false);

            var count = 0L;
            var it = t.entrySet().iterator();
            while (it.hasNext() && count < 1000) {
                it.next();
                it.remove();
                count++;
            }

            return count;
        });
    }

    public Collection<CoremonMetric> metrics() {
        return table.values();
    }

    public void putAll(String operationId, int numId, Collection<CoremonMetric> metrics) {
        try (var batch = new CoremonMetricArray(metrics.toArray(CoremonMetric[]::new))) {
            bulkUpsert(operationId, numId, batch).join();
        }
    }

    private <T> CompletableFuture<T> async(Supplier<T> fn) {
        return before().thenApplyAsync(ignore -> fn.get());
    }

    private CompletableFuture<?> before() {
        var copy = beforeSupplier;
        if (copy == null) {
            return CompletableFuture.completedFuture(null);
        }

        return copy.get();
    }

    private record Pk(String operationId, int numId, String labels) implements Comparable<Pk> {

        private static final Comparator<Pk> cmp =
            Comparator.comparing(Pk::operationId)
                .thenComparing(Pk::numId)
                .thenComparing(Pk::labels);

        static Pk of(String operationId, int numId, @Nullable Labels labels) {
            var labelsStr = Optional.ofNullable(labels)
                .map(LabelListSortedSerialize::format)
                .orElse("");

            return new Pk(operationId, numId, labelsStr);
        }

        @Override
        public int compareTo(Pk that) {
            return cmp.compare(this, that);
        }
    }
}
