package ru.yandex.solomon.coremon.meta.db.memory;

import java.util.Collection;
import java.util.List;
import java.util.OptionalLong;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Lists;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.CoremonMetricArray;
import ru.yandex.solomon.coremon.meta.FileCoremonMetric;
import ru.yandex.solomon.coremon.meta.db.MetricsDao;

import static java.util.concurrent.CompletableFuture.completedFuture;


/**
 * Note: Only for tests
 *
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class InMemoryMetricsDao implements MetricsDao {

    private final CompletableFuture<Void> initFuture;
    private final ConcurrentMap<Labels, CoremonMetric> recordByLabels = new ConcurrentHashMap<>();
    public volatile Supplier<CompletableFuture<?>> beforeSupplier;
    public volatile long responseDelayMillis;

    public InMemoryMetricsDao() {
        this(completedFuture(null));
    }

    InMemoryMetricsDao(CompletableFuture<Void> initFuture) {
        this.initFuture = initFuture;
    }

    public void add(List<CoremonMetric> metrics) {
        for (var metric : metrics) {
            recordByLabels.put(metric.getLabels(), metric);
        }
    }

    public Collection<CoremonMetric> metrics() {
        return recordByLabels.values();
    }

    @Override
    public CompletableFuture<Void> createSchema() {
        return delayVoid();
    }

    @Override
    public CompletableFuture<Void> dropSchema() {
        return delayVoid();
    }

    @Override
    public CompletableFuture<Long> getMetricCount() {
        return delay(() -> (long) recordByLabels.size());
    }

    @Override
    public CompletableFuture<Long> findMetrics(Consumer<CoremonMetricArray> consumer, OptionalLong metricCount) {
        return delay(() -> {
            var root = new CompletableFuture<Long>();
            var future = root;
            var copy = List.copyOf(recordByLabels.values());
            int maxBatchSize = Math.max(copy.size() / 3, 1001);
            int batchSize = ThreadLocalRandom.current().nextInt(1000, maxBatchSize);
            for (var batch : Lists.partition(copy, batchSize)) {
                future = future.thenCompose(size -> delay(() -> {
                    CoremonMetric[] s = batch.toArray(new CoremonMetric[0]);
                    try (var result = new CoremonMetricArray(s)) {
                        consumer.accept(result);
                    }
                    return size + batch.size();
                }));
            }
            root.complete(0L);
            return future;
        }).thenCompose(future -> future);
    }

    @Override
    public CompletableFuture<CoremonMetricArray> replaceMetrics(CoremonMetricArray metrics) {
        return delay(() -> {
            var newMetrics = new CoremonMetricArray(metrics.size());
            metrics.stream().forEach(s -> {
                if (!recordByLabels.containsKey(s.getLabels())) {
                    // create copy of metric to avoid keeping reference to a metrics array
                    FileCoremonMetric copy = new FileCoremonMetric(s);
                    recordByLabels.put(s.getLabels(), copy);
                    newMetrics.add(copy);
                } else {
                    var oldMetric = recordByLabels.get(s.getLabels());
                    if (oldMetric != null && oldMetric.getType() != s.getType()
                            && oldMetric.getShardId() == s.getShardId()
                            && oldMetric.getLocalId() == s.getLocalId()) {

                        FileCoremonMetric copy = new FileCoremonMetric(s);
                        recordByLabels.put(s.getLabels(), copy);
                        newMetrics.add(copy);
                    }
                }
            });
            return newMetrics;
        });
    }

    @Override
    public CompletableFuture<Void> deleteMetrics(Collection<Labels> keys) {
        return delay(() -> {
            for (Labels key : keys) {
                recordByLabels.remove(key);
            }
            return null;
        });
    }

    @Override
    public CompletableFuture<Long> deleteMetricsBatch() {
        return delay(() -> {
            int maxBatchSize = Math.max(recordByLabels.size() / 3, 1001);
            int batchSize = ThreadLocalRandom.current().nextInt(1000, maxBatchSize);

            long result = 0;
            var it = recordByLabels.values().iterator();
            while (it.hasNext() && result < batchSize) {
                it.next();
                it.remove();
                result++;
            }

            return result;
        });
    }

    private <T> CompletableFuture<T> delay(Supplier<T> fn) {
        return before().thenCompose(ignore -> {
            var future = new CompletableFuture<T>();
            CompletableFuture.delayedExecutor(responseDelayMillis, TimeUnit.MILLISECONDS)
                    .execute(() -> CompletableFutures.safeComplete(future, fn));
            return future;
        });
    }

    private CompletableFuture<Void> delayVoid() {
        return delay(() -> null);
    }

    private CompletableFuture<?> before() {
        var copy = beforeSupplier;
        if (copy == null) {
            return initFuture;
        }

        return initFuture.thenCompose(i -> copy.get());
    }
}
