package ru.yandex.solomon.coremon.meta.db.memory;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.labels.LabelAllocator;
import ru.yandex.solomon.coremon.meta.db.MetricsDaoFactory;

import static java.util.concurrent.CompletableFuture.completedFuture;


/**
 * NOTE: Only for tests
 *
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class InMemoryMetricsDaoFactory implements MetricsDaoFactory, AutoCloseable {

    private final ConcurrentMap<Integer, InMemoryMetricsDao> shardDaoById = new ConcurrentHashMap<>();
    private final ConcurrentMap<Integer, CompletableFuture<Void>> shardInitFutures = new ConcurrentHashMap<>();
    private volatile boolean suspendShardOnCreate;
    private volatile long responseDelayMillis;

    @Override
    public InMemoryMetricsDao create(int numId, LabelAllocator labelAllocator) {
        return shardDaoById.computeIfAbsent(numId, ignore -> {
            var initFuture = shardInitFutures.computeIfAbsent(
                numId,
                i -> suspendShardOnCreate ? new CompletableFuture<>() : completedFuture(null));

            var dao = new InMemoryMetricsDao(initFuture);
            dao.responseDelayMillis = responseDelayMillis;
            return dao;
        });
    }

    public void setReplayDelay(long value, TimeUnit unit) {
        responseDelayMillis = unit.toMillis(value);
        shardDaoById.values().forEach(dao -> {
            dao.responseDelayMillis = responseDelayMillis;
        });
    }

    public void setSuspendShardInitOnCreate(boolean value) {
        this.suspendShardOnCreate = value;
    }

    public void resumeShardInit(int numId) {
        shardInitFutures.get(numId).completeAsync(() -> null);
    }

    @Override
    public void close() {
        shardInitFutures.values().forEach(f -> f.complete(null));
    }
}
