package ru.yandex.solomon.coremon.meta.db.ydb;

import java.time.Duration;
import java.util.Collection;
import java.util.concurrent.CompletableFuture;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.primitives.UnsignedInteger;
import com.yandex.ydb.core.Result;
import com.yandex.ydb.table.SessionRetryContext;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.settings.AutoPartitioningPolicy;
import com.yandex.ydb.table.settings.BulkUpsertSettings;
import com.yandex.ydb.table.settings.CreateTableSettings;
import com.yandex.ydb.table.settings.ExecuteDataQuerySettings;
import com.yandex.ydb.table.settings.PartitioningPolicy;
import com.yandex.ydb.table.settings.ReadTableSettings;
import com.yandex.ydb.table.transaction.TxControl;
import com.yandex.ydb.table.values.OptionalValue;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.TupleValue;

import ru.yandex.monlib.metrics.labels.LabelAllocator;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.coremon.meta.CoremonMetricArray;
import ru.yandex.solomon.coremon.meta.db.DeletedMetricsDao;
import ru.yandex.solomon.coremon.meta.db.ydb.YdbDeletedMetricsTable.DeletedMetricsReader;

import static com.google.common.base.Preconditions.checkArgument;
import static com.yandex.ydb.core.utils.Async.safeCall;
import static com.yandex.ydb.table.values.PrimitiveValue.uint32;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
class YdbDeletedMetricsDao implements DeletedMetricsDao {

    private static final int MAX_RETRIES = 10;
    private static final Duration SESSION_SUPPLY_TIMEOUT = Duration.ofSeconds(30);

    private static final Duration READ_TABLE_TIMEOUT = Duration.ofMinutes(1);
    private static final Duration BULK_UPSERT_TIMEOUT = Duration.ofMinutes(1);

    private final SessionRetryContext retryCtx;
    private final YdbDeletedMetricsQuery query;

    YdbDeletedMetricsDao(TableClient tableClient, String root) {
        this.retryCtx = SessionRetryContext.create(tableClient)
            .maxRetries(MAX_RETRIES)
            .sessionSupplyTimeout(SESSION_SUPPLY_TIMEOUT)
            .build();
        this.query = new YdbDeletedMetricsQuery(root);
    }

    @Override
    public CompletableFuture<Void> createSchema() {
        var partitioningPolicy = new PartitioningPolicy()
            .setAutoPartitioning(AutoPartitioningPolicy.AUTO_SPLIT_MERGE);
        var settings = new CreateTableSettings()
            .setPartitioningPolicy(partitioningPolicy);

        return retryCtx.supplyStatus(
                session -> session.createTable(
                    tablePath(),
                    YdbDeletedMetricsTable.description(),
                    settings))
            .thenAccept(status -> status.expect("can't create table " + tablePath()));
    }

    @Override
    public CompletableFuture<Long> count(String operationId, int numId) {
        return safeCall(() -> {
            var params = Params.of(
                "$operationId", utf8(operationId),
                "$shardId", uint32(numId));

            return execute(query.count, params)
                .thenApply(result -> {
                    var rs = result.expect(onFailureMessage("count", operationId, numId)).getResultSet(0);
                    if (!rs.next()) {
                        return 0L;
                    }

                    return rs.getColumn(0).getUint64();
                });
        });
    }

    @Override
    public CompletableFuture<Void> find(
        String operationId,
        int numId,
        int limit,
        @Nullable Labels lastKey,
        CoremonMetricArray buffer,
        LabelAllocator labelAllocator)
    {
        return safeCall(() -> {
            var settings = prepareReadTableSettings(operationId, numId, limit, lastKey);
            var reader = new DeletedMetricsReader(labelAllocator, buffer);

            return retryCtx.supplyStatus(session -> {
                    // reset reader before every try to clear previous read attempt effect
                    reader.reset();
                    return session.readTable(tablePath(), settings, reader);
                })
                .thenAccept(status -> status.expect(onFailureMessage("read table", operationId, numId)));
        });
    }

    @Override
    public CompletableFuture<Void> bulkUpsert(
        String operationId,
        int numId,
        CoremonMetricArray metrics)
    {
        return safeCall(() -> {
            checkArgument(!metrics.isEmpty(), "metrics for bulkUpsert should not be empty");

            var listValue = YdbDeletedMetricsTable.metricsToListValue(operationId, numId, metrics);

            return retryCtx.supplyStatus(session -> session.executeBulkUpsert(
                tablePath(),
                listValue,
                new BulkUpsertSettings().setTimeout(BULK_UPSERT_TIMEOUT)
            )).thenAccept(status -> status.expect(onFailureMessage("bulk upsert", operationId, numId)));
        });
    }

    @Override
    public CompletableFuture<Void> delete(String operationId, int numId, Collection<Labels> keys) {
        return safeCall(() -> {
            var params = Params.of("$keys", YdbDeletedMetricsTable.keysToList(operationId, numId, keys));
            return execute(query.delete, params)
                .thenAccept(result -> result.expect(onFailureMessage("delete", operationId, numId)));
        });
    }

    @Override
    public CompletableFuture<Long> deleteBatch(String operationId, int numId) {
        return safeCall(() -> {
            var params = Params.of(
                "$operationId", utf8(operationId),
                "$shardId", uint32(numId));

            return execute(query.deleteBatch, params)
                .thenApply(result -> {
                    var rs = result.expect(onFailureMessage("delete batch", operationId, numId)).getResultSet(0);
                    if (!rs.next()) {
                        return 0L;
                    }

                    return rs.getColumn(0).getUint64();
                });
        });
    }

    private String tablePath() {
        return query.tablePath;
    }

    private ReadTableSettings prepareReadTableSettings(
        String operationId,
        int numId,
        int limit,
        @Nullable Labels lastKey)
    {
        var operationIdValue = utf8(operationId).makeOptional();
        var shardIdValue = uint32(numId).makeOptional();

        OptionalValue hashFromValue;
        OptionalValue labelsFromValue;
        if (lastKey == null) {
            hashFromValue = PrimitiveType.uint32().makeOptional().emptyValue();
            labelsFromValue = PrimitiveType.utf8().makeOptional().emptyValue();
        } else {
            var labels = LabelListSortedSerialize.format(lastKey);
            hashFromValue = uint32(labels.hashCode()).makeOptional();
            labelsFromValue = utf8(labels).makeOptional();
        }

        var hashToValue = uint32(UnsignedInteger.MAX_VALUE.intValue()).makeOptional();
        var labelsToValue = utf8(LabelListSortedSerialize.SENTINEL).makeOptional();

        return ReadTableSettings.newBuilder()
            .fromKeyExclusive(TupleValue.of(
                operationIdValue,
                shardIdValue,
                hashFromValue,
                labelsFromValue))
            .toKeyInclusive(TupleValue.of(
                operationIdValue,
                shardIdValue,
                hashToValue,
                labelsToValue))
            .timeout(READ_TABLE_TIMEOUT)
            .rowLimit(limit)
            .orderedRead(true)
            .build();
    }

    private String onFailureMessage(String action, String operationId, int numId) {
        return String.format(
            "can't %s: table=%s operationId=%s shard=%s",
            action,
            tablePath(),
            operationId,
            Integer.toUnsignedLong(numId));
    }

    private CompletableFuture<Result<DataQueryResult>> execute(String query, Params params) {
        return safeCall(query, params, (q, p) -> retryCtx.supplyResult(session -> {
            var settings = new ExecuteDataQuerySettings().keepInQueryCache();
            var tx = TxControl.serializableRw();
            return session.executeDataQuery(q, tx, p, settings);
        }));
    }
}
