package ru.yandex.solomon.coremon.meta.db.ydb;

import java.util.Collection;
import java.util.Map;
import java.util.function.Consumer;

import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.ListType;
import com.yandex.ydb.table.values.ListValue;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.StructType;
import com.yandex.ydb.table.values.Value;

import ru.yandex.monlib.metrics.labels.LabelAllocator;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.coremon.meta.CoremonMetricArray;
import ru.yandex.solomon.ydb.YdbResultSets;

import static java.util.stream.Collectors.toList;
import static ru.yandex.solomon.ydb.YdbResultSets.uint32;
import static ru.yandex.solomon.ydb.YdbResultSets.uint64;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
final class YdbDeletedMetricsTable {
    static final String TABLE_NAME = "DeletedMetrics";

    private static final StructType DELETED_METRIC_TYPE = StructType.of(Map.of(
        "operationId", PrimitiveType.utf8(),
        "shardId", PrimitiveType.uint32(),
        "hash", PrimitiveType.uint32(),
        "labels", PrimitiveType.utf8(),
        "spShardId", PrimitiveType.uint32(),
        "spLocalId", PrimitiveType.uint64(),
        "createdAt", PrimitiveType.uint64(),
        "flags", PrimitiveType.uint32()
    ));

    private static final StructType KEY_TYPE = StructType.of(Map.of(
        "operationId", PrimitiveType.utf8(),
        "shardId", PrimitiveType.uint32(),
        "hash", PrimitiveType.uint32(),
        "labels", PrimitiveType.utf8()
    ));

    static final ListType KEYS_LIST_TYPE = ListType.of(KEY_TYPE);

    static ListValue metricsToListValue(
        String operationId,
        int shardId,
        CoremonMetricArray metrics)
    {
        assert !metrics.isEmpty();
        var values = new Value<?>[metrics.size()];

        for (int i = 0; i < metrics.size(); i++) {
            var labels = LabelListSortedSerialize.format(metrics.getLabels(i));
            var hash = labels.hashCode();

            var spShardId = metrics.getShardId(i);
            var spLocalId = metrics.getLocalId(i);
            var createdAt = metrics.getCreatedAtSeconds(i);
            var flags = Flags.buildFlags(metrics.getType(i));

            // NOTE: unsafe construction, do not modify order of parameters!
            values[i] = DELETED_METRIC_TYPE.newValueUnsafe(
                PrimitiveValue.uint64(createdAt),
                PrimitiveValue.uint32(flags),
                PrimitiveValue.uint32(hash),
                PrimitiveValue.utf8(labels),
                PrimitiveValue.utf8(operationId),
                PrimitiveValue.uint32(shardId),
                PrimitiveValue.uint64(spLocalId),
                PrimitiveValue.uint32(spShardId)
            );
        }

        return ListValue.of(values);
    }

    static Value<?> keysToList(
        String operationId,
        int shardId,
        Collection<Labels> keys)
    {
        return KEYS_LIST_TYPE.newValue(keys.stream()
                .map(key -> {
                    var labels = LabelListSortedSerialize.format(key);
                    var hash = labels.hashCode();

                    // NOTE: unsafe construction, do not modify order of parameters!
                    return KEY_TYPE.newValueUnsafe(
                        PrimitiveValue.uint32(hash),
                        PrimitiveValue.utf8(labels),
                        PrimitiveValue.utf8(operationId),
                        PrimitiveValue.uint32(shardId));
                })
                .collect(toList()));
    }

    static TableDescription description() {
        return TableDescription.newBuilder()
            .addNullableColumn("operationId", PrimitiveType.utf8())
            .addNullableColumn("shardId", PrimitiveType.uint32())
            .addNullableColumn("hash", PrimitiveType.uint32())
            .addNullableColumn("labels", PrimitiveType.utf8())
            .addNullableColumn("spShardId", PrimitiveType.uint32())
            .addNullableColumn("spLocalId", PrimitiveType.uint64())
            .addNullableColumn("createdAt", PrimitiveType.uint64())
            .addNullableColumn("flags", PrimitiveType.uint32())
            .setPrimaryKeys("operationId", "shardId", "hash", "labels")
            .build();
    }

    private YdbDeletedMetricsTable() {
    }

    static final class DeletedMetricsReader implements Consumer<ResultSetReader> {
        private final LabelAllocator labelAllocator;
        private final CoremonMetricArray metrics;

        DeletedMetricsReader(LabelAllocator labelAllocator, CoremonMetricArray metrics) {
            this.labelAllocator = labelAllocator;
            this.metrics = metrics;
        }

        void reset() {
            if (!metrics.isEmpty()) {
                metrics.clear();
            }
        }

        @Override
        public void accept(ResultSetReader resultSet) {
            var rowCount = resultSet.getRowCount();
            if (rowCount == 0) {
                return;
            }

            new ColumnReader(resultSet, labelAllocator).read(metrics);
        }
    }

    static final class ColumnReader {

        private final int labelsIdx;
        private final int shardIdIdx;
        private final int localIdIdx;
        private final int createdAtIdx;
        private final int flagsIdx;

        private final ResultSetReader resultSet;
        private final LabelAllocator labelAllocator;

        ColumnReader(ResultSetReader resultSet, LabelAllocator labelAllocator) {
            this.labelsIdx = resultSet.getColumnIndex("labels");
            this.shardIdIdx = resultSet.getColumnIndex("spShardId");
            this.localIdIdx = resultSet.getColumnIndex("spLocalId");
            this.createdAtIdx = resultSet.getColumnIndex("createdAt");
            this.flagsIdx = resultSet.getColumnIndex("flags");

            this.resultSet = resultSet;
            this.labelAllocator = labelAllocator;
        }

        void read(CoremonMetricArray metrics) {
            var labelsBuilder = Labels.builder(Labels.MAX_LABELS_COUNT, labelAllocator);

            while (resultSet.next()) {
                labelsBuilder.clear();
                metrics.add(
                    uint32(resultSet, shardIdIdx),
                    uint64(resultSet, localIdIdx),
                    LabelListSortedSerialize.parse(YdbResultSets.utf8(resultSet, labelsIdx), labelsBuilder),
                    (int) uint64(resultSet, createdAtIdx),
                    Flags.readMetricType(uint32(resultSet, flagsIdx)));
            }
        }
    }
}
