package ru.yandex.solomon.coremon.meta.file;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.function.Consumer;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.ThreadSafe;

import io.netty.util.IllegalReferenceCountException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.actor.ActorRunner;
import ru.yandex.misc.thread.WhatThreadDoes;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.CoremonMetricArray;
import ru.yandex.solomon.coremon.meta.MetricsCollection;
import ru.yandex.solomon.coremon.meta.SearchIteratorResult;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.util.collection.CloseableIterator;
import ru.yandex.solomon.util.collection.queue.ArrayListLockQueueMem;
import ru.yandex.solomon.util.labelStats.LabelValuesStats;


/**
 * Collection of metrics which allowed to lookup metrics by labels or search them by selectors using FTS.
 * This collection designed to be wait-free for reads and lock-free for updates.
 *
 * @author Sergey Polovko
 */
@ParametersAreNonnullByDefault
@ThreadSafe
public class FileMetricsCollection implements MetricsCollection<CoremonMetric>, AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(FileMetricsCollection.class);

    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(FileMetricsCollection.class);

    private final String uniqueId;
    private final ActorRunner actor;
    private final Executor executor;
    private final ArrayListLockQueueMem<UpdateRequest> updates = new ArrayListLockQueueMem<>(2);
    private final ArrayListLockQueueMem<RemoveRequest> removals = new ArrayListLockQueueMem<>(1);

    private volatile LevelsArray levels;
    private volatile boolean stop = false;

    /**
     * used for debug through manager WebUI
     */
    @Nullable
    private Labels lastCreatedMetric;


    public FileMetricsCollection(String uniqueId, Executor executor) {
        this(uniqueId, executor, new CoremonMetricArray(0));
    }

    // takes ownership of metrics
    public FileMetricsCollection(String uniqueId, Executor executor, CoremonMetricArray metrics) {
        this.uniqueId = uniqueId;
        this.actor = new ActorRunner(this::act, executor);
        this.executor = executor;
        this.levels = new LevelsArray(metrics);
    }

    // takes ownership of metrics
    public FileMetricsCollection(String uniqueId, Executor executor, int[] levelsSizes, CoremonMetricArray metrics) {
        this.uniqueId = uniqueId;
        this.actor = new ActorRunner(this::act, executor);
        this.executor = executor;
        this.levels = new LevelsArray(levelsSizes, metrics);
    }

    protected void act() {
        if (stop) {
            ArrayList<UpdateRequest> updates = this.updates.dequeueAll();
            for (UpdateRequest update : updates) {
                update.release();
            }

            ArrayList<RemoveRequest> removals = this.removals.dequeueAll();

            LevelsArray levels = this.levels;
            if (levels != LevelsArray.EMPTY) {
                this.levels = LevelsArray.EMPTY;
                levels.release();
            }

            if (!updates.isEmpty()) {
                completeInDifferentThread(updates);
            }
            if (!removals.isEmpty()) {
                completeInDifferentThread(removals);
            }
        } else {
            WhatThreadDoes.withNr(uniqueId + " process updates", this::processUpdates);
            WhatThreadDoes.withNr(uniqueId + " process removals", this::processRemovals);
            WhatThreadDoes.withNr(uniqueId + " process merges", this::processMerges);
        }
    }

    private void processUpdates() {
        final List<UpdateRequest> updates = this.updates.dequeueAll();
        if (updates.isEmpty()) {
            return;
        }

        WhatThreadDoes.Handle h = WhatThreadDoes.push("processUpdates in " + uniqueId);
        try {
            final LevelsArray oldLevels = levels;
            final CoremonMetricArray newMetrics = mergeUpdates(uniqueId, oldLevels, updates);
            if (newMetrics.isEmpty()) {
                newMetrics.close();
            } else {
                lastCreatedMetric = newMetrics.getLabels(0);

                LevelsArray newLevels = oldLevels.updateZeroLevel(newMetrics);
                if (newLevels != oldLevels) {
                    this.levels = newLevels;
                    oldLevels.release();
                } else {
                    newMetrics.close();
                }
            }

            // avoid blocking current thread
            completeInDifferentThread(updates);
        } catch (Throwable t) {
            logger.error("error while processUpdates in " + uniqueId, t);
            failInDifferentThread(updates, t);
        } finally {
            h.pop();
        }
    }

    private static CoremonMetricArray mergeUpdates(String shardId, LevelsArray levels, List<UpdateRequest> updates) {
        int expectedSize = 0;
        for (UpdateRequest update : updates) {
            expectedSize += update.size();
        }

        CoremonMetricArray newMetrics = new CoremonMetricArray(expectedSize);
        HashSet<Labels> newLabels = new HashSet<>(expectedSize);

        try {
            for (UpdateRequest update : updates) {
                try {
                    if (update.isMany()) {
                        CoremonMetricArray metrics = update.getMetrics();
                        for (int i = 0; i < metrics.size(); i++) {
                            Labels labels = metrics.getLabels(i);
                            try (var metric = levels.getOrNull(labels)) {
                                if (metric != null) {
                                    metric.setType(metrics.getType(i));
                                    continue;
                                }
                            }

                            if (newLabels.add(labels)) {
                                newMetrics.add(
                                    metrics.getShardId(i),
                                    metrics.getLocalId(i),
                                    labels,
                                    metrics.getCreatedAtSeconds(i),
                                    metrics.getType(i));
                            }
                        }
                    } else {
                        CoremonMetric m = update.getMetric();
                        try (var metric = levels.getOrNull(m.getLabels())) {
                            if (metric != null) {
                                metric.setType(m.getType());
                                continue;
                            }
                        }
                        if (newLabels.add(m.getLabels())) {
                            newMetrics.add(
                                m.getShardId(),
                                m.getLocalId(),
                                m.getLabels(),
                                m.getCreatedAtSeconds(),
                                m.getType());
                        }
                    }
                } finally {
                    update.release();
                }
            }
        } catch (Throwable t) {
            newMetrics.close();
            throw new RuntimeException("cannot merge updates in " + shardId, t);
        }

        return newMetrics;
    }

    private void processRemovals() {
        final List<RemoveRequest> removals = this.removals.dequeueAll();
        if (removals.isEmpty()) {
            return;
        }

        WhatThreadDoes.Handle h = WhatThreadDoes.push("processRemovals in " + uniqueId);
        try {
            LevelsArray oldLevels = this.levels;
            LevelsArray newLevels = oldLevels.remove(removals);
            if (newLevels != oldLevels) {
                this.levels = newLevels;
                oldLevels.release();
            }

            // avoid blocking current thread
            completeInDifferentThread(removals);
        } catch (Throwable t) {
            logger.error("error while processRemovals in " + uniqueId, t);
            failInDifferentThread(removals, t);
        } finally {
            h.pop();
        }
    }

    private void processMerges() {
        WhatThreadDoes.Handle h = WhatThreadDoes.push("processMerges in " + uniqueId);
        try {
            LevelsArray oldLevels = this.levels;
            LevelsArray newLevels = oldLevels.mergeOverfullLevel();
            if (newLevels != oldLevels) {
                this.levels = newLevels;
                oldLevels.release();
            }
        } catch (Throwable t) {
            logger.error("error while processMerges in " + uniqueId, t);
        } finally {
            h.pop();
        }
    }

    private <C extends CompletableFuture<Void>> void completeInDifferentThread(List<C> cfs) {
        executor.execute(() -> {
            for (C cf : cfs) {
                cf.complete(null);
            }
        });
    }

    private <C extends CompletableFuture<Void>> void failInDifferentThread(List<C> cfs, Throwable t) {
        executor.execute(() -> {
            for (C cf : cfs) {
                cf.completeExceptionally(t);
            }
        });
    }

    private LevelsArray retainLevels() {
        for (int i = 0; i < 10; i++) {
            try {
                LevelsArray levels = this.levels;
                return (LevelsArray) levels.retain();
            } catch (IllegalReferenceCountException e) {
                // we got reference to already released levels
            }

            // try again
        }

        throw new IllegalStateException("cannot retain levels after 10 retries");
    }

    @Override
    public CoremonMetric getOrNull(Labels key) {
        try (LevelsArray levels = retainLevels()) {
            return levels.getOrNull(key);
        }
    }

    @Override
    public boolean has(Labels key) {
        try (LevelsArray levels = retainLevels()) {
            return levels.has(key);
        }
    }

    @Override
    public CompletableFuture<Void> put(CoremonMetric metric) {
        UpdateRequest update = new UpdateRequest(metric);
        updates.enqueue(update);
        actor.schedule();
        return update;
    }

    @Override
    public CompletableFuture<Void> putAll(CoremonMetricArray metrics) {
        UpdateRequest update = new UpdateRequest(metrics);
        updates.enqueue(update);
        actor.schedule();
        return update;
    }

    @Override
    public CompletableFuture<Void> removeAll(Collection<Labels> keys) {
        RemoveRequest remove = new RemoveRequest(keys);
        removals.enqueue(remove);
        actor.schedule();
        return remove;
    }

    @Override
    public int searchMetrics(Selectors selectors, int offset, int limit, Consumer<CoremonMetric> fn) {
        try (LevelsArray levels = retainLevels()) {
            return levels.doSearch(offset, limit, fn, level -> level.searchMetrics(selectors));
        }
    }

    @Override
    public int searchLabels(Selectors selectors, int offset, int limit, Consumer<Labels> fn) {
        try (LevelsArray levels = retainLevels()) {
            return levels.doSearch(offset, limit, fn, level -> level.searchLabels(selectors));
        }
    }

    @Override
    public int searchCount(Selectors selectors) {
        try (LevelsArray levels = retainLevels()) {
            return levels.searchCount(selectors);
        }
    }

    @Override
    public SearchIteratorResult searchIterator(Selectors selectors) {
        var levels = retainLevels();
        try {
            var streamAndCount = levels.searchStreamAndCount(level -> level.searchMetrics(selectors));
            if (streamAndCount == null || streamAndCount.getCount() == 0) {
                levels.release();
                return SearchIteratorResult.empty();
            }

            var iterator = CloseableIterator.of(
                streamAndCount.getStream().iterator(),
                levels::release);

            return new SearchIteratorResult(iterator, streamAndCount.getCount());
        } catch (Throwable t) {
            levels.release();
            throw t;
        }
    }

    @Override
    public Set<String> labelNames() {
        try (LevelsArray levels = retainLevels()) {
            return levels.labelNames();
        }
    }

    @Override
    public LabelValuesStats labelStats(Set<String> requestedNames) {
        try (LevelsArray levels = retainLevels()) {
            return levels.labelStats(requestedNames);
        }
    }

    @Override
    public long searchIndexSize() {
        try (LevelsArray levels = retainLevels()) {
            return levels.searchIndexSize();
        }
    }

    @Override
    public long searchIndexCacheSize() {
        try (LevelsArray levels = retainLevels()) {
            return levels.searchIndexCacheSize();
        }
    }

    @Override
    public int size() {
        try (LevelsArray levels = retainLevels()) {
            return levels.size();
        }
    }

    @Override
    public long memorySizeIncludingSelf() {
        long size = SELF_SIZE;

        size += updates.memorySizeIncludingSelf();
        size += removals.memorySizeIncludingSelf();

        try (LevelsArray levels = retainLevels()) {
            size += levels.memorySizeIncludingSelf();
        }

        return size;
    }

    @Override
    public String toString() {
        try (LevelsArray levels = retainLevels()) {
            StringBuilder sb = new StringBuilder();
            sb.append("FileMetricsCollection{\n");
            for (int i = 0; i < levels.count(); i++) {
                sb.append("  ").append("level").append(i);
                sb.append(" { size: ").append(levels.levelSize(i)).append(", max: ");
                int maxSize = levels.levelMaxSize(i);
                if (maxSize == Integer.MAX_VALUE) {
                    sb.append("inf");
                } else {
                    sb.append(maxSize);
                }
                sb.append(" }\n");
            }
            sb.append("}");
            return sb.toString();
        }
    }

    @Override
    public CloseableIterator<CoremonMetric> iterator() {
        return new IteratorImpl(retainLevels());
    }

    @Override
    public void close() {
        stop = true;
        actor.schedule();
    }

    public int levelSize(int i) {
        return levels.levelSize(i);
    }
}
