package ru.yandex.solomon.coremon.meta.file;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;

import io.netty.util.AbstractReferenceCounted;
import it.unimi.dsi.fastutil.objects.Object2IntOpenHashMap;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.CoremonMetricArray;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.search.SearchEngine;
import ru.yandex.solomon.search.SearchIndex;
import ru.yandex.solomon.search.result.SearchResult;
import ru.yandex.solomon.util.collection.StreamAndCount;


/**
 * @author Sergey Polovko
 */
@Immutable
final class Level extends AbstractReferenceCounted implements MemMeasurable {

    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(Level.class);
    private static final SearchEngine SEARCH_ENGINE = SearchEngine.defaultSearchEngine;
    private static final SearchIndex EMPTY_SEARCH_INDEX = SEARCH_ENGINE.build(Collections.emptyList());

    private static final Object2IntOpenHashMap<Labels> EMPTY_LABELS_INDEX = newLabelIndex(0);

    private final CoremonMetricArray metrics;
    private final Object2IntOpenHashMap<Labels> labelsIndex;
    private final SearchIndex searchIndex;

    Level() {
        this(new CoremonMetricArray(0), EMPTY_LABELS_INDEX, EMPTY_SEARCH_INDEX);
    }

    Level(CoremonMetricArray metrics) {
        this.metrics = metrics;
        this.labelsIndex = newLabelIndex(metrics.size());
        for (int i = 0; i < metrics.size(); i++) {
            this.labelsIndex.put(metrics.getLabels(i), i);
        }
        this.searchIndex = SEARCH_ENGINE.build(metrics.labelsIterator());
    }

    Level(CoremonMetricArray metrics, Object2IntOpenHashMap<Labels> labelsIndex, SearchIndex searchIndex) {
        this.metrics = metrics;
        this.labelsIndex = labelsIndex;
        this.searchIndex = searchIndex;
    }

    /**
     * returned metric must be released (closed)
     */
    @Nullable
    CoremonMetric getMetricByLabels(Labels key) {
        int index = labelsIndex.getInt(key);
        return getMetricByIndex(index);
    }

    /**
     * returned metric must be released (closed)
     */
    @Nullable
    CoremonMetric getMetricByIndex(int index) {
        if (index < 0 || index >= metrics.size()) {
            return null;
        }
        return metrics.getWithRetain(index);
    }

    boolean has(Labels key) {
        return labelsIndex.containsKey(key);
    }

    int size() {
        return metrics.size();
    }

    @Override
    public Level touch(Object hint) {
        return this;
    }

    @Override
    protected void deallocate() {
        labelsIndex.clear();
        metrics.close();
    }

    SearchIndex getSearchIndex() {
        return searchIndex;
    }

    // takes ownership of newMetrics
    Level update(CoremonMetricArray newMetrics) {
        if (newMetrics.isEmpty()) {
            return this;
        }

        // (1) add metrics of current level
        newMetrics.addAll(metrics);
        newMetrics.shrinkToFit();

        // (2) create new labels index
        Object2IntOpenHashMap<Labels> labelsIndex = newLabelIndex(newMetrics.size());
        for (int i = 0; i < newMetrics.size(); i++) {
            labelsIndex.put(newMetrics.getLabels(i), i);
        }

        // (4) rebuild search index
        SearchIndex searchIndex = SEARCH_ENGINE.build(newMetrics.labelsIterator());

        // (5) create new level
        return new Level(newMetrics, labelsIndex, searchIndex);
    }

    Level mergeWith(Level level) {
        if (size() == 0) {
            return level;
        } else if (level.size() == 0) {
            return this;
        }

        final int newSize = size() + level.size();

        // (1) merge metrics
        CoremonMetricArray metrics = new CoremonMetricArray(newSize);
        metrics.addAll(this.metrics);
        metrics.addAll(level.metrics);

        try {
            // (2) merge labels indexes
            Object2IntOpenHashMap<Labels> labelsIndex = newLabelIndex(newSize);
            labelsIndex.putAll(this.labelsIndex);
            for (int index = size(); index < newSize; index++) {
                labelsIndex.put(metrics.getLabels(index), index);
            }

            // (3) rebuild search index
            SearchIndex searchIndex = SEARCH_ENGINE.build(metrics.labelsIterator());

            // (5) create new level
            return new Level(metrics, labelsIndex, searchIndex);
        } catch (Throwable t) {
            metrics.close();
            throw new RuntimeException("cannot merge levels", t);
        }
    }

    Level remove(List<RemoveRequest> removals) {
        final int[] toRemove = removals.stream()
            .flatMap(r -> Arrays.stream(r.getKeys()))
            .mapToInt(labelsIndex::getInt)
            .filter(index -> index != -1)
            .sorted()
            .distinct()
            .toArray();

        if (toRemove.length == 0) {
            // no metrics found
            return this;
        }

        if (size() == toRemove.length) {
            // all elements have to be removed
            return new Level();
        }

        final int newSize = size() - toRemove.length;

        CoremonMetricArray metrics = new CoremonMetricArray(newSize);
        Object2IntOpenHashMap<Labels> labelsIndex = newLabelIndex(newSize);

        try {
            // (1) sieve to be removed metrics
            int srcIndex = 0;
            int dstIndex = 0;
            for (int toRemoveIndex : toRemove) {
                if (srcIndex < toRemoveIndex) {
                    metrics.addAll(this.metrics, srcIndex, toRemoveIndex);
                }
                while (srcIndex < toRemoveIndex) {
                    Labels labels = this.metrics.getLabels(srcIndex++);
                    labelsIndex.put(labels, dstIndex++);
                }
                srcIndex++; // skip toRemoveIndex
            }

            // (2) copy the rest
            if (srcIndex < this.metrics.size()) {
                metrics.addAll(this.metrics, srcIndex, this.metrics.size());
            }
            while (srcIndex < this.metrics.size()) {
                Labels labels = this.metrics.getLabels(srcIndex++);
                labelsIndex.put(labels, dstIndex++);
            }

            // (3) rebuild search index
            SearchIndex searchIndex = SEARCH_ENGINE.build(metrics.labelsIterator());

            // (4) create new level
            return new Level(metrics, labelsIndex, searchIndex);
        } catch (Throwable t) {
            metrics.close();
            throw new RuntimeException("cannot remove metrics", t);
        }
    }

    StreamAndCount<CoremonMetric> searchMetrics(Selectors selectors) {
        if (selectors.isEmpty()) {
            return new StreamAndCount<>(metrics.stream(), metrics.size());
        }

        final SearchResult searchResult = searchIndex.search(selectors);
        return new StreamAndCount<>(
            searchResult.mapToObj(metrics::get),
            searchResult.size());
    }

    StreamAndCount<Labels> searchLabels(Selectors selectors) {
        if (selectors.isEmpty()) {
            return new StreamAndCount<>(metrics.streamLabels(), metrics.size());
        }

        final SearchResult searchResult = searchIndex.search(selectors);
        return new StreamAndCount<>(
            searchResult.mapToObj(metrics::getLabels),
            searchResult.size());
    }

    int searchCount(Selectors selectors) {
        return selectors.isEmpty()
            ? metrics.size()
            : searchIndex.search(selectors).size();
    }

    private static Object2IntOpenHashMap<Labels> newLabelIndex(int newSize) {
        Object2IntOpenHashMap<Labels> labelsIndex = new Object2IntOpenHashMap<>(newSize);
        labelsIndex.defaultReturnValue(-1);
        return labelsIndex;
    }

    @Override
    public long memorySizeIncludingSelf() {
        long size = SELF_SIZE;
        size += searchIndex.memorySizeIncludingSelf();
        size += metrics.memorySizeIncludingSelf();
        size += MemoryCounter.object2IntOpenHashMapSize(labelsIndex);
        return size;
    }

    @Override
    public String toString() {
        return "Level{size: " + size() + ", refCnt: " + refCnt() + '}';
    }

    // for tests only
    CoremonMetricArray getMetrics() {
        return metrics;
    }
}
