package ru.yandex.solomon.coremon.meta.file;

import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;

import io.netty.util.AbstractReferenceCounted;
import io.netty.util.ReferenceCounted;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.CoremonMetricArray;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.util.collection.Slicer;
import ru.yandex.solomon.util.collection.StreamAndCount;
import ru.yandex.solomon.util.labelStats.LabelValuesStats;


/**
 * @author Sergey Polovko
 */
@Immutable
final class LevelsArray extends AbstractReferenceCounted implements MemMeasurable, AutoCloseable {

    static final LevelsArray EMPTY = new LevelsArray(new int[0], new Level[0]);

    private static final int[] DEFAULT_LEVELS_SIZES = {
        20000,             // level0    20k
        2000000,           // level1    2m
        20000000,          // level2    20m
        Integer.MAX_VALUE, // level3    inf
    };

    private final int[] levelsSizes;
    private final Level[] levels;

    private LevelsArray(int[] levelsSizes, Level[] levels) {
        this.levelsSizes = levelsSizes;
        this.levels = levels;
    }

    LevelsArray(CoremonMetricArray metrics) {
        this(DEFAULT_LEVELS_SIZES, metrics);
    }

    LevelsArray(int[] levelsSizes, CoremonMetricArray metrics) {
        final int lastLevel = levelsSizes.length - 1;

        // last level must never be overfull
        levelsSizes[lastLevel] = Integer.MAX_VALUE;
        this.levelsSizes = levelsSizes;

        // init levels (last level can be non empty from the begin)
        final Level[] levels = new Level[levelsSizes.length];
        try {
            for (int i = 0; i < lastLevel; i++) {
                levels[i] = new Level();
            }
            levels[lastLevel] = new Level(metrics);
            this.levels = levels;
            metrics = null;
        } catch (Throwable e) {
            for (int i = 0; i < levels.length; i++) {
                Level level = levels[i];
                if (level != null) {
                    level.release();
                    levels[i] = null;
                }
            }

            if (metrics != null) {
                metrics.close();
            }

            throw new RuntimeException(e);
        }
    }

    @Override
    protected void deallocate() {
        for (int i = 0; i < levels.length; i++) {
            Level level = levels[i];
            levels[i] = null;
            level.release();
        }
    }

    @Override
    public ReferenceCounted touch(Object hint) {
        return this;
    }

    // takes ownership of newMetrics
    LevelsArray updateZeroLevel(CoremonMetricArray newMetrics) {
        if (levels.length == 0 || newMetrics.isEmpty()) {
            return this;
        }

        Level[] newLevels = new Level[levels.length];
        newLevels[0] = levels[0].update(newMetrics);
        return retainOtherLevels(newLevels);
    }

    boolean has(Labels labels) {
        for (int l = levels.length - 1; l >= 0; l--) {
            if (levels[l].has(labels)) {
                return true;
            }
        }
        return false;
    }

    LevelsArray remove(List<RemoveRequest> removals) {
        Level[] newLevels = null;
        for (int i = 0; i < levels.length; i++) {
            final Level oldLevel = levels[i];
            final Level newLevel = oldLevel.remove(removals);

            if (oldLevel != newLevel) {
                if (newLevels == null) {
                    newLevels = new Level[levels.length];
                }
                newLevels[i] = newLevel;
            }
        }

        if (newLevels == null) {
            return this;
        }

        return retainOtherLevels(newLevels);
    }

    LevelsArray mergeOverfullLevel() {
        if (levels.length < 2) {
            return this;
        }

        // start from the bottom level to avoid stuck on merging only smaller levels
        int overfullLevelIdx = -1;
        for (int i = levels.length - 1; i >= 0; i--) {
            if (levels[i].size() > levelsSizes[i]) {
                overfullLevelIdx = i;
                break;
            }
        }

        if (overfullLevelIdx == -1) {
            return this;
        }

        // merge only one level at a time to release current thread for other work

        Level overfullLevel = levels[overfullLevelIdx];
        Level nextLevel = levels[overfullLevelIdx + 1];

        Level[] newLevels = new Level[levels.length];
        newLevels[overfullLevelIdx] = new Level();

        if (nextLevel.size() == 0) {
            // just replace the next level with overfull one
            newLevels[overfullLevelIdx + 1] = (Level) overfullLevel.retain();
        } else {
            newLevels[overfullLevelIdx + 1] = nextLevel.mergeWith(overfullLevel);
        }

        return retainOtherLevels(newLevels);
    }

    CoremonMetric getOrNull(Labels key) {
        // start from the bigger one, because it is more probable to find metric there
        for (int i = levels.length - 1; i >= 0; i--) {
            final CoremonMetric metric = levels[i].getMetricByLabels(key);
            if (metric != null) {
                return metric;
            }
        }
        return null;
    }

    private LevelsArray retainOtherLevels(Level[] newLevels) {
        assert newLevels.length == levels.length;

        // fill empty array items by retaining current levels
        for (int i = 0; i < levels.length; i++) {
            if (newLevels[i] == null) {
                newLevels[i] = (Level) levels[i].retain();
            }
        }
        return new LevelsArray(levelsSizes, newLevels);
    }

    int size() {
        int size = 0;
        for (Level level : levels) {
            size += level.size();
        }
        return size;
    }

    /**
     * @return levels count
     */
    int count() {
        return levels.length;
    }

    /**
     * @return max size of level
     */
    int levelMaxSize(int i) {
        return levelsSizes[i];
    }

    /**
     * @return size (metric count) of level
     */
    int levelSize(int i) {
        return levels[i].size();
    }

    @Override
    public long memorySizeIncludingSelf() {
        long size = 0;
        for (Level level : levels) {
            size += level.memorySizeIncludingSelf();
        }
        return size;
    }

    int searchCount(Selectors selectors) {
        int count = 0;
        for (Level level : levels) {
            count += level.searchCount(selectors);
        }
        return count;
    }

    Set<String> labelNames() {
        HashSet<String> result = new HashSet<>(16);
        for (Level level : levels) {
            result.addAll(level.getSearchIndex().labelNames());
        }
        return result;
    }

    LabelValuesStats labelStats(Set<String> requestedNames) {
        LabelValuesStats result = new LabelValuesStats();
        for (Level level : levels) {
            result.combine(level.getSearchIndex().labelStats(requestedNames));
        }
        return result;
    }

    long searchIndexSize() {
        long size = 0;
        for (Level level : levels) {
            size += level.getSearchIndex().getIndexSize();
        }
        return size;
    }

    long searchIndexCacheSize() {
        long size = 0;
        for (Level level : levels) {
            size += level.getSearchIndex().getIndexCacheSize();
        }
        return size;
    }

    <T> int doSearch(
        int offset, int limit,
        Consumer<T> resultConsumer,
        Function<Level, StreamAndCount<T>> searchFunc)
    {
        StreamAndCount<T> streamAndCount = searchStreamAndCount(searchFunc);
        if (streamAndCount == null) {
            return 0;
        }

        if (limit <= 0) {
            streamAndCount.getStream().forEach(resultConsumer);
            return streamAndCount.getCount();
        }

        Stream<T> stream = Slicer.slice(streamAndCount.getStream(), offset, limit);
        stream.forEach(resultConsumer);
        return streamAndCount.getCount();
    }

    @Nullable
    <T> StreamAndCount<T> searchStreamAndCount(Function<Level, StreamAndCount<T>> searchFunc) {
        StreamAndCount<T> streamAndCount = null;
        for (int i = levels.length - 1; i >= 0; i--) {
            var level = levels[i];
            if (streamAndCount == null) {
                streamAndCount = searchFunc.apply(level);
            } else {
                streamAndCount = streamAndCount.concat(searchFunc.apply(level));
            }
        }
        return streamAndCount;
    }

    Level getLevel(int levelIdx) {
        return levels[levelIdx];
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("LevelsArray{ refCnt: ").append(refCnt()).append(", ");
        for (int i = 0; i < levels.length; i++) {
            Level level = levels[i];
            if (level == null) {
                continue;
            }

            sb.append("  ").append("level").append(i);
            sb.append(":{ size: ").append(level.size()).append(", max: ");

            int maxSize = levelsSizes[i];
            if (maxSize == Integer.MAX_VALUE) {
                sb.append("inf");
            } else {
                sb.append(maxSize);
            }
            sb.append(" }, ");
        }
        sb.setLength(sb.length() - 2);
        sb.append(" }");
        return sb.toString();
    }

    @Override
    public void close() {
        release();
    }
}
