package ru.yandex.solomon.coremon.meta.gc;

import java.util.ArrayList;
import java.util.List;
import java.util.OptionalLong;
import java.util.concurrent.CompletableFuture;

import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.ints.IntArrayList;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.coremon.meta.CoremonMetricArray;
import ru.yandex.solomon.coremon.meta.db.MetricsDao;
import ru.yandex.solomon.util.future.RetryCompletableFuture;
import ru.yandex.solomon.util.future.RetryConfig;

/**
 * @author Vladimir Gordiychuk
 */
public class GcMetricsTask {
    private static final int MAX_BATCH_SIZE = 10_000;

    private final int numId;
    private final MetricsDao dao;
    private final StockpileDeleter stockpile;
    private final RetryConfig retry;
    private long totalMetricsCount;
    private long deletedMetricsCount;
    private int metricIdx = 0;
    private CoremonMetricArray metrics;
    private final Int2ObjectMap<IntArrayList> batchByShardId = new Int2ObjectOpenHashMap<>();
    private final CompletableFuture<Long> doneFuture = new CompletableFuture<>();

    public GcMetricsTask(int numId, MetricsDao dao, StockpileDeleter stockpile, RetryConfig retryConfig) {
        this.numId = numId;
        this.dao = dao;
        this.stockpile = stockpile;
        this.retry = retryConfig;
    }

    public CompletableFuture<Long> run() {
        resolveMetricsCount()
            .thenCompose(ignore -> loadMetrics())
            .whenComplete((ignore, e) -> {
                if (e != null) {
                    doneFuture.completeExceptionally(e);
                }
                continueDelete();
            });
        return doneFuture;
    }

    private CompletableFuture<?> resolveMetricsCount() {
        var future = RetryCompletableFuture.runWithRetries(() -> {
            if (doneFuture.isDone()) {
                return CompletableFuture.completedFuture(0L);
            }

            return dao.getMetricCount();
        }, retry);

        return future.thenAccept(count -> {
            totalMetricsCount = count;
        });
    }

    private CompletableFuture<?> loadMetrics() {
        if (totalMetricsCount == 0) {
            return CompletableFuture.completedFuture(null);
        }

        var future = RetryCompletableFuture.runWithRetries(() -> {
            if (doneFuture.isDone()) {
                return CompletableFuture.completedFuture(null);
            }

            return load();
        }, retry);

        return future.thenAccept(metrics -> {
            this.metrics = metrics;
        });
    }

    private void continueDelete() {
        if (doneFuture.isDone()) {
            if (metrics != null) {
                metrics.close();
                metrics = null;
            }

            return;
        }

        if (metrics == null) {
            doneFuture.complete(0L);
            return;
        }

        try {
            for (; metricIdx < metrics.size(); metricIdx++) {
                int shardId = metrics.getShardId(metricIdx);
                var batch = batchByShardId.get(shardId);
                if (batch == null) {
                    batch = new IntArrayList();
                    batchByShardId.put(shardId, batch);
                }

                batch.add(metricIdx);
                if (batch.size() >= MAX_BATCH_SIZE) {
                    metricIdx++;
                    batchByShardId.remove(shardId);
                    deleteMetrics(shardId, batch);
                    return;
                }
            }

            var batchIt = batchByShardId.int2ObjectEntrySet().iterator();
            if (batchIt.hasNext()) {
                var entry = batchIt.next();
                int shardId = entry.getIntKey();
                var batch = entry.getValue();
                batchIt.remove();
                deleteMetrics(shardId, batch);
                return;
            }

            metrics.close();
            doneFuture.complete(deletedMetricsCount);
        } catch (Throwable e) {
            doneFuture.completeExceptionally(e);
        }
    }

    private void deleteMetrics(int shardId, IntArrayList batch) {
        deletedMetricsCount += batch.size();
        deleteFromStockpile(shardId, batch)
                .thenCompose(ignore -> deleteFromMetabase(batch))
                .whenComplete((ignore, e) -> {
                    if (e != null) {
                        doneFuture.completeExceptionally(e);
                    }

                    continueDelete();
                });
    }

    private CompletableFuture<Void> deleteFromStockpile(int shardId, IntArrayList batch) {
        try {
            var localIds = prepareStockpileDelete(batch);
            return RetryCompletableFuture.runWithRetries(() -> stockpile.delete(shardId, localIds), retry);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    private CompletableFuture<Void> deleteFromMetabase(IntArrayList batch) {
        try {
            var req = prepareMetabaseDelete(batch);
            return RetryCompletableFuture.runWithRetries(() -> dao.deleteMetrics(req), retry);
        } catch (Throwable e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    private List<Labels> prepareMetabaseDelete(IntArrayList batch) {
        var result = new ArrayList<Labels>(batch.size());
        for (int index = 0; index < batch.size(); index++) {
            result.add(metrics.getLabels(batch.getInt(index)));
        }
        return result;
    }

    private long[] prepareStockpileDelete(IntArrayList batch) {
        long[] localIds = new long[batch.size()];
        for (int index = 0; index < batch.size(); index++) {
            localIds[index] = metrics.getLocalId(batch.getInt(index));
        }
        return localIds;
    }

    public CompletableFuture<CoremonMetricArray> load() {
        CoremonMetricArray metrics = new CoremonMetricArray(Math.toIntExact(totalMetricsCount));
        try {
            return dao.findMetrics(metrics::addAll, OptionalLong.of(totalMetricsCount))
                    .handle((ignore, e) -> {
                        if (e != null) {
                            metrics.close();
                            throw new RuntimeException(e);
                        }
                        return metrics;
                    });
        } catch (Throwable e) {
            metrics.close();
            return CompletableFuture.failedFuture(e);
        }
    }

}
