package ru.yandex.solomon.coremon.meta.mem;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.memory.layout.MemoryCounter;


/**
 * @author Sergey Polovko
 */
@ParametersAreNonnullByDefault
public class MemOnlyMetricsCollectionImpl implements MemOnlyMetricsCollection {
    private static final int CHM_ENTRY_SIZE = MemoryCounter.OBJECT_HEADER_SIZE + (4 + 3 * MemoryCounter.OBJECT_POINTER_SIZE);

    // Reasons to use map of map:
    // 1. reduce amount of optLabels in heap(natural deduplication)
    // 2. reduce amount of Node for hash map(optLabels in general it's DC, and we have limited DC amount)
    // 3. composite key require more memory for key
    private final ConcurrentHashMap<Labels, ConcurrentHashMap<Labels, Object>> metricsByOptLabels;
    private final AtomicInteger size = new AtomicInteger();
    private final AtomicLong valueMemorySize = new AtomicLong();

    public MemOnlyMetricsCollectionImpl() {
        this.metricsByOptLabels = new ConcurrentHashMap<>();
    }

    @Override
    @Nullable
    public Object getOrNull(Labels optLabels, Labels key) {
        var metrics = metricsByOptLabels.get(optLabels);
        if (metrics == null) {
            return null;
        }
        return metrics.get(key);
    }

    @Override
    public void put(Labels optLabels, Labels labels, Object aggrMetrics) {
        var metrics = metricsByOptLabels.get(optLabels);
        if (metrics == null) {
            metrics = metricsByOptLabels.computeIfAbsent(optLabels, ignore -> new ConcurrentHashMap<>());
        }
        var prev = metrics.put(labels, aggrMetrics);
        if (prev == null) {
            size.incrementAndGet();
        }
        valueMemorySize.addAndGet(memorySize(aggrMetrics) - memorySize(prev));
    }

    @Override
    public int size() {
        return size.get();
    }

    public void clear() {
        size.set(0);
        valueMemorySize.set(0);
        metricsByOptLabels.clear();
    }

    public long memorySizeIncludingSelf() {
        // rough estimation
        long size = size();
        long valueSize = valueMemorySize.get();
        return size * ((long) CHM_ENTRY_SIZE) + valueSize;
    }

    private static long memorySize(@Nullable Object value) {
        if (value instanceof int[]) {
            return MemoryCounter.arrayObjectSize((int[]) value);
        }

        return 0;
    }
}
