package ru.yandex.solomon.coremon.meta.service;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Message;
import com.google.protobuf.TextFormat;
import io.grpc.StatusException;
import io.grpc.StatusRuntimeException;
import io.grpc.stub.StreamObserver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.metabase.client.impl.GrpcStatusMapping;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.thread.ThreadLocalTimeoutException;
import ru.yandex.monitoring.metabase.MetabaseServiceGrpc;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.core.ShardIsNotLocalException;
import ru.yandex.solomon.core.urlStatus.UrlStatusTypeAwareException;
import ru.yandex.solomon.labels.query.SelectorsException;
import ru.yandex.solomon.metabase.api.protobuf.CreateManyRequest;
import ru.yandex.solomon.metabase.api.protobuf.CreateManyResponse;
import ru.yandex.solomon.metabase.api.protobuf.CreateOneRequest;
import ru.yandex.solomon.metabase.api.protobuf.CreateOneResponse;
import ru.yandex.solomon.metabase.api.protobuf.DeleteManyRequest;
import ru.yandex.solomon.metabase.api.protobuf.DeleteManyResponse;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metabase.api.protobuf.FindRequest;
import ru.yandex.solomon.metabase.api.protobuf.FindResponse;
import ru.yandex.solomon.metabase.api.protobuf.MetricNamesRequest;
import ru.yandex.solomon.metabase.api.protobuf.MetricNamesResponse;
import ru.yandex.solomon.metabase.api.protobuf.ResolveManyRequest;
import ru.yandex.solomon.metabase.api.protobuf.ResolveManyResponse;
import ru.yandex.solomon.metabase.api.protobuf.ResolveOneRequest;
import ru.yandex.solomon.metabase.api.protobuf.ResolveOneResponse;
import ru.yandex.solomon.metabase.api.protobuf.TLabelNamesRequest;
import ru.yandex.solomon.metabase.api.protobuf.TLabelNamesResponse;
import ru.yandex.solomon.metabase.api.protobuf.TLabelValuesRequest;
import ru.yandex.solomon.metabase.api.protobuf.TLabelValuesResponse;
import ru.yandex.solomon.metabase.api.protobuf.TResolveLogsRequest;
import ru.yandex.solomon.metabase.api.protobuf.TResolveLogsResponse;
import ru.yandex.solomon.metabase.api.protobuf.TServerStatusRequest;
import ru.yandex.solomon.metabase.api.protobuf.TServerStatusResponse;
import ru.yandex.solomon.metabase.api.protobuf.TUniqueLabelsRequest;
import ru.yandex.solomon.metabase.api.protobuf.TUniqueLabelsResponse;
import ru.yandex.solomon.util.InvalidMetricNameException;

import static ru.yandex.grpc.utils.StreamObservers.asyncComplete;
import static ru.yandex.solomon.coremon.meta.service.GrpcMetabaseValidator.ensureValid;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class GrpcMetabaseService extends MetabaseServiceGrpc.MetabaseServiceImplBase {
    private static final Logger logger = LoggerFactory.getLogger(GrpcMetabaseService.class);

    private final MetabaseService metabaseService;
    private final MetricRegistry registry;

    public GrpcMetabaseService(MetabaseService metabaseService, MetricRegistry registry) {
        this.metabaseService = metabaseService;
        this.registry = registry;
    }

    @Override
    public void serverStatus(TServerStatusRequest request, StreamObserver<TServerStatusResponse> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> metabaseService.serverStatus(ensureValid(request)))
                .exceptionally(e -> {
                    var status = classifyError(e);
                    logError("serverStatus", request, status.code, e);

                    return TServerStatusResponse.newBuilder()
                            .setStatus(status.code)
                            .setStatusMessage(status.message)
                            .build();
                })
                .thenApply(response -> {
                    incrementStatus(MetabaseServiceGrpc.getServerStatusMethod().getFullMethodName(), response.getStatus());
                    return response;
                }), responseObserver);
    }

    @Override
    public void createOne(CreateOneRequest request, StreamObserver<CreateOneResponse> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> metabaseService.createOne(ensureValid(request)))
                .exceptionally(e -> {
                    var status = classifyError(e);
                    logError("createOne", request, status.code, e);

                    return CreateOneResponse.newBuilder()
                            .setStatus(status.code)
                            .setStatusMessage(status.message)
                            .build();
                })
                .thenApply(response -> {
                    boolean useNewFormat = !request.getMetric().getName().isEmpty();
                    incrementFormat(MetabaseServiceGrpc.getCreateOneMethod().getFullMethodName(), useNewFormat);
                    incrementStatus(MetabaseServiceGrpc.getCreateOneMethod().getFullMethodName(), response.getStatus());
                    return response;
                }), responseObserver);
    }

    @Override
    public void createMany(CreateManyRequest request, StreamObserver<CreateManyResponse> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> metabaseService.createMany(ensureValid(request)))
                .exceptionally(e -> {
                    var status = classifyError(e);
                    logError("createMany", CreateManyRequest.getDefaultInstance(), status.code, e);

                    return CreateManyResponse.newBuilder()
                            .setStatus(status.code)
                            .setStatusMessage(status.message)
                            .build();
                })
                .thenApply(response -> {
                    boolean useNewFormat = request.getMetricsList().stream()
                        .noneMatch(s -> s.getName().isEmpty());
                    incrementFormat(MetabaseServiceGrpc.getCreateManyMethod().getFullMethodName(), useNewFormat);
                    incrementStatus(MetabaseServiceGrpc.getCreateManyMethod().getFullMethodName(), response.getStatus());
                    return response;
                }), responseObserver);
    }

    @Override
    public void resolveOne(ResolveOneRequest request, StreamObserver<ResolveOneResponse> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> metabaseService.resolveOne(ensureValid(request)))
                .exceptionally(e -> {
                    var status = classifyError(e);
                    logError("resolveOne", request, status.code, e);

                    return ResolveOneResponse.newBuilder()
                            .setStatus(status.code)
                            .setStatusMessage(status.message)
                            .build();
                })
                .thenApply(response -> {
                    boolean useNewFormat = !request.getName().isEmpty();
                    incrementFormat(MetabaseServiceGrpc.getResolveOneMethod().getFullMethodName(), useNewFormat);
                    incrementStatus(MetabaseServiceGrpc.getResolveOneMethod().getFullMethodName(), response.getStatus());
                    return response;
                }), responseObserver);
    }

    @Override
    public void resolveMany(ResolveManyRequest request, StreamObserver<ResolveManyResponse> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> metabaseService.resolveMany(ensureValid(request)))
                .exceptionally(e -> {
                    var status = classifyError(e);
                    logError("resolveMany", ResolveManyRequest.getDefaultInstance(), status.code, e);

                    return ResolveManyResponse.newBuilder()
                            .setStatus(status.code)
                            .setStatusMessage(status.message)
                            .build();
                })
                .thenApply(response -> {
                    boolean useNewFormat = !request.getMetricsList().isEmpty();
                    incrementFormat(MetabaseServiceGrpc.getResolveManyMethod().getFullMethodName(), useNewFormat);
                    incrementStatus(MetabaseServiceGrpc.getResolveManyMethod().getFullMethodName(), response.getStatus());
                    return response;
                }), responseObserver);
    }

    @Override
    public void deleteMany(DeleteManyRequest request, StreamObserver<DeleteManyResponse> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> metabaseService.deleteMany(ensureValid(request)))
                .exceptionally(e -> {
                    var status = classifyError(e);
                    logError("deleteMany", request, status.code, e);

                    return DeleteManyResponse.newBuilder()
                            .setStatus(status.code)
                            .setStatusMessage(status.message)
                            .build();

                })
                .thenApply(response -> {
                    boolean useNewFormat = request.getMetricsList().stream()
                        .noneMatch(s -> s.getName().isEmpty());
                    incrementFormat(MetabaseServiceGrpc.getDeleteManyMethod().getFullMethodName(), useNewFormat);
                    incrementStatus(MetabaseServiceGrpc.getDeleteManyMethod().getFullMethodName(), response.getStatus());
                    return response;
                }), responseObserver);
    }

    @Override
    public void find(FindRequest request, StreamObserver<FindResponse> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> metabaseService.find(ensureValid(request)))
                .exceptionally(e -> {
                    var status = classifyError(e);
                    logError("find", request, status.code, e);

                    return FindResponse.newBuilder()
                            .setStatus(status.code)
                            .setStatusMessage(status.message)
                            .build();

                })
                .thenApply(response -> {
                    boolean useNewFormat = request.hasNewSelectors();
                    incrementFormat(MetabaseServiceGrpc.getFindMethod().getFullMethodName(), useNewFormat);
                    incrementStatus(MetabaseServiceGrpc.getFindMethod().getFullMethodName(), response.getStatus());
                    return response;
                }), responseObserver);
    }

    @Override
    public void labelValues(TLabelValuesRequest request, StreamObserver<TLabelValuesResponse> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> metabaseService.labelValues(ensureValid(request)))
                .handle((response, e) -> {
                    if (e != null) {
                        var status = classifyError(e);
                        logError("labelValues", request, status.code, e);

                        response = TLabelValuesResponse.newBuilder()
                                .setStatus(status.code)
                                .setStatusMessage(status.message)
                                .build();
                    }

                    boolean useNewFormat = request.hasNewSelectors();
                    incrementFormat(MetabaseServiceGrpc.getLabelValuesMethod().getFullMethodName(), useNewFormat);
                    incrementStatus(MetabaseServiceGrpc.getLabelValuesMethod().getFullMethodName(), response.getStatus());
                    return response;
                }), responseObserver);
    }

    @Override
    public void metricNames(MetricNamesRequest request, StreamObserver<MetricNamesResponse> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> metabaseService.metricNames(ensureValid(request)))
                .handle((response, e) -> {
                    if (e != null) {
                        var status = classifyError(e);
                        logError("metricNames", request, status.code, e);

                        response = MetricNamesResponse.newBuilder()
                                .setStatus(status.code)
                                .setStatusMessage(status.message)
                                .build();
                    }

                    incrementStatus(MetabaseServiceGrpc.getMetricNamesMethod().getFullMethodName(), response.getStatus());
                    return response;
                }), responseObserver);
    }

    @Override
    public void labelNames(TLabelNamesRequest request, StreamObserver<TLabelNamesResponse> responseObserver) {
        asyncComplete(
            CompletableFutures.safeCall(() -> metabaseService.labelNames(ensureValid(request)))
                .handle((response, e) -> {
                    if (e != null) {
                        var status = classifyError(e);
                        logError("labelNames", request, status.code, e);

                        response = TLabelNamesResponse.newBuilder()
                            .setStatus(status.code)
                            .setStatusMessage(status.message)
                            .build();
                    }

                    boolean useNewFormat = request.hasNewSelectors();
                    incrementFormat(MetabaseServiceGrpc.getLabelNamesMethod().getFullMethodName(), useNewFormat);
                    incrementStatus(MetabaseServiceGrpc.getLabelNamesMethod().getFullMethodName(), response.getStatus());
                    return response;
                }), responseObserver);
    }

    @Override
    public void uniqueLabels(TUniqueLabelsRequest request, StreamObserver<TUniqueLabelsResponse> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> metabaseService.uniqueLabels(ensureValid(request)))
                .handle((response, e) -> {
                    if (e != null) {
                        var status = classifyError(e);
                        logError("uniqueLabels", request, status.code, e);

                        response = TUniqueLabelsResponse.newBuilder()
                                .setStatus(status.code)
                                .setStatusMessage(status.message)
                                .build();
                    }

                    boolean useNewFormat = request.hasNewSelectors();
                    incrementFormat(MetabaseServiceGrpc.getUniqueLabelsMethod().getFullMethodName(), useNewFormat);
                    incrementStatus(MetabaseServiceGrpc.getUniqueLabelsMethod().getFullMethodName(), response.getStatus());
                    return response;
                }), responseObserver);
    }

    @Override
    public void resolveLogs(TResolveLogsRequest request, StreamObserver<TResolveLogsResponse> responseObserver) {
        asyncComplete(CompletableFutures.safeCall(() -> metabaseService.resolveLogs(request))
            .handle((response, e) -> {
                if (e != null) {
                    var status = classifyError(e);
                    logError("resolveLogs", TResolveLogsRequest.getDefaultInstance(), status.code, e);

                    response = TResolveLogsResponse.newBuilder()
                        .setStatus(status.code)
                        .setStatusMessage(status.message)
                        .build();
                }

                incrementStatus(MetabaseServiceGrpc.getResolveLogsMethod().getFullMethodName(), response.getStatus());
                return response;
            }), responseObserver);
    }

    static Status classifyError(Throwable throwable) {
        Throwable cause = throwable;
        while (cause != null) {
            if (cause instanceof ValidationException) {
                return new Status(EMetabaseStatusCode.INVALID_REQUEST, cause.getMessage());
            }

            if (cause instanceof InvalidMetricNameException) {
                return new Status(EMetabaseStatusCode.INVALID_REQUEST, cause.getMessage());
            }

            if (cause instanceof SelectorsException) {
                return new Status(EMetabaseStatusCode.INVALID_REQUEST, cause.getMessage());
            }

            if (cause instanceof ThreadLocalTimeoutException) {
                return new Status(EMetabaseStatusCode.DEADLINE_EXCEEDED, cause.getMessage());
            }

            if (cause instanceof MetabaseNotInitialized) {
                return new Status(EMetabaseStatusCode.NODE_UNAVAILABLE, cause.getMessage());
            }

            if (cause.getCause() instanceof ShardIsNotLocalException) {
                return new Status(EMetabaseStatusCode.SHARD_NOT_FOUND, cause.getMessage());
            }

            if (cause instanceof ShardIsNotLocalException) {
                return new Status(EMetabaseStatusCode.SHARD_NOT_FOUND, cause.getMessage());
            }

            if (cause instanceof ShardNotReadyException) {
                return new Status(EMetabaseStatusCode.SHARD_NOT_READY, cause.getMessage());
            }

            if (cause instanceof ShardWriteOnlyException) {
                return new Status(EMetabaseStatusCode.SHARD_WRITE_ONLY, cause.getMessage());
            }

            if (cause instanceof ShardReadOnlyException) {
                return new Status(EMetabaseStatusCode.SHARD_READ_ONLY, cause.getMessage());
            }

            if (cause instanceof UrlStatusTypeAwareException url) {
                return switch (url.urlStatusType()) {
                    case QUOTA_ERROR -> new Status(EMetabaseStatusCode.QUOTA_ERROR, cause.getMessage());
                    case SHARD_NOT_INITIALIZED, SHARD_IS_NOT_WRITABLE -> new Status(EMetabaseStatusCode.SHARD_NOT_READY, cause.getMessage());
                    case IPC_QUEUE_OVERFLOW -> new Status(EMetabaseStatusCode.RESOURCE_EXHAUSTED, cause.getMessage());
                    case UNKNOWN_SHARD -> new Status(EMetabaseStatusCode.SHARD_NOT_FOUND, cause.getMessage());
                    default -> new Status(EMetabaseStatusCode.INTERNAL_ERROR, cause.getMessage());
                };
            }

            var grpcStatus = grpcExceptionStatus(cause);
            if (grpcStatus != null) {
                return new Status(GrpcStatusMapping.toMetabaseStatusCode(grpcStatus.getCode()), throwable.getMessage());
            }

            cause = cause.getCause();
        }

        return new Status(EMetabaseStatusCode.INTERNAL_ERROR, throwable.getMessage());
    }

    @Nullable
    private static io.grpc.Status grpcExceptionStatus(Throwable cause) {
        if (cause instanceof StatusException) {
            return ((StatusException) cause).getStatus();
        } else if (cause instanceof StatusRuntimeException) {
            return ((StatusRuntimeException) cause).getStatus();
        }
        return null;
    }

    public static void logError(String endpoint, Message request, EMetabaseStatusCode code, Throwable error) {
        if (isWarn(code)) {
            logger.warn("{} - {}", code, endpoint);
        } else {
            logger.error("{} - {} for request {}", code, endpoint, TextFormat.shortDebugString(request), error);
        }
    }

    private static boolean isWarn(EMetabaseStatusCode code) {
        switch (code) {
            case DEADLINE_EXCEEDED:
            case SHARD_NOT_FOUND:
            case SHARD_NOT_READY:
            case NODE_UNAVAILABLE:
            case NOT_FOUND:
            case QUOTA_ERROR:
                return true;
            default:
                return false;
        }
    }

    private void incrementStatus(String endpoint, EMetabaseStatusCode code) {
        registry.subRegistry("endpoint", endpoint)
                .subRegistry("code", code.name())
                .rate("metabaseServer.responseStatus")
                .inc();
    }

    private void incrementFormat(String endpoint, boolean useNewFormat) {
        registry.subRegistry("endpoint", endpoint)
            .rate("metabaseServer.requestsIn" + (useNewFormat ? "NewFormat" : "OldFormat"))
            .inc();
    }

    static class Status {
        final EMetabaseStatusCode code;
        final String message;

        Status(EMetabaseStatusCode code, String message) {
            this.code = code;
            this.message = message;
        }
    }
}
