package ru.yandex.solomon.coremon.meta.service;

import java.util.Arrays;
import java.util.List;

import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.LabelValidator;
import ru.yandex.solomon.metabase.api.protobuf.CreateManyRequest;
import ru.yandex.solomon.metabase.api.protobuf.CreateOneRequest;
import ru.yandex.solomon.metabase.api.protobuf.DeleteManyRequest;
import ru.yandex.solomon.metabase.api.protobuf.FindRequest;
import ru.yandex.solomon.metabase.api.protobuf.Metric;
import ru.yandex.solomon.metabase.api.protobuf.MetricNamesRequest;
import ru.yandex.solomon.metabase.api.protobuf.ResolveManyRequest;
import ru.yandex.solomon.metabase.api.protobuf.ResolveOneRequest;
import ru.yandex.solomon.metabase.api.protobuf.TLabelNamesRequest;
import ru.yandex.solomon.metabase.api.protobuf.TLabelValuesRequest;
import ru.yandex.solomon.metabase.api.protobuf.TServerStatusRequest;
import ru.yandex.solomon.metabase.api.protobuf.TUniqueLabelsRequest;
import ru.yandex.solomon.model.protobuf.Label;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.shard.StockpileShardId;

/**
 * @author Vladimir Gordiychuk
 */
final class GrpcMetabaseValidator {

    private static final int SHARD_KEY_LENGTH = 3; // project, cluster, service

    private GrpcMetabaseValidator() {
    }

    static TServerStatusRequest ensureValid(TServerStatusRequest request) {
        return request;
    }

    static CreateOneRequest ensureValid(CreateOneRequest request) {
        Metric metric = request.getMetric();
        ensureMetricTypeValid(metric, metric.getType());
        ensureMetricIdValid(metric);

        List<Label> labels = metric.getLabelsList();
        if (labels.size() < SHARD_KEY_LENGTH) {
            throw invalid("New metric should have all shard keys: %s, but metric have only: %s",
                Arrays.asList(LabelKeys.PROJECT, LabelKeys.CLUSTER, LabelKeys.SERVICE), labels);
        }

        if (labels.size() == SHARD_KEY_LENGTH) {
            throw invalid("Metric should have at least one label apart from shard key, but was ", labels);
        }

        ensureLabelsValid(labels);
        return request;
    }

    static CreateManyRequest ensureValid(CreateManyRequest request) {
        if (request.getCommonLabelsCount() < SHARD_KEY_LENGTH) {
            throw invalid("Common labels should have all shard keys parts: %s, but common labels have: %s",
                Arrays.asList(LabelKeys.PROJECT, LabelKeys.CLUSTER, LabelKeys.SERVICE), request.getCommonLabelsList());
        }

        ensureLabelsValid(request.getCommonLabelsList());

        MetricType commonType = request.getType();
        for (Metric metric : request.getMetricsList()) {
            MetricType type = metric.getType();
            if (type == MetricType.UNRECOGNIZED || type == MetricType.METRIC_TYPE_UNSPECIFIED) {
                type = commonType;
            }

            ensureMetricTypeValid(metric, type);
            ensureLabelsValid(metric.getLabelsList());
            ensureMetricIdValid(metric);
        }

        return request;
    }

    static ResolveOneRequest ensureValid(ResolveOneRequest request) {
        return request;
    }

    static ResolveManyRequest ensureValid(ResolveManyRequest request) {
        return request;
    }

    static DeleteManyRequest ensureValid(DeleteManyRequest request) {
        return request;
    }

    static FindRequest ensureValid(FindRequest request) {
        return request;
    }

    static TLabelValuesRequest ensureValid(TLabelValuesRequest request) {
        return request;
    }

    static MetricNamesRequest ensureValid(MetricNamesRequest request) {
        return request;
    }

    static TLabelNamesRequest ensureValid(TLabelNamesRequest request) {
        return request;
    }

    static TUniqueLabelsRequest ensureValid(TUniqueLabelsRequest request) {
        if (request.getNamesCount() == 0) {
            throw invalid("No specified labels names to group: %s", request);
        }
        return request;
    }

    private static ValidationException invalid(String message, Object... args) {
        throw new ValidationException(String.format(message, args));
    }

    private static void ensureMetricTypeValid(Metric metric, MetricType type) {
        if (type == MetricType.UNRECOGNIZED || type == MetricType.METRIC_TYPE_UNSPECIFIED) {
            throw invalid("Unsupported metric type %s for metric %s", type, metric.getLabelsList());
        }
    }

    private static void ensureMetricIdValid(Metric metric) {
        var id = metric.getMetricId();
        if (id.getShardId() == 0 && id.getLocalId() == 0) {
            return;
        }
        StockpileShardId.validate(id.getShardId());
        StockpileLocalId.validate(id.getLocalId());
    }

    private static void ensureLabelsValid(List<Label> labels) {
        for (Label label : labels) {
            if (LabelKeys.isShardKeyPart(label.getKey())) {
                continue;
            }

            if (!LabelValidator.isValidName(label.getKey())) {
                throw invalid("Invalid label name %s, valid pattern ^[-0-9a-zA-Z_/]{1,100}$",
                    label.getKey());
            }

            if (!LabelValidator.isValidValue(label.getValue())) {
                throw invalid("Invalid value %s for label name %s, valid pattern ^[-0-9a-zA-Z_/]{1,200}$",
                    label.getKey(), label.getValue());
            }
        }
    }
}
