package ru.yandex.solomon.coremon.meta.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monitoring.coremon.EShardState;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Label;
import ru.yandex.monlib.metrics.labels.LabelAllocator;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.labels.LabelsBuilder;
import ru.yandex.monlib.metrics.labels.validate.LabelValidationFilter;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.core.db.model.ShardState;
import ru.yandex.solomon.core.urlStatus.UrlStatusTypeException;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.CoremonMetricArray;
import ru.yandex.solomon.coremon.meta.CoremonMetricsResolver;
import ru.yandex.solomon.coremon.meta.FileCoremonMetric;
import ru.yandex.solomon.coremon.meta.MetabaseMetricId;
import ru.yandex.solomon.coremon.meta.MetricsCollection;
import ru.yandex.solomon.coremon.meta.NamedCoremonMetric;
import ru.yandex.solomon.coremon.meta.db.MetabaseShardStorageImpl;
import ru.yandex.solomon.coremon.meta.db.MetricsDaoFactory;
import ru.yandex.solomon.coremon.meta.db.StockpileMetricIdFactory;
import ru.yandex.solomon.coremon.meta.file.FileMetricsCollection;
import ru.yandex.solomon.coremon.meta.mem.MemOnlyMetricsCollectionImpl;
import ru.yandex.solomon.labels.protobuf.LabelConverter;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.query.SelectorsException;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.memory.layout.MemoryBySubsystem;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metabase.api.protobuf.Metric;
import ru.yandex.solomon.metabase.api.protobuf.TResolveLogsRequest;
import ru.yandex.solomon.metabase.api.protobuf.TResolveLogsResponse;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.model.protobuf.MetricTypeConverter;
import ru.yandex.solomon.proto.UrlStatusType;
import ru.yandex.solomon.util.MetricNameHelper;
import ru.yandex.solomon.util.labelStats.LabelStats;
import ru.yandex.solomon.util.labelStats.LabelValuesStats;
import ru.yandex.solomon.util.time.InstantUtils;
import ru.yandex.stockpile.api.DeleteMetricRequest;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.TShardCommandResponse;
import ru.yandex.stockpile.client.StockpileClient;
import ru.yandex.stockpile.client.mem.AccumulatedShardCommand;
import ru.yandex.stockpile.client.mem.CrossShardCommandAccumulator;


/**
 * @author Sergey Polovko
 */
public class MetabaseShardImpl implements MetabaseShard {

    private final int numId;
    private final String id;
    private final String folderId;
    private final int numPartitions;
    private final ShardKey shardKey;
    private final LabelAllocator labelAllocator;
    private final MetabaseShardStorageImpl storage;
    private final StockpileClient stockpileClient;
    private final AtomicReference<String> metricNameLabelRef;
    private final AtomicBoolean onlyNewFormatReadsRef;
    private final AtomicBoolean onlyNewFormatWritesRef;
    private final AtomicReference<String> serviceProviderRef;
    private final long createdAtMillis = System.currentTimeMillis();
    private volatile ShardState state = ShardState.ACTIVE;
    private volatile int maxFileMetrics = Shard.DEFAULT_FILE_METRIC_QUOTA;

    private final MetabaseShardMetrics metrics = null;

    public MetabaseShardImpl(
        int numId,
        String id,
        String folderId,
        int numPartitions,
        ShardKey shardKey,
        MetricsDaoFactory metricsDaoFactory,
        LabelAllocator labelAllocator,
        Executor miscThreadPool,
        StockpileClient stockpileClient)
    {
        this.numId = numId;
        this.id = id;
        this.folderId = folderId;
        this.shardKey = shardKey;
        this.numPartitions = numPartitions;
        this.labelAllocator = labelAllocator;
        this.storage = new MetabaseShardStorageImpl(
            id,
            metricsDaoFactory.create(getNumId(), labelAllocator),
            new StockpileMetricIdFactory(numId, stockpileClient),
            miscThreadPool);
        this.stockpileClient = stockpileClient;

        this.metricNameLabelRef = new AtomicReference<>("");
        this.onlyNewFormatReadsRef = new AtomicBoolean();
        this.onlyNewFormatWritesRef = new AtomicBoolean();
        this.serviceProviderRef = new AtomicReference<>("");
    }

    @Override
    public void setMetricNameLabel(String metricNameLabel) {
        metricNameLabelRef.set(metricNameLabel);
    }

    @Override
    public void setOnlyNewFormatWrites(boolean onlyNewFormatWrites) {
        onlyNewFormatWritesRef.set(onlyNewFormatWrites);
    }

    @Override
    public void setOnlyNewFormatReads(boolean onlyNewFormatReads) {
        onlyNewFormatReadsRef.set(onlyNewFormatReads);
    }

    @Override
    public void setServiceProvider(String serviceProvider) {
        serviceProviderRef.set(serviceProvider);
    }

    @Override
    public int getNumId() {
        return numId;
    }

    @Override
    public String getId() {
        return id;
    }

    @Override
    public String getFolderId() {
        return folderId;
    }

    @Override
    public int getNumPartitions() {
        return numPartitions;
    }

    @Override
    public ShardKey getShardKey() {
        return shardKey;
    }

    public String getMetricNameLabel() {
        return metricNameLabelRef.get();
    }

    @Override
    public String getServiceProvider() {
        return serviceProviderRef.get();
    }

    public boolean isSupportsMetricNames() {
        return !metricNameLabelRef.get().isEmpty();
    }

    public ShardState getShardState() {
        return state;
    }

    @Override
    public void setShardState(ShardState state) {
        this.state = state;
    }

    @Override
    public void setMaxFileMetrics(int maxFileMetrics) {
        this.maxFileMetrics = maxFileMetrics;
    }

    public void awaitReady() {
        storage.awaitLoadComplete();
    }

    @Override
    public boolean isLoaded() {
        return storage.isLoaded();
    }

    public boolean isReadyWrite() {
        return state != ShardState.READ_ONLY && state != ShardState.INACTIVE;
    }

    public boolean isReadyAcceptNewMetrics() {
        if (!isReadyWrite()) {
            return false;
        }

        return !reachFileMetricsQuota();
    }

    public boolean isReadyRead() {
        return state != ShardState.WRITE_ONLY && state != ShardState.INACTIVE;
    }

    @Nullable
    public NamedCoremonMetric resolveMetric(MetabaseMetricId metabaseMetricId, boolean useNewFormat) {
        validateNewFormatReads(useNewFormat);

        if (useNewFormat) {
            return resolveMetricInNewFormat(metabaseMetricId);
        }

        CoremonMetric coremonMetric = resolveMetricInOldFormat(metabaseMetricId.getLabels());
        return coremonMetric == null ? null : NamedCoremonMetric.from(coremonMetric);
    }

    @Nullable
    private NamedCoremonMetric resolveMetricInNewFormat(MetabaseMetricId metabaseMetricId) {
        validateMetricNameSupport();

        String metricNameLabel = getMetricNameLabel();

        return resolveMetricInNewFormat(metabaseMetricId, metricNameLabel);
    }

    private NamedCoremonMetric resolveMetricInNewFormat(
        MetabaseMetricId metabaseMetricId,
        String metricNameLabel)
    {
        validateMetricNameSupport();

        Labels internalKey = convertToInternalLabels(metabaseMetricId, metricNameLabel);

        try (CoremonMetric metric = storage.getFileMetrics().getOrNull(internalKey)) {
            return metric == null ? null : NamedCoremonMetric.from(metric, metricNameLabel);
        }
    }

    @Nullable
    private CoremonMetric resolveMetricInOldFormat(Labels key) {
        try (CoremonMetric metric = storage.getFileMetrics().getOrNull(key)) {
            return metric == null ? null : new FileCoremonMetric(metric);
        }
    }

    private List<CoremonMetric> readMetrics(List<MetabaseMetricId> keys, boolean useNewFormat) {
        validateNewFormatReads(useNewFormat);

        if (useNewFormat) {
            validateMetricNameSupport();
        }

        if (!isLoaded()) {
            throw new RuntimeException("Shard not initialized yet");
        }

        String metricNameLabel = getMetricNameLabel();

        return keys.stream()
            .map(key -> {
                Labels internalLabels = useNewFormat ? convertToInternalLabels(key, metricNameLabel) : key.getLabels();
                return resolveMetricInOldFormat(internalLabels);
            })
            .filter(Objects::nonNull)
            .collect(Collectors.toList());
    }

    public CompletableFuture<List<NamedCoremonMetric>> deleteMetrics(List<MetabaseMetricId> keys, boolean useNewFormat) {
        List<CoremonMetric> metricsToRemove = readMetrics(keys, useNewFormat);

        CrossShardCommandAccumulator accumulator = new CrossShardCommandAccumulator();
        for (CoremonMetric metric : metricsToRemove) {
            accumulator.append(DeleteMetricRequest.newBuilder()
                .setMetricId(MetricId.newBuilder()
                    .setShardId(metric.getShardId())
                    .setLocalId(metric.getLocalId())
                    .build())
                .build());
        }

        Collection<AccumulatedShardCommand> batches = accumulator.build().values();
        return CompletableFutures.allOf(batches.stream()
            .map(stockpileClient::bulkShardCommand)
            .collect(Collectors.toList()))
            .thenAccept(responses -> {
                List<TShardCommandResponse> failedResponses =
                    responses.stream()
                        .filter(response -> response.getStatus() != EStockpileStatusCode.OK)
                        .collect(Collectors.toList());
                if (!failedResponses.isEmpty()) {
                    throw new RuntimeException("Not all request successfully executed into stockpile: " + failedResponses);
                }
            })
            .thenCompose(aVoid -> storage.remove(metricsToRemove))
            .thenApply(coremonMetrics -> convertToNamedCoremonMetrics(useNewFormat, coremonMetrics));
    }

    /**
     * @return total number of matched metrics
     */
    public int searchMetrics(Selectors metricSelectors, int offset, int limit, boolean useNewFormat, Consumer<NamedCoremonMetric> fn) {
        validateNewFormatReads(useNewFormat);

        if (useNewFormat) {
            return searchMetricsInNewFormat(metricSelectors, offset, limit, fn);
        }

        return searchMetricsInOldFormat(metricSelectors, offset, limit, fn);
    }

    private int searchMetricsInOldFormat(Selectors metricSelectors, int offset, int limit, Consumer<NamedCoremonMetric> fn) {
        return searchMetrics(metricSelectors, offset, limit, coremonMetric -> {
            fn.accept(NamedCoremonMetric.from(coremonMetric));
        });
    }

    private int searchMetricsInNewFormat(Selectors metricSelectors, int offset, int limit, Consumer<NamedCoremonMetric> fn) {
        validateMetricNameSupport();

        String metricNameLabel = getMetricNameLabel();

        Selectors internalSelectors = convertToInternalSelectors(metricSelectors, metricNameLabel);

        return searchMetrics(internalSelectors, offset, limit, coremonMetric -> {
            fn.accept(NamedCoremonMetric.from(coremonMetric, metricNameLabel));
        });
    }

    private int searchMetrics(Selectors metricSelectors, int offset, int limit, Consumer<CoremonMetric> fn) {
        if (!isLoaded()) {
            throw new RuntimeException("Shard not initialized yet");
        }
        return storage.getFileMetrics().searchMetrics(metricSelectors, offset, limit, fn);
    }

    /* Metric names */
    public LabelStats searchMetricNames(Selectors metricSelectors, String textSearch, LabelValidationFilter validationFilter) {
        LabelStats metricNameStats = LabelStats.create();

        String metricNameLabel = getMetricNameLabel();
        Selectors internalSelectors =
            convertToInternalSelectors(metricSelectors, metricNameLabel);

        storage.getFileMetrics().searchMetrics(internalSelectors, labels -> {
            Label label = labels.getLabels().findByKey(metricNameLabel);

            if (label != null) {
                String metricName = label.getValue();

                if (!textSearch.isEmpty() && !StringUtils.containsIgnoreCase(metricName, textSearch)) {
                    return;
                }

                if (!validationFilter.filterValue(metricName)) {
                    return;
                }

                metricNameStats.add(metricName);
            }
        });

        return metricNameStats;
    }

    /* Search labels */
    public int searchLabels(Selectors metricSelectors, int offset, int limit, boolean useNewFormat, Consumer<Labels> fn) {
        validateNewFormatReads(useNewFormat);

        if (useNewFormat) {
            return searchLabelsInNewFormat(metricSelectors, offset, limit, fn);
        }

        return searchLabelsInOldFormat(metricSelectors, offset, limit, fn);
    }

    public int searchLabelsInNewFormat(Selectors metricSelectors, int offset, int limit, Consumer<Labels> fn) {
        if (!isSupportsMetricNames()) {
            throw new RuntimeException("Shard doesn't support new format");
        }

        String metricNameLabel = getMetricNameLabel();

        Selectors internalSelectors = convertToInternalSelectors(metricSelectors, metricNameLabel);

        return searchLabelsInOldFormat(internalSelectors, offset, limit, internalLabels -> {
            Labels newLabels = internalLabels.removeByKey(metricNameLabel);
            fn.accept(newLabels);
        });
    }

    private int searchLabelsInOldFormat(Selectors metricSelectors, int offset, int limit, Consumer<Labels> fn) {
        if (!isLoaded()) {
            throw new RuntimeException("Shard not initialized yet");
        }
        return storage.getFileMetrics().searchLabels(metricSelectors, offset, limit, fn);
    }

    /* Search count */
    public int searchCount(Selectors metricSelectors, boolean useNewFormat) {
        validateNewFormatReads(useNewFormat);

        if (useNewFormat) {
            return searchCountInNewFormat(metricSelectors);
        }

        return searchCountInOldFormat(metricSelectors);
    }

    private int searchCountInNewFormat(Selectors metricSelectors) {
        validateMetricNameSupport();

        String metricNameLabel = getMetricNameLabel();

        Selectors internalSelectors = convertToInternalSelectors(metricSelectors, metricNameLabel);

        return searchCountInOldFormat(internalSelectors);
    }

    private int searchCountInOldFormat(Selectors metricSelectors) {
        if (!isLoaded()) {
            throw new RuntimeException("Shard not initialized yet");
        }
        return storage.getFileMetrics().searchCount(metricSelectors);
    }

    @Override
    public int fileMetricsCount() {
        return storage.getFileMetrics().size();
    }

    @Override
    public int maxFileMetrics() {
        return maxFileMetrics;
    }

    @Override
    public int fileAndMemOnlyMetrics() {
        int count = storage.getFileMetrics().size();
        if (count == 0 && storage.getState() != EShardState.READY) {
            count = (int) Math.max(storage.getEstimatedRowsTotal(), storage.getRowsLoaded());
        }
        return count + storage.getMemOnlyMetrics().size();
    }

    /* Label stats */
    public LabelValuesStats labelStats(Set<String> requestedNames, boolean useNewFormat) {
        validateNewFormatReads(useNewFormat);

        if (useNewFormat) {
            return labelStatsInNewFormat(requestedNames);
        }

        return labelStatsInOldFormat(requestedNames);
    }

    private LabelValuesStats labelStatsInNewFormat(Set<String> requestedNames) {
        validateMetricNameSupport();

        String metricNameLabel = getMetricNameLabel();

        Set<String> internalRequestedNames = requestedNames.stream()
            .filter(name -> !metricNameLabel.equals(name))
            .collect(Collectors.toSet());

        LabelValuesStats internalLabelStats = labelStatsInOldFormat(internalRequestedNames);

        internalLabelStats.removeByKey(metricNameLabel);

        return internalLabelStats;
    }

    private LabelValuesStats labelStatsInOldFormat(Set<String> requestedNames) {
        if (!isLoaded()) {
            throw new RuntimeException("Shard not initialized yet");
        }
        return storage.getFileMetrics().labelStats(requestedNames);
    }

    /* Create metrics */
    public CompletableFuture<List<NamedCoremonMetric>> createMetrics(Labels commonLabels, List<Metric> metrics) {
        boolean useNewFormat = metrics.stream().noneMatch(s -> s.getName().isEmpty());

        validateNewFormatWrites(useNewFormat);

        if (useNewFormat) {
            validateMetricNameSupport();
        }

        if (!isLoaded()) {
            throw new RuntimeException("Shard is not initialized yet");
        }

        if (reachFileMetricsQuota()) {
            String message = "more than " + maxFileMetrics + " metrics in shard " + getId();
            return CompletableFuture.failedFuture(new UrlStatusTypeException(message, UrlStatusType.QUOTA_ERROR));
        }

        MetricsCollection<CoremonMetric> fileMetrics = storage.getFileMetrics();

        List<NamedCoremonMetric> result = new ArrayList<>(metrics.size());
        List<Labels> lookup = new ArrayList<>();
        CoremonMetricArray toCreate = new CoremonMetricArray(metrics.size());

        String metricNameLabel = getMetricNameLabel();

        try {
            var metricIdProvider = storage.getMetricIdFactory().forLabels();
            for (var metric : metrics) {
                LabelsBuilder metricKeyBuilder = LabelConverter.protoToLabelsBuilder(metric.getLabelsList(), labelAllocator)
                    .addAll(commonLabels);

                MetricNameHelper.convertToOldFormat(metricKeyBuilder, metricNameLabel, metric.getName());
                Labels internalMetricKey = metricKeyBuilder.build();

                try (CoremonMetric coremonMetric = fileMetrics.getOrNull(internalMetricKey)) {
                    MetricType type = MetricTypeConverter.fromProto(metric.getType());
                    if (coremonMetric == null) {
                        int shardId = metric.getMetricId().getShardId();
                        long localId = metric.getMetricId().getLocalId();

                        if (shardId == 0) {
                            var metricId = metricIdProvider.metricId(internalMetricKey);
                            shardId = metricId.getShardId();
                            localId = metricId.getLocalId();
                        }

                        int createdAtSeconds = InstantUtils.currentTimeSeconds();
                        toCreate.add(shardId, localId, internalMetricKey, createdAtSeconds, type);
                        lookup.add(internalMetricKey);
                    } else if (coremonMetric.getType() != type) {
                        toCreate.add(coremonMetric.getShardId(), coremonMetric.getLocalId(), internalMetricKey, coremonMetric
                            .getCreatedAtSeconds(), type);
                        lookup.add(internalMetricKey);
                    } else {
                        result.add(toNamedMetric(useNewFormat, metricNameLabel, coremonMetric));
                    }
                }
            }

            if (toCreate.isEmpty()) {
                toCreate.close();
                return CompletableFuture.completedFuture(result);
            }

            return storage.write(toCreate)
                .thenApply(ignore -> {
                    for (var labels : lookup) {
                        try (var metric = fileMetrics.getOrNull(labels)) {
                            if (metric == null) {
                                continue;
                            }
                            result.add(toNamedMetric(useNewFormat, metricNameLabel, metric));
                        }
                    }
                    return result;
                });
        } catch (Throwable t) {
            toCreate.close();
            return CompletableFuture.failedFuture(t);
        }
    }

    public CompletableFuture<TResolveLogsResponse> resolveLogs(TResolveLogsRequest request) {
        int numId = request.getNumId();
        var fileMetrics = storage.getFileMetrics();
        var resolveResults = new ArrayList<CoremonMetricsResolver.ResolveResult>(request.getUnresolvedLogMetaCount());
        int unresolvedCnt = 0;
        var metricIdProvider = storage.getMetricIdFactory().forLabels();
        try {
            for (var meta : request.getUnresolvedLogMetaList()) {
                var result = CoremonMetricsResolver.resolve(meta, fileMetrics, metricIdProvider, labelAllocator);
                unresolvedCnt += result.unresolved.size();
                resolveResults.add(result);
            }

            if (unresolvedCnt == 0) {
                var result = TResolveLogsResponse.newBuilder()
                    .setStatus(EMetabaseStatusCode.OK)
                    .addAllResolvedLogMetrics(CoremonMetricsResolver.pack(numId, resolveResults))
                    .build();
                CoremonMetricsResolver.close(resolveResults);
                return CompletableFuture.completedFuture(result);
            }

            if (reachFileMetricsQuota()) {
                var result = TResolveLogsResponse.newBuilder()
                    .setStatus(EMetabaseStatusCode.QUOTA_ERROR)
                    .setStatusMessage("more than " + maxFileMetrics + " metrics in shard " + getId())
                    .addAllResolvedLogMetrics(CoremonMetricsResolver.pack(numId, resolveResults))
                    .build();
                CoremonMetricsResolver.close(resolveResults);
                return CompletableFuture.completedFuture(result);
            }

            return storage.write(CoremonMetricsResolver.combineUnresolved(resolveResults, unresolvedCnt))
                .thenApply(ignore -> {
                    var response = TResolveLogsResponse.newBuilder()
                        .setStatus(EMetabaseStatusCode.OK);

                    var metrics = storage.getFileMetrics();
                    for (var resolved : resolveResults) {
                        CoremonMetricsResolver.resolveUnresolved(resolved.resolved, resolved.unresolved, metrics);
                        response.addResolvedLogMetrics(CoremonMetricsResolver.pack(numId, resolved.resolved));
                    }

                    return response.build();
                })
                .whenComplete((ignore, e) -> CoremonMetricsResolver.close(resolveResults));
        } catch (Throwable e) {
            CoremonMetricsResolver.close(resolveResults);
            return CompletableFuture.failedFuture(e);
        }
    }

    private NamedCoremonMetric toNamedMetric(boolean useNewFormat, String metricNameLabel, CoremonMetric coremonMetric) {
        if (useNewFormat) {
            return NamedCoremonMetric.from(coremonMetric, metricNameLabel);
        }

        return NamedCoremonMetric.from(coremonMetric);
    }

    /* Label names */
    public Set<String> labelNames(boolean useNewFormat) {
        validateNewFormatReads(useNewFormat);

        if (useNewFormat) {
            return labelNamesInNewFormat();
        }

        return labelNamesInOldFormat();
    }

    private Set<String> labelNamesInNewFormat() {
        validateMetricNameSupport();

        String metricNameLabel = getMetricNameLabel();

        return labelNamesInOldFormat().stream()
            .filter(name -> !metricNameLabel.equals(name))
            .collect(Collectors.toSet());
    }

    private Set<String> labelNamesInOldFormat() {
        if (!isLoaded()) {
            throw new RuntimeException("Shard not initialized yet");
        }
        return storage.getFileMetrics().labelNames();
    }

    /* Old <=> New format converters */

    private static Labels convertToInternalLabels(
        MetabaseMetricId metabaseMetricId,
        String metricNameLabel)
    {
        LabelsBuilder builder = metabaseMetricId.getLabels().toBuilder();
        MetricNameHelper.convertToOldFormat(builder, metricNameLabel, metabaseMetricId.getName());
        return builder.build();
    }

    private static Selectors convertToInternalSelectors(
        Selectors labelSelectors,
        String metricNameLabel)
    {
        if (labelSelectors.hasKey(metricNameLabel)) {
            String message =
                "Unexpected metric name label \"" + metricNameLabel + "\" in label selectors: "
                    + labelSelectors;
            throw new SelectorsException(message);
        }

        if (labelSelectors.getNameSelector().isEmpty()) {
            return labelSelectors;
        }

        return labelSelectors.toBuilder()
            .add(Selector.glob(metricNameLabel, labelSelectors.getNameSelector()))
            .build();
    }

    private List<NamedCoremonMetric> convertToNamedCoremonMetrics(
        boolean useNewFormat,
        List<CoremonMetric> coremonMetrics)
    {
        // validateNewFormatReads is not required here, because it was validated above
        if (useNewFormat) {
            String metricNameLabel = getMetricNameLabel();

            return coremonMetrics.parallelStream()
                .map(s -> NamedCoremonMetric.from(s, metricNameLabel))
                .collect(Collectors.toList());
        }

        return coremonMetrics.parallelStream()
            .map(NamedCoremonMetric::from)
            .collect(Collectors.toList());
    }

    /* Other methods */

    private void validateNewFormatReads(boolean useNewFormat) {
        if (this.onlyNewFormatReadsRef.get() && !useNewFormat) {
            throw new RuntimeException("Shard " + id + " doesn't support old format read requests");
        }
    }

    private void validateNewFormatWrites(boolean useNewFormat) {
        if (this.onlyNewFormatWritesRef.get() && !useNewFormat) {
            throw new RuntimeException("Shard " + id + " doesn't support old format write requests");
        }
    }

    private void validateMetricNameSupport() {
        if (!isSupportsMetricNames()) {
            throw new RuntimeException("Shard " + id + " doesn't support metric names");
        }
    }

    @Override
    public MetabaseShardStorageImpl getStorage() {
        return storage;
    }

    @Override
    public void addMemoryInfo(MemoryBySubsystem r) {
        r.addMemory(MemOnlyMetricsCollectionImpl.class, storage.getMemOnlyMetrics().memorySizeIncludingSelf());

        {
            FileMetricsCollection fileMetrics = storage.getFileMetrics();
            long indexSize = fileMetrics.searchIndexSize();
            long indexCacheSize = fileMetrics.searchIndexCacheSize();
            String fileMetricsCollection = FileMetricsCollection.class.getSimpleName();
            r.addMemory(fileMetricsCollection + ".sensors", fileMetrics.memorySizeIncludingSelf() - indexSize);
            r.addMemory(fileMetricsCollection + ".index", indexSize);
            r.addMemory(fileMetricsCollection + ".indexCache", indexCacheSize);
        }

        {
            long writeQueueMemSize = storage.getWriteQueueMemSize();
            r.addMemory(MetabaseShardStorageImpl.class.getSimpleName() + ".writeQueue", writeQueueMemSize);
        }
    }

    @Override
    public void stop() {
        storage.stop();
    }

    @Override
    public MemOnlyMetricsCollectionImpl getMemOnlyMetrics() {
        return storage.getMemOnlyMetrics();
    }

    @Override
    public long getUptimeMillis() {
        return System.currentTimeMillis() - createdAtMillis;
    }

    public EShardState getState() {
        return storage.getState();
    }

    @Override
    public CompletableFuture<Void> getLoadFuture() {
        return storage.getLoadFuture();
    }

    @Override
    public LabelAllocator getLabelAllocator() {
        return labelAllocator;
    }

    @Override
    public MetabaseShardMetrics getMetrics() {
        return metrics;
    }
}
