package ru.yandex.solomon.coremon.meta.service;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.OptionalInt;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.core.ShardIsNotLocalException;
import ru.yandex.solomon.coremon.meta.db.MetricsDaoFactory;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.stockpile.client.StockpileClient;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class MetabaseShardResolverStub implements MetabaseShardResolver<MetabaseShardImpl>, AutoCloseable {
    private final Map<ShardKey, MetabaseShardImpl> shardByKey = new HashMap<>();
    private final Int2ObjectMap<MetabaseShardImpl> shardByNumId = new Int2ObjectOpenHashMap<>();
    private final AtomicInteger totalShardCount = new AtomicInteger(MetabaseTotalShardCounter.SHARD_COUNT_UNKNOWN);
    private final AtomicInteger inactiveShardCount = new AtomicInteger(0);

    private volatile boolean initCompleted = true;

    public MetabaseShardResolverStub(List<MetabaseShardConf> shardConfs, MetricsDaoFactory metricsDaoFactory, StockpileClient stockpileClient) {
        ExecutorService threadPool = Executors.newFixedThreadPool(4);

        for (MetabaseShardConf shardConf : shardConfs) {
            ShardKey key = shardConf.getShardKey();
            String shardId = key.toUniqueId();

            MetabaseShardImpl shard = new MetabaseShardImpl(
                shardConf.getNumId(),
                shardId,
                shardConf.getFolderId(),
                shardConf.getNumPartitions(),
                key,
                metricsDaoFactory,
                Labels.allocator,
                threadPool,
                stockpileClient);

            shard.setMetricNameLabel(shardConf.getMetricName());

            shardByKey.put(shard.getShardKey(), shard);
            shardByNumId.put(shard.getNumId(), shard);
        }
    }

    @Override
    public boolean isLoading() {
        return !initCompleted;
    }

    @Override
    public MetabaseShardImpl resolveShard(ShardKey shardKey) {
        checkInitializedSuccessfully();
        MetabaseShardImpl shard = shardByKey.get(shardKey);
        if (shard == null) {
            throw new ShardIsNotLocalException(shardKey.toString());
        }
        return shard;
    }

    @Nullable
    @Override
    public MetabaseShardImpl resolveShardOrNull(int numId) {
        checkInitializedSuccessfully();
        return shardByNumId.get(numId);
    }

    @Override
    public MetabaseShardImpl resolveShard(int numId) {
        checkInitializedSuccessfully();
        MetabaseShardImpl shard = shardByNumId.get(numId);
        if (shard == null) {
            throw new ShardIsNotLocalException(Integer.toUnsignedString(numId));
        }
        return shard;
    }

    @Override
    public Stream<MetabaseShardImpl> resolveShard(String folderId, Selectors selector) {
        checkInitializedSuccessfully();
        return getShards()
                .filter(shard -> folderId.isEmpty() || folderId.equals(shard.getFolderId()))
                .filter(shard -> selector.match(shard.getShardKey()));
    }

    @Override
    public long getShardKeysHash() {
        return shardByKey.keySet().hashCode();
    }

    @Override
    public Stream<MetabaseShardImpl> getShards() {
        return shardByKey.values().stream();
    }

    public void awaitShardsReady() {
        for (MetabaseShardImpl shard : shardByKey.values()) {
            shard.awaitReady();
        }
    }

    public void setInitCompleted(boolean initCompleted) {
        this.initCompleted = initCompleted;
    }

    public void setLastPointSeconds(int numId, Map<Labels, Integer> lastPointSecondsByLabels) {
        var shard = resolveShard(numId);

        try (var it = shard.getStorage().getFileMetrics().iterator()) {
            while (it.hasNext()) {
                try (var metric = it.next()) {
                    var lastPointSeconds = lastPointSecondsByLabels.get(metric.getLabels());
                    metric.setLastPointSeconds(lastPointSeconds);
                }
            }
        }
    }

    @Override
    public void close() {
        shardByNumId.values().forEach(MetabaseShard::stop);
    }

    private void checkInitializedSuccessfully() {
        if (isLoading()) {
            throw new MetabaseNotInitialized();
        }
    }

    @Override
    public void setTotalShardCount(int totalShardCount) {
        this.totalShardCount.set(totalShardCount);
    }

    @Override
    public OptionalInt getTotalShardCount() {
        int total = totalShardCount.get();
        if (total < 0) {
            return OptionalInt.empty();
        }
        return OptionalInt.of(total);
    }

    @Override
    public int getInactiveShardCount() {
        return inactiveShardCount.get();
    }
}
