package ru.yandex.solomon.coremon.meta.service.cloud;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import javax.annotation.Nullable;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.coremon.meta.service.MetabaseShard;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.name.resolver.client.Resource;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static ru.yandex.solomon.coremon.meta.service.handler.MetabaseShards.ensureSameProject;

/**
 * @author Vladimir Gordiychuk
 */
public class ReferenceMap {
    private final ReferenceResolver resolver;
    private final ResourceFinder finder;
    private final Map<String, ReferenceLabel> referenceByLabel;
    private final Map<String, ResourceMap> resourcesByLabel;

    public ReferenceMap(ReferenceResolver resolver, ResourceFinder finder) {
        this.resolver = resolver;
        this.finder = finder;
        this.referenceByLabel = new HashMap<>();
        this.resourcesByLabel = new HashMap<>();
    }

    public void addReferences(Collection<? extends MetabaseShard> shards) {
        for (var shard : shards) {
            addReferences(shard);
        }
    }

    public void addReferences(MetabaseShard shard) {
        addReferences(resolver.resolve(shard.getNumId()));
    }

    public void addResource(String label, ResourceMap resourceMap) {
        resourcesByLabel.put(label, resourceMap);
    }

    public void addReferences(List<ReferenceLabel> references) {
        for (var reference : references) {
            var prev = referenceByLabel.put(reference.label(), reference);
            if (prev != null) {
                referenceByLabel.put(reference.label(), ReferenceLabel.combine(prev, reference));
            }
        }
    }

    public void addResolvedReference(Map<String, ResourceMap> references) {
        resourcesByLabel.putAll(references);
    }

    public boolean hasUnresolvedReference() {
        return !referenceByLabel.keySet().equals(resourcesByLabel.keySet());
    }

    public Set<String> referenceLabels() {
        return referenceByLabel.keySet();
    }

    public Set<String> unresolvedReferenceLabel() {
        Set<String> result = new HashSet<>(referenceByLabel.size());
        for (var key : referenceByLabel.keySet()) {
            var resources = resourcesByLabel.get(key);
            if (resources == null || resources.isEmpty()) {
                result.add(key);
            }
        }
        return result;
    }

    public Map<String, Selectors> resourceSelectors(Selectors selectors) {
        return SelectorsTransform.toResourceSelectors(selectors, referenceByLabel);
    }

    public Selectors metricSelectors(Selectors selectors) {
        return SelectorsTransform.toMetricSelectors(selectors, resourcesByLabel);
    }

    public boolean isResolvedLabel(String label) {
        return !referenceByLabel.containsKey(label) || resourcesByLabel.containsKey(label);
    }

    public CompletableFuture<Void> resolveReferenceAffectedBySelector(Collection<? extends MetabaseShard> shards, Selectors selectors, long expiredAt) {
        if (!hasUnresolvedReference()) {
            return completedFuture(null);
        }

        var selectorsByLabel = resourceSelectors(selectors);
        if (selectorsByLabel.isEmpty()) {
            return completedFuture(null);
        }

        var projectId = ensureSameProject(shards);
        return finder.find(projectId, selectorsByLabel, expiredAt)
                .thenAccept(this::addResolvedReference);
    }

    public Map<String, ResourceMap> resolved() {
        return resourcesByLabel;
    }

    @Nullable
    public Resource getResourceById(String label, String resourceId) {
        var map = resourcesByLabel.get(label);
        if (map == null) {
            return null;
        }

        return map.getResourceById(resourceId);
    }

    @Nullable
    public Resource getResourceById(String resourceId) {
        for (var map : resourcesByLabel.values()) {
            var resource = map.getResourceById(resourceId);
            if (resource != null) {
                return resource;
            }
        }
        return null;
    }

    public boolean isReplaced(Labels labels) {
        for (int index = 0; index < labels.size(); index++) {
            var label = labels.at(index);
            var resource = getResourceById(label.getKey(), label.getValue());
            if (resource == null) {
                continue;
            }

            if (resource.replaced) {
                return true;
            }
        }
        return false;
    }

    public Labels replaceReference(Labels labels) {
        var builder = labels.toBuilder();
        var it = labels.stream().iterator();
        while (it.hasNext()) {
            var label = it.next();
            var resourceId = getResourceById(label.getKey(), label.getValue());
            if (resourceId == null) {
                continue;
            }
            builder.add(label.getKey(), resourceId.getNameOrId());
        }
        return builder.build();
    }
}
