package ru.yandex.solomon.coremon.meta.service.cloud;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.stream.Collectors;

import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.SelectorType;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.name.resolver.client.Resource;
import ru.yandex.solomon.name.resolver.client.ResourceLabels;

/**
 * @author Vladimir Gordiychuk
 */
public class SelectorsTransform {

    public static Map<String, Selectors> toResourceSelectors(Selectors selectors, Map<String, ReferenceLabel> referenceByLabel) {
        if (referenceByLabel.isEmpty() || selectors.isEmpty()) {
            return Map.of();
        }

        var selectorByLabel = new HashMap<String, Selectors>(referenceByLabel.size());
        for (var reference : referenceByLabel.values()) {
            var selector = selectors.findByKey(reference.label());
            if (selector == null || selector.getType() == SelectorType.ANY || selector.getType() == SelectorType.ABSENT) {
                continue;
            }

            var resourceSelectors = SelectorsTransform.toResourceSelectors(selectors, reference);
            selectorByLabel.put(reference.label(), resourceSelectors);
        }

        return selectorByLabel;
    }

    public static Selectors toResourceSelectors(Selectors selectors, ReferenceLabel reference) {
        var result = Selectors.builder(1);
        for (var selector : selectors) {
            if (reference.label().equals(selector.getKey())) {
                result.add(Selector.of(ResourceLabels.RESOURCE, selector.getValue(), selector.getType()));
            }
        }

        if (!reference.folders().isEmpty()) {
            result.add(Selector.glob(ResourceLabels.FOLDER_ID, String.join("|", reference.folders())));
        }

        if (!reference.services().isEmpty()) {
            result.add(Selector.glob(ResourceLabels.SERVICE, String.join("|", reference.services())));
        }

        if (!reference.types().isEmpty()) {
            result.add(Selector.glob(ResourceLabels.TYPE, String.join("|", reference.types())));
        }

        return result.build();
    }

    public static Selector toMetricSelector(Selector selector, Collection<Resource> resources) {
        switch (selector.getType()) {
            case ANY:
            case ABSENT:
                return selector;
            case GLOB:
            case EXACT:
                var resourceIds = resources.stream()
                        .filter(resource -> !resource.replaced)
                        .map(resource -> resource.resourceId)
                        .collect(Collectors.joining("|"));

                if (resourceIds.isEmpty() || resourceIds.equals(selector.getValue())) {
                    return selector;
                }

                return Selector.glob(selector.getKey(), selector.getValue() + "|" + resourceIds);
            default:
                var value = resources.stream()
                        .filter(resource -> !resource.replaced)
                        .map(resource -> resource.resourceId)
                        .collect(Collectors.joining("|"));

                if (value.isEmpty()) {
                    return Selector.absent(selector.getKey());
                }

                return Selector.glob(selector.getKey(), value);
        }
    }

    public static Selectors toMetricSelectors(Selectors selectors, Map<String, ResourceMap> resourcesByLabel) {
        if (resourcesByLabel.isEmpty() || selectors.isEmpty()) {
            return selectors;
        }

        var result = Selectors.builder(selectors.size())
                .setNameSelector(selectors.getNameSelector());
        for (var selector : selectors) {
            if (!resourcesByLabel.containsKey(selector.getKey())) {
                result.add(selector);
                continue;
            }

            var resources = resourcesByLabel.get(selector.getKey());
            var updated = toMetricSelector(selector, resources);
            result.add(updated);
        }

        return result.build();
    }
}
