package ru.yandex.solomon.coremon.meta.service.handler;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.coremon.meta.MetabaseMetricId;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardImpl;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardResolver;
import ru.yandex.solomon.coremon.meta.service.ShardKeyAndMetricKey;
import ru.yandex.solomon.labels.protobuf.LabelConverter;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.metabase.api.protobuf.DeleteManyRequest;
import ru.yandex.solomon.metabase.api.protobuf.DeleteManyResponse;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;

import static ru.yandex.solomon.coremon.meta.service.handler.MetabaseShards.ensureReadyToWrite;

/**
 * @author Vladimir Gordiychuk
 */
public class DeleteManyHandler {
    private final MetabaseShardResolver<MetabaseShardImpl> shardResolver;

    public DeleteManyHandler(MetabaseShardResolver<MetabaseShardImpl> shardResolver) {
        this.shardResolver = shardResolver;
    }

    public CompletableFuture<DeleteManyResponse> deleteMany(DeleteManyRequest request) {
        Map<ShardKey, List<MetabaseMetricId>> grouped = request.getMetricsList().stream()
                .map(s -> Pair.of(s.getName(), ShardKeyAndMetricKey.of(LabelConverter.protoToLabels(s.getLabelsList()))))
                .collect(
                        Collectors.groupingBy(
                                pair -> pair.getRight().getShardKey(),
                                Collectors.mapping(pair -> new MetabaseMetricId(pair.getLeft(), pair.getRight().getMetricKey()), Collectors.toList())
                        )
                );

        for (var key : grouped.keySet()) {
            var shard = shardResolver.resolveShard(key);
            ensureReadyToWrite(shard);
        }

        boolean useNewFormat = request.getMetricsList().stream().noneMatch(s -> s.getName().isEmpty());
        return CompletableFutures.allOf(grouped.entrySet()
                .stream()
                .map(entry -> {
                    ShardKey shardKey = entry.getKey();
                    List<MetabaseMetricId> metabaseMetricIds = entry.getValue();

                    MetabaseShardImpl shard = shardResolver.resolveShard(shardKey);
                    Labels shardLabels = shard.getShardKey().toLabels();

                    return shard.deleteMetrics(metabaseMetricIds, useNewFormat)
                            .thenApply(deletedMetrics -> deletedMetrics.stream()
                                    .map(metric -> Proto.toProto(metric, shardLabels))
                                    .collect(Collectors.toList()));
                })
                .collect(Collectors.toList()))
                .thenApply(results -> DeleteManyResponse.newBuilder()
                        .setStatus(EMetabaseStatusCode.OK)
                        .addAllMetrics(results.stream()
                                .flatMap(Collection::stream)
                                .collect(Collectors.toList()))
                        .build());
    }
}
