package ru.yandex.solomon.coremon.meta.service.handler;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;

import com.google.protobuf.TextFormat;
import io.grpc.Context;
import io.grpc.Status;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.thread.WhatThreadDoes;
import ru.yandex.solomon.coremon.meta.NamedCoremonMetric;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardImpl;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardResolver;
import ru.yandex.solomon.coremon.meta.service.cloud.ReferenceMap;
import ru.yandex.solomon.coremon.meta.service.cloud.ReferenceResolver;
import ru.yandex.solomon.coremon.meta.service.cloud.ResourceFinder;
import ru.yandex.solomon.labels.protobuf.LabelConverter;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metabase.api.protobuf.FindRequest;
import ru.yandex.solomon.metabase.api.protobuf.FindResponse;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static ru.yandex.solomon.coremon.meta.service.handler.MetabaseShards.ensureSameProject;
import static ru.yandex.solomon.coremon.meta.service.handler.MetabaseShards.resolveReadyToReadShards;

/**
 * step 1: find resources affected by selector
 * step 2: patch selector by resource names
 * step 3: find metrics by selector
 * step 4: find not resolved resources by metrics
 * step 5: filter replaced resources
 * step 6: apply limits
 * step 7: prepare response
 *
 * @author Vladimir Gordiychuk
 */
public class FindHandler {
    private static final Logger logger = LoggerFactory.getLogger(FindHandler.class);
    private final MetabaseShardResolver<MetabaseShardImpl> shardResolver;
    private final ResourceFinder resourceFinder;

    private String wtd;
    private Context context;
    private ParsedFindRequest req;
    private Collection<MetabaseShardImpl> shards;
    private final ReferenceMap referenceMap;

    public FindHandler(MetabaseShardResolver<MetabaseShardImpl> shardResolver, ReferenceResolver referenceResolver, ResourceFinder resourceFinder) {
        this.shardResolver = shardResolver;
        this.resourceFinder = resourceFinder;
        this.referenceMap = new ReferenceMap(referenceResolver, resourceFinder);
    }

    public CompletableFuture<FindResponse> find(FindRequest request) {
        wtd = "Metabase#find " + TextFormat.shortDebugString(request);
        logger.debug(wtd);
        context = Context.current();
        return callInContext(() -> {
            req = new ParsedFindRequest(request);
            shards = resolveReadyToReadShards(shardResolver, request.getShardId(), req.folderId, req.shardSelector);
            if (shards.isEmpty()) {
                return completedFuture(toResponse(List.of()));
            }
            referenceMap.addReferences(shards);
            return findResourcesAffectedBySelector()
                    .thenCompose(unused -> {
                        var selectors = patchSelectorByResourceName();
                        if (!referenceMap.hasUnresolvedReference()) {
                            return completedFuture(findMetricsBySelector(selectors, req.offset, req.limit));
                        }

                        var metricsByShard = findMetricsBySelector(selectors, 0, Integer.MAX_VALUE);
                        return findNotResolvedResourceByMetrics(metricsByShard)
                                .thenApply(ignore -> filterReplacedAndLimit(metricsByShard, req.offset, req.limit));
                    })
                    .thenApply(this::toResponse);
        });
    }

    private <T> T callInContext(Supplier<T> supplier) {
        var h = WhatThreadDoes.push(wtd);
        var previous = context.attach();
        try {
            return supplier.get();
        } finally {
            h.popSafely();
            context.detach(previous);
        }
    }

    private CompletableFuture<Void> findResourcesAffectedBySelector() {
        return referenceMap.resolveReferenceAffectedBySelector(shards, req.metricsSelector, req.expiredAt)
                .thenRun(this::ensureNotExpired);
    }

    private Selectors patchSelectorByResourceName() {
        return referenceMap.metricSelectors(req.metricsSelector);
    }

    private List<FindResult> findMetricsBySelector(Selectors selectors, int offset, int limit) {
        return callInContext(() -> {
            if (logger.isDebugEnabled()) {
                logger.debug("{} metrics selector {}", wtd, Selectors.format(selectors));
            }
            int remaining = limit;
            List<FindResult> results = new ArrayList<>(shards.size());
            for (var shard : shards) {
                ensureNotExpired();
                var result = findMetricsByShard(shard, selectors, offset, remaining, req.useNewFormat);
                if (result.total == 0) {
                    continue;
                }

                remaining -= result.total;
                results.add(result);
            }
            return results;
        });
    }

    private CompletableFuture<Void> findNotResolvedResourceByMetrics(List<FindResult> metricsByShard) {
        return callInContext(() -> {
            Set<String> unresolvedResourceIds = new HashSet<>();
            Set<String> unresolvedLabels = Set.copyOf(referenceMap.unresolvedReferenceLabel());
            for (var result : metricsByShard) {
                for (var metric : result.metrics) {
                    for (var unresolvedLabel : unresolvedLabels) {
                        var label = metric.getLabels().findByKey(unresolvedLabel);
                        if (label == null) {
                            continue;
                        }

                        unresolvedResourceIds.add(label.getValue());
                    }
                }
            }

            if (unresolvedResourceIds.isEmpty()) {
                return completedFuture(null);
            }

            String projectId = ensureSameProject(shards);
            return resourceFinder.resolve(projectId, unresolvedResourceIds, req.expiredAt)
                    .thenAccept(map -> {
                        for (var label : unresolvedLabels) {
                            referenceMap.addResource(label, map);
                        }
                    });
        });
    }

    private List<FindResult> filterReplacedAndLimit(List<FindResult> metricsByShard, int offset, int limit) {
        List<FindResult> filtered = new ArrayList<>(metricsByShard.size());
        int remaining = limit;
        for (var shard : metricsByShard) {
            if (remaining == 0) {
                filtered.add(new FindResult(shard.shardKey, shard.total, List.of()));
                continue;
            }

            int total = shard.total;
            List<NamedCoremonMetric> metrics = new ArrayList<>(shard.metrics.size());
            for (var metric : shard.metrics) {
                if (referenceMap.isReplaced(metric.getLabels())) {
                    total--;
                    continue;
                }

                if (offset > 0) {
                    offset--;
                    continue;
                }

                if (remaining-- == 0) {
                    break;
                }

                metrics.add(metric);
            }

            filtered.add(new FindResult(shard.shardKey, total, metrics));
        }
        return filtered;
    }

    private FindResult findMetricsByShard(MetabaseShardImpl shard, Selectors selectors, int offset, int limit, boolean useNewFormat) {
        if (limit > 0) {
            List<NamedCoremonMetric> metrics = new ArrayList<>(100);
            int total = shard.searchMetrics(selectors, offset, limit, useNewFormat, metrics::add);
            logger.debug("{} at shard {} by selector {}, found {}", wtd, shard.getId(), selectors, total);
            return new FindResult(shard.getShardKey(), total, metrics);
        }

        int total = shard.searchCount(selectors, useNewFormat);
        logger.debug("{} at shard {} by selector, {} found {} only count", wtd, shard.getId(), selectors, total);
        return new FindResult(shard.getShardKey(), total, List.of());
    }

    private FindResponse toResponse(List<FindResult> results) {
        return callInContext(() -> {
            var response = FindResponse.newBuilder().setStatus(EMetabaseStatusCode.OK);
            int total = 0;
            for (FindResult result : results) {
                total += result.total;
                var shardLabels = LabelConverter.labelsToProtoList(result.shardKey.toLabels());
                for (var metric : result.metrics) {
                    response.addMetrics(Proto.toProto(metric, shardLabels, referenceMap.resolved()));
                }
            }
            return response.setTotalCount(total).build();
        });
    }

    private void ensureNotExpired() {
        if (req.expiredAt != 0 && System.currentTimeMillis() + 100 >= req.expiredAt) {
            throw Status.DEADLINE_EXCEEDED.asRuntimeException();
        }
    }

    private static record FindResult(ShardKey shardKey, int total, List<NamedCoremonMetric> metrics) {
    }
}
