package ru.yandex.solomon.coremon.meta.service.handler;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import com.google.protobuf.TextFormat;
import io.grpc.Context;
import io.grpc.Status;

import ru.yandex.misc.thread.WhatThreadDoes;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardImpl;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardResolver;
import ru.yandex.solomon.coremon.meta.service.cloud.ReferenceMap;
import ru.yandex.solomon.coremon.meta.service.cloud.ReferenceResolver;
import ru.yandex.solomon.coremon.meta.service.cloud.ResourceFinder;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.protobuf.LabelSelectorConverter;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.query.ShardSelectors;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metabase.api.protobuf.TLabelNamesRequest;
import ru.yandex.solomon.metabase.api.protobuf.TLabelNamesResponse;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static ru.yandex.solomon.coremon.meta.service.handler.MetabaseShards.resolveReadyToReadShards;

/**
 * @author Vladimir Gordiychuk
 */
public class LabelNamesHandler {
    private final MetabaseShardResolver<MetabaseShardImpl> shardResolver;

    private String wtd;
    private Context context;
    private boolean useNewFormat;
    private int shardId;
    private long expiredAt;
    private String folderId;
    private Selectors shardSelector;
    private Selectors metricSelector;
    private Collection<MetabaseShardImpl> shards;
    private final ReferenceMap referenceMap;

    public LabelNamesHandler(MetabaseShardResolver<MetabaseShardImpl> shardResolver, ReferenceResolver referenceResolver, ResourceFinder resourceFinder) {
        this.shardResolver = shardResolver;
        this.referenceMap = new ReferenceMap(referenceResolver, resourceFinder);
    }

    public CompletableFuture<TLabelNamesResponse> labelNames(TLabelNamesRequest request) {
        wtd = "Metabase#labelNames " + TextFormat.shortDebugString(request);
        context = Context.current();
        return callInContext(() -> {
            parseRequest(request);
            resolveShards();
            if (shards.isEmpty()) {
                return completedFuture(toResponse(List.of()));
            }

            referenceMap.addReferences(shards);
            return findResourcesBySelectorReference()
                    .thenApply(ignore -> toResponse(labelNames()));
        });
    }

    private <T> T callInContext(Supplier<T> supplier) {
        var h = WhatThreadDoes.push(wtd);
        var previous = context.attach();
        try {
            return supplier.get();
        } finally {
            h.popSafely();
            context.detach(previous);
        }
    }

    private void resolveShards() {
        shards = resolveReadyToReadShards(shardResolver, shardId, folderId, shardSelector);
    }

    private CompletableFuture<Void> findResourcesBySelectorReference() {
        return referenceMap.resolveReferenceAffectedBySelector(shards, metricSelector, expiredAt)
                .thenRun(this::ensureNotExpired);
    }

    private List<String> labelNames() {
        return callInContext(() -> {
            var selectors = referenceMap.metricSelectors(metricSelector);
            var names = new HashSet<String>();
            for (MetabaseShardImpl shard : shards) {
                ensureNotExpired();
                if (metricSelector.isEmpty()) {
                    names.addAll(shard.labelNames(useNewFormat));
                } else {
                    shard.searchLabels(selectors, 0, 0, useNewFormat, labels -> {
                        labels.forEach(label -> names.add(label.getKey()));
                    });
                }
            }

            List<String> filteredNames = names.stream()
                    .filter(name -> !selectors.hasKey(name))
                    .collect(Collectors.toList());

            if (shardId == 0) {
                // metrics inside shard don't contain project-cluster-service labels
                appendShardKeyIfNeed(filteredNames, LabelKeys.PROJECT, shardSelector);
                appendShardKeyIfNeed(filteredNames, LabelKeys.CLUSTER, shardSelector);
                appendShardKeyIfNeed(filteredNames, LabelKeys.SERVICE, shardSelector);
            }

            return filteredNames;
        });
    }

    private void parseRequest(TLabelNamesRequest request) {
        useNewFormat = request.hasNewSelectors();
        folderId = request.getFolderId();

        Selectors selector = useNewFormat
                ? LabelSelectorConverter.protoToSelectors(request.getNewSelectors())
                : LabelSelectorConverter.protoToSelectors(request.getSelectorsList());

        shardId = request.getShardId();
        shardSelector = ShardSelectors.onlyShardKey(selector);
        metricSelector = ShardSelectors.withoutShardKey(selector);
        expiredAt = request.getDeadlineMillis();
    }

    private TLabelNamesResponse toResponse(List<String> labelNames) {
        return TLabelNamesResponse.newBuilder()
                .setStatus(EMetabaseStatusCode.OK)
                .addAllNames(labelNames)
                .build();
    }

    private void ensureNotExpired() {
        if (expiredAt != 0 && System.currentTimeMillis() + 100 >= expiredAt) {
            throw Status.DEADLINE_EXCEEDED.asRuntimeException();
        }
    }

    private static void appendShardKeyIfNeed(
            List<String> names,
            String labelName,
            Selectors shardSelector)
    {
        if (!shardSelector.hasKey(labelName)) {
            names.add(labelName);
        }
    }
}
