package ru.yandex.solomon.coremon.meta.service.handler;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Predicate;
import java.util.function.Supplier;

import com.google.protobuf.TextFormat;
import io.grpc.Context;
import io.grpc.Status;

import ru.yandex.misc.thread.WhatThreadDoes;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardImpl;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardResolver;
import ru.yandex.solomon.coremon.meta.service.cloud.ReferenceMap;
import ru.yandex.solomon.coremon.meta.service.cloud.ReferenceResolver;
import ru.yandex.solomon.coremon.meta.service.cloud.ResourceFinder;
import ru.yandex.solomon.coremon.meta.service.cloud.ResourceMap;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.metabase.api.protobuf.TLabelValuesRequest;
import ru.yandex.solomon.metabase.api.protobuf.TLabelValuesResponse;
import ru.yandex.solomon.util.labelStats.LabelStats;
import ru.yandex.solomon.util.labelStats.LabelStatsConverter;
import ru.yandex.solomon.util.labelStats.LabelValuesStats;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static ru.yandex.solomon.coremon.meta.service.handler.MetabaseShards.ensureSameProject;
import static ru.yandex.solomon.coremon.meta.service.handler.MetabaseShards.resolveReadyToReadShards;

/**
 * @author Vladimir Gordiychuk
 */
public class LabelValuesHandler {
    private final MetabaseShardResolver<MetabaseShardImpl> shardResolver;
    private final ResourceFinder resourceFinder;

    private String wtd;
    private Context context;
    private ParsedLabelValuesRequest req;
    private Collection<MetabaseShardImpl> shards;
    private final ReferenceMap referenceMap;
    private LabelValuesStats result;

    public LabelValuesHandler(MetabaseShardResolver<MetabaseShardImpl> shardResolver, ReferenceResolver referenceResolver, ResourceFinder resourceFinder) {
        this.shardResolver = shardResolver;
        this.resourceFinder = resourceFinder;
        this.referenceMap = new ReferenceMap(referenceResolver, resourceFinder);
    }

    public CompletableFuture<TLabelValuesResponse> labelValues(TLabelValuesRequest request) {
        wtd = "Metabase#labelValues " + TextFormat.shortDebugString(request);
        context = Context.current();
        return callInContext(() -> {
            req = new ParsedLabelValuesRequest(request);
            resolveShards();
            if (shards.isEmpty()) {
                return completedFuture(toResponse(new LabelValuesStats()));
            }

            referenceMap.addReferences(shards);
            return findResourcesBySelectorReference()
                    .thenAccept(ignore -> result = labelValues())
                    .thenCompose(ignore -> findResourcesByOtherReference())
                    .thenApply(ignore -> toResponse(result));
        });
    }

    private <T> T callInContext(Supplier<T> supplier) {
        var h = WhatThreadDoes.push(wtd);
        var previous = context.attach();
        try {
            return supplier.get();
        } finally {
            h.popSafely();
            context.detach(previous);
        }
    }

    private void resolveShards() {
        shards = resolveReadyToReadShards(shardResolver, req.shardId, req.getFolderId(), req.getShardSelector());
    }

    private CompletableFuture<Void> findResourcesBySelectorReference() {
        return referenceMap.resolveReferenceAffectedBySelector(shards, req.getMetricSelector(), req.expiredAt);
    }

    private LabelValuesStats labelValues() {
        return callInContext(() -> {
            LabelValuesStats stats = new LabelValuesStats();
            for (MetabaseShardImpl shard : shards) {
                ensureNotExpired();
                stats.combine(labelValuesByShard(shard));
            }
            // Stats already limited and filtered, but as a result of combine cross shard request we can overflow limit
            stats.limit(req.getLimit(), this::isResolvedLabel);
            return stats;
        });
    }

    private CompletableFuture<Void> findResourcesByOtherReference() {
        if (!referenceMap.hasUnresolvedReference()) {
            return completedFuture(null);
        }

        return callInContext(() -> {
            Set<String> unresolvedLabel = new HashSet<>();
            Set<String> unresolvedResourceIds = new HashSet<>();
            for (var entry : result.getStatsByLabelKey().entrySet()) {
                var label = entry.getKey();
                var labelStats = entry.getValue();
                if (!isResolvedLabel(label)) {
                    unresolvedLabel.add(label);
                    unresolvedResourceIds.addAll(labelStats.getValues());
                }
            }
            return resolveIds(unresolvedResourceIds)
                    .thenAccept(map -> {
                        for (var label : unresolvedLabel) {
                            replaceReference(result, label, map);
                        }
                        result.filter(req.getTextSearch(), unresolvedLabel::contains);
                        result.filter(req.getValidationFilter(), unresolvedLabel::contains);
                        result.limit(req.getLimit(), unresolvedLabel::contains);
                    });
        });
    }

    private CompletableFuture<ResourceMap> resolveIds(Collection<String> unresolvedIds) {
        if (unresolvedIds.isEmpty()) {
            return completedFuture(new ResourceMap(List.of()));
        }

        String projectId = ensureSameProject(shards);
        return resourceFinder.resolve(projectId, unresolvedIds, req.expiredAt);
    }

    private LabelValuesStats labelValuesByShard(MetabaseShardImpl shard) {
        var selectors = referenceMap.metricSelectors(req.getMetricSelector());
        LabelValuesStats result = labelValuesWithoutShard(shard, selectors);
        if (result.getMetricsCount() > 0) {
            final ShardKey shardKey = shard.getShardKey();
            shardKey.forEachLabel((key, value) -> {
                if (req.getShardLabels().contains(key)) {
                    result.add(key, LabelStats.single(value, result.getMetricsCount()));
                }
            });
        }

        replaceReference(result, referenceMap);
        result.filter(req.getTextSearch(), this::isResolvedLabel);
        result.filter(req.getValidationFilter(), this::isResolvedLabel);
        result.limit(req.getLimit(), this::isResolvedLabel);
        return result;
    }

    private boolean isResolvedLabel(String label) {
        return referenceMap.isResolvedLabel(label);
    }

    private LabelValuesStats labelValuesWithoutShard(MetabaseShardImpl shard, Selectors selectors) {
        if (req.showOnlyClusterServiceLabels()) {
            int metricsCount;
            if (selectors.isEmpty()) {
                metricsCount = shard.fileAndMemOnlyMetrics();
            } else {
                metricsCount = shard.searchCount(selectors, req.isUseNewFormat());
            }
            return new LabelValuesStats(new HashMap<>(), metricsCount);
        }

        if (selectors.isEmpty()) {
            return shard.labelStats(req.getMetricLabels(), req.isUseNewFormat());
        }

        LabelValuesStats stats = new LabelValuesStats();
        Set<String> requestedNames = req.getMetricLabels();
        final Predicate<String> labelPredicate;
        if (requestedNames.isEmpty()) {
            labelPredicate = label -> true;
        } else {
            labelPredicate = requestedNames::contains;
        }

        shard.searchLabels(selectors, 0, 0, req.isUseNewFormat(), labels -> {
            stats.add(labels, labelPredicate);
        });
        return stats;
    }

    private void replaceReference(LabelValuesStats stats, ReferenceMap referenceMap) {
        for (var entry : referenceMap.resolved().entrySet()) {
            replaceReference(stats, entry.getKey(), entry.getValue());
        }
    }

    private void replaceReference(LabelValuesStats stats, String label, ResourceMap resources) {
        var labelStats = stats.getStatsByLabelKey().get(label);
        if (labelStats == null) {
            return;
        }

        stats.replace(label, replaceReference(labelStats, resources));
    }

    private LabelStats replaceReference(LabelStats stats, ResourceMap resources) {
        Set<String> result = new HashSet<>(stats.getValues().size());
        int sameNameCount = 0;
        int replacedCount = 0;
        for (var resourceId : stats.getValues()) {
            var resource = resources.getResourceById(resourceId);
            if (resource == null || !resource.hasName()) {
                result.add(resourceId);
            } else if (resource.replaced) {
                replacedCount++;
            } else if (!result.add(resource.getNameOrId())) {
                sameNameCount++;
            }
        }
        int count = stats.getCount() - sameNameCount - replacedCount;
        return new LabelStats(result, count, stats.isTruncated());
    }

    private TLabelValuesResponse toResponse(LabelValuesStats stats) {
        return LabelStatsConverter.toProto(stats);
    }

    private void ensureNotExpired() {
        if (req.expiredAt != 0 && System.currentTimeMillis() + 100 >= req.expiredAt) {
            throw Status.DEADLINE_EXCEEDED.asRuntimeException();
        }
    }

}
