package ru.yandex.solomon.coremon.meta.service.handler;

import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import io.grpc.Status;

import ru.yandex.solomon.coremon.meta.service.MetabaseShard;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardImpl;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardResolver;
import ru.yandex.solomon.coremon.meta.service.ShardNotReadyException;
import ru.yandex.solomon.coremon.meta.service.ShardReadOnlyException;
import ru.yandex.solomon.coremon.meta.service.ShardWriteOnlyException;
import ru.yandex.solomon.labels.query.Selectors;

/**
 * @author Vladimir Gordiychuk
 */
public class MetabaseShards {
    /**
     * Resolves shard by numId or selectors. NumId takes precedence over selectors.
     */
    public static Collection<MetabaseShardImpl> resolveReadyToReadShards(MetabaseShardResolver<MetabaseShardImpl> resolver, int numId, String folderId, Selectors selectors) {
        if (numId != 0) {
            // (1) single shard request
            // resolveShard() will throw ShardIsNotLocalException if it fails to find shard by id
            var shard = resolver.resolveShard(numId);
            if (!shard.isReadyRead()) {
                return List.of();
            }

            ensureLoaded(shard);
            return List.of(shard);
        }

        // (2) multi shard request
        // resolveShard() will returns an empty stream if it fails to find shards by selector
        List<MetabaseShardImpl> shards = resolver.resolveShard(folderId, selectors)
                .filter(MetabaseShardImpl::isReadyRead)
                .collect(Collectors.toList());

        for (MetabaseShardImpl shard : shards) {
            ensureLoaded(shard);
        }
        return shards;
    }

    public static String ensureSameProject(Collection<? extends MetabaseShard> shards) {
        var it = shards.iterator();
        var expect = it.next().getShardKey().getProject();
        while (it.hasNext()) {
            var next = it.next();
            var project = next.getShardKey().getProject();
            if (!expect.equals(project)) {
                throw Status.INVALID_ARGUMENT
                        .withDescription("Cross project request unavailable")
                        .asRuntimeException();
            }
        }
        return expect;
    }

    public static void ensureReadyToRead(MetabaseShardImpl shard) {
        ensureLoaded(shard);
        if (!shard.isReadyRead()) {
            throw new ShardWriteOnlyException(shard.getId());
        }
    }

    public static void ensureLoaded(MetabaseShardImpl shard) {
        if (!shard.isLoaded()) {
            throw new ShardNotReadyException(shard.getId());
        }
    }

    public static void ensureReadyToWrite(MetabaseShardImpl shard) {
        ensureLoaded(shard);
        if (!shard.isReadyWrite()) {
            throw new ShardReadOnlyException(shard.getId());
        }
    }
}
