package ru.yandex.solomon.coremon.meta.service.handler;

import java.util.Collection;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;

import com.google.protobuf.TextFormat;
import io.grpc.Context;
import io.grpc.Status;

import ru.yandex.misc.thread.WhatThreadDoes;
import ru.yandex.monlib.metrics.labels.validate.LabelValidationFilter;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardImpl;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardResolver;
import ru.yandex.solomon.coremon.meta.service.cloud.EmptyReferenceResolver;
import ru.yandex.solomon.coremon.meta.service.cloud.EmptyResourceFinder;
import ru.yandex.solomon.coremon.meta.service.cloud.ReferenceMap;
import ru.yandex.solomon.coremon.meta.service.cloud.ReferenceResolver;
import ru.yandex.solomon.coremon.meta.service.cloud.ResourceFinder;
import ru.yandex.solomon.labels.protobuf.LabelSelectorConverter;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.query.ShardSelectors;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metabase.api.protobuf.MetricNamesRequest;
import ru.yandex.solomon.metabase.api.protobuf.MetricNamesResponse;
import ru.yandex.solomon.util.labelStats.LabelStats;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static ru.yandex.solomon.coremon.meta.service.handler.MetabaseShards.resolveReadyToReadShards;
import static ru.yandex.solomon.labels.protobuf.LabelValidationFilterConverter.protoToFilter;

/**
 * @author Vladimir Gordiychuk
 */
public class MetricNamesHandler {
    private final MetabaseShardResolver<MetabaseShardImpl> shardResolver;

    private String wtd;
    private Context context;

    private int shardId;
    private String folderId;
    private Selectors shardSelector;
    private Selectors metricSelector;
    private String textSearch;
    private LabelValidationFilter validationFilter;
    private int limit;
    private long expiredAt;

    private Collection<MetabaseShardImpl> shards;
    private final ReferenceMap referenceMap;

    public MetricNamesHandler(MetabaseShardResolver<MetabaseShardImpl> shardResolver) {
        this(shardResolver, new EmptyReferenceResolver(), new EmptyResourceFinder());
    }

    public MetricNamesHandler(MetabaseShardResolver<MetabaseShardImpl> shardResolver, ReferenceResolver referenceResolver, ResourceFinder resourceFinder) {
        this.shardResolver = shardResolver;
        this.referenceMap = new ReferenceMap(referenceResolver, resourceFinder);
    }

    public CompletableFuture<MetricNamesResponse> metricNames(MetricNamesRequest request) {
        wtd = "Metabase#metricNames " + TextFormat.shortDebugString(request);
        context = Context.current();
        return callInContext(() -> {
            ensureNotExpired();
            parseRequest(request);
            resolveShards();
            if (shards.isEmpty()) {
                return completedFuture(toResponse(LabelStats.create()));
            }
            referenceMap.addReferences(shards);
            return findResourcesBySelectorReference()
                    .thenApply(ignore -> toResponse(metricNames()));
        });
    }

    private <T> T callInContext(Supplier<T> supplier) {
        var h = WhatThreadDoes.push(wtd);
        var previous = context.attach();
        try {
            return supplier.get();
        } finally {
            h.popSafely();
            context.detach(previous);
        }
    }

    private void parseRequest(MetricNamesRequest request) {
        Selectors selector = LabelSelectorConverter.protoToSelectors(request.getSelectors());

        shardId = request.getShardId();
        folderId = request.getFolderId();
        shardSelector = ShardSelectors.onlyShardKey(selector);
        metricSelector = ShardSelectors.withoutShardKey(selector);

        textSearch = request.getTextSearch();
        validationFilter = protoToFilter(request.getValidationFilter());
        limit = request.getLimit();
        expiredAt = request.getDeadlineMillis();
    }

    private void resolveShards() {
        shards = resolveReadyToReadShards(shardResolver, shardId, folderId, shardSelector);
    }

    private CompletableFuture<Void> findResourcesBySelectorReference() {
        return referenceMap.resolveReferenceAffectedBySelector(shards, metricSelector, expiredAt)
                .thenRun(this::ensureNotExpired);
    }

    private LabelStats metricNames() {
        return callInContext(() -> {
            var selectors = referenceMap.metricSelectors(metricSelector);
            final LabelStats result = LabelStats.create();
            for (MetabaseShardImpl shard : shards) {
                ensureNotExpired();
                result.combine(shard.searchMetricNames(selectors, textSearch, validationFilter));
            }

            result.limit(limit);
            return result;
        });
    }

    private MetricNamesResponse toResponse(LabelStats stats) {
        return MetricNamesResponse.newBuilder()
                .setStatus(EMetabaseStatusCode.OK)
                .addAllNames(stats.getValues())
                .setTruncated(stats.isTruncated())
                .build();
    }

    private void ensureNotExpired() {
        if (expiredAt != 0 && System.currentTimeMillis() + 100 >= expiredAt) {
            throw Status.DEADLINE_EXCEEDED.asRuntimeException();
        }
    }
}
