package ru.yandex.solomon.coremon.meta.service.handler;

import java.util.Collections;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.base.Strings;
import com.google.common.collect.ImmutableSet;

import ru.yandex.monlib.metrics.labels.validate.LabelValidationFilter;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.protobuf.LabelSelectorConverter;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.query.ShardSelectors;
import ru.yandex.solomon.metabase.api.protobuf.TLabelValuesRequest;

import static ru.yandex.solomon.labels.protobuf.LabelValidationFilterConverter.protoToFilter;

/**
 * @author Vladimir Gordiychuk
 */
class ParsedLabelValuesRequest {
    public final int shardId;
    public final long expiredAt;
    private final String folderId;
    private final Selectors shardSelector;
    private final Selectors metricSelector;
    private final boolean includeAllMetricLabels;
    private final Set<String> shardLabels;
    private final Set<String> metricLabels;
    private final int limit;
    private final String textSearch;
    private final LabelValidationFilter validationFilter;
    private final boolean useNewFormat;

    ParsedLabelValuesRequest(TLabelValuesRequest request) {
        this.shardId = request.getShardId();
        this.expiredAt = request.getDeadlineMillis();
        this.useNewFormat = request.hasNewSelectors();
        this.folderId = request.getFolderId();

        Selectors selector = useNewFormat
            ? LabelSelectorConverter.protoToSelectors(request.getNewSelectors())
            : LabelSelectorConverter.protoToSelectors(request.getSelectorsList());

        this.shardSelector = ShardSelectors.onlyShardKey(selector);
        this.metricSelector = ShardSelectors.withoutShardKey(selector);
        this.limit = request.getLimit();
        this.textSearch = Strings.nullToEmpty(request.getTextSearch());
        this.validationFilter = protoToFilter(request.getValidationFilter());

        this.includeAllMetricLabels = request.getLabelsCount() == 0;
        if (includeAllMetricLabels) {
            this.shardLabels = ImmutableSet.of(LabelKeys.PROJECT, LabelKeys.CLUSTER, LabelKeys.SERVICE);
            metricLabels = Collections.emptySet();
        } else {
            shardLabels = request.getLabelsList()
                    .stream()
                    .filter(label -> LabelKeys.isShardKeyPart(label))
                    .collect(Collectors.toSet());
            metricLabels = request.getLabelsList().stream()
                    .filter(s -> !shardLabels.contains(s))
                    .collect(Collectors.toSet());
        }
    }

    public String getFolderId() {
        return folderId;
    }

    Selectors getShardSelector() {
        return shardSelector;
    }

    Selectors getMetricSelector() {
        return metricSelector;
    }

    boolean showOnlyClusterServiceLabels() {
        return !includeAllMetricLabels && metricLabels.isEmpty();
    }

    Set<String> getShardLabels() {
        return shardLabels;
    }

    Set<String> getMetricLabels() {
        return metricLabels;
    }

    public int getLimit() {
        return limit;
    }

    public String getTextSearch() {
        return textSearch;
    }

    public LabelValidationFilter getValidationFilter() {
        return validationFilter;
    }

    public boolean isUseNewFormat() {
        return useNewFormat;
    }
}
