package ru.yandex.solomon.coremon.meta.service.handler;

import java.util.List;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Stream;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.coremon.meta.MetabaseMetricId;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardImpl;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardResolver;
import ru.yandex.solomon.coremon.meta.service.ShardKeyAndMetricKey;
import ru.yandex.solomon.labels.protobuf.LabelConverter;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metabase.api.protobuf.Metric;
import ru.yandex.solomon.metabase.api.protobuf.ResolveManyRequest;
import ru.yandex.solomon.metabase.api.protobuf.ResolveManyResponse;

import static java.util.stream.Collectors.toList;
import static ru.yandex.solomon.coremon.meta.service.handler.MetabaseShards.ensureReadyToRead;

/**
 * @author Vladimir Gordiychuk
 */
public class ResolveManyHandler {
    private final MetabaseShardResolver<MetabaseShardImpl> shardResolver;

    public ResolveManyHandler(MetabaseShardResolver<MetabaseShardImpl> shardResolver) {
        this.shardResolver = shardResolver;
    }

    public CompletableFuture<ResolveManyResponse> resolveMany(ResolveManyRequest request) {
        // cross shard resolve not able
        ShardKeyAndMetricKey key = ShardKeyAndMetricKey.of(LabelConverter.protoToLabels(request.getCommonLabelsList()));

        MetabaseShardImpl shard = shardResolver.resolveShard(key.getShardKey());
        ensureReadyToRead(shard);

        Labels shardLabels = key.getShardKey().toLabels();
        Labels commonLabels = key.getMetricKey();

        Stream<MetabaseMetricId> metricsStream;

        boolean useNewFormat = request.getMetricsCount() > 0;

        if (useNewFormat) {
            metricsStream = request.getMetricsList().parallelStream()
                    .map(id -> {
                        Labels metricLabels = LabelConverter.protoToLabels(id.getLabelsList());
                        if (!commonLabels.isEmpty()) {
                            metricLabels = metricLabels.toBuilder().addAll(commonLabels).build();
                        }

                        return new MetabaseMetricId(id.getName(), metricLabels);
                    });
        } else {
            metricsStream = request.getListLabelsList().parallelStream()
                    .map(labelsList -> {
                        Labels metricLabels = LabelConverter.protoToLabels(labelsList);
                        if (!commonLabels.isEmpty()) {
                            metricLabels = metricLabels.toBuilder().addAll(commonLabels).build();
                        }

                        return new MetabaseMetricId("", metricLabels);
                    });
        }

        List<Metric> resolvedMetrics =
                metricsStream.map(metabaseMetricId -> shard.resolveMetric(metabaseMetricId, useNewFormat))
                        .filter(Objects::nonNull)
                        .map(metric -> Proto.toProto(metric, shardLabels))
                        .collect(toList());

        return CompletableFuture.completedFuture(ResolveManyResponse.newBuilder()
                .setStatus(EMetabaseStatusCode.OK)
                .addAllMetrics(resolvedMetrics)
                .build());
    }
}
