package ru.yandex.solomon.coremon.meta.service.handler;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;

import javax.annotation.Nullable;

import com.google.common.collect.Iterables;
import com.google.protobuf.TextFormat;
import io.grpc.Context;
import io.grpc.Status;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.thread.WhatThreadDoes;
import ru.yandex.solomon.coremon.meta.MetabaseMetricId;
import ru.yandex.solomon.coremon.meta.NamedCoremonMetric;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardImpl;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardResolver;
import ru.yandex.solomon.coremon.meta.service.ShardKeyAndMetricKey;
import ru.yandex.solomon.coremon.meta.service.cloud.ReferenceMap;
import ru.yandex.solomon.coremon.meta.service.cloud.ReferenceResolver;
import ru.yandex.solomon.coremon.meta.service.cloud.ResourceFinder;
import ru.yandex.solomon.labels.protobuf.LabelConverter;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metabase.api.protobuf.ResolveOneRequest;
import ru.yandex.solomon.metabase.api.protobuf.ResolveOneResponse;

import static ru.yandex.solomon.coremon.meta.service.handler.MetabaseShards.ensureReadyToRead;

/**
 * @author Vladimir Gordiychuk
 */
public class ResolveOneHandler {
    private final Logger logger = LoggerFactory.getLogger(ResolveOneHandler.class);
    private final MetabaseShardResolver<MetabaseShardImpl> shardResolver;
    private final ReferenceMap referenceMap;

    private String wtd;
    private Context context;
    private Request req;
    private MetabaseShardImpl shard;

    public ResolveOneHandler(MetabaseShardResolver<MetabaseShardImpl> shardResolver, ReferenceResolver referenceResolver, ResourceFinder resourceFinder) {
        this.shardResolver = shardResolver;
        this.referenceMap = new ReferenceMap(referenceResolver, resourceFinder);
    }

    public CompletableFuture<ResolveOneResponse> resolveOne(ResolveOneRequest request) {
        wtd = "Metabase#resolveOne " + TextFormat.shortDebugString(request);
        logger.debug(wtd);
        context = Context.current();

        return callInContext(() -> {
            req = Request.of(request);

            shard = shardResolver.resolveShard(req.shardId);
            ensureReadyToRead(shard);

            referenceMap.addReferences(shard);

            return findResourcesAffectedBySelector()
                    .thenApply(ignore -> toResponse(resolveMetric()));
        });
    }

    private ResolveOneResponse toResponse(@Nullable NamedCoremonMetric metric) {
        if (metric == null) {
            return ResolveOneResponse.newBuilder()
                    .setStatus(EMetabaseStatusCode.NOT_FOUND)
                    .setStatusMessage("By key: " + req.metricId())
                    .build();
        }

        var shardLabels = LabelConverter.labelsToProtoList(req.shardId.toLabels());
        return ResolveOneResponse.newBuilder()
                .setStatus(EMetabaseStatusCode.OK)
                .setMetric(Proto.toProto(metric, shardLabels, referenceMap.resolved()))
                .build();
    }

    private NamedCoremonMetric resolveMetric() {
        return callInContext(() -> {
            switch (referenceMap.resolved().size()) {
                case 0:
                    return resolveById();
                case 1:
                    return resolveByName();
                default:
                    return resolveByFind();
            }
        });
    }

    private NamedCoremonMetric resolveById() {
        return shard.resolveMetric(req.metricId, req.useNewFormat);
    }

    @Nullable
    private NamedCoremonMetric resolveByName() {
        var entry = Iterables.getOnlyElement(referenceMap.resolved().entrySet());
        var labels = entry.getKey();
        var resources = entry.getValue();
        for (var resource : resources) {
            if (resource.replaced) {
                continue;
            }

            var metricId = new MetabaseMetricId(req.metricId.getName(), req.metricId.getLabels().add(labels, resource.resourceId));
            var metric = shard.resolveMetric(metricId, req.useNewFormat);
            if (metric != null) {
                return metric;
            }
        }
        return shard.resolveMetric(req.metricId, req.useNewFormat);
    }

    @Nullable
    private NamedCoremonMetric resolveByFind() {
        var selector = referenceMap.metricSelectors(toSelector(req.metricId));
        var metrics = new ArrayList<NamedCoremonMetric>();
        shard.searchMetrics(selector, 0, Integer.MAX_VALUE, req.useNewFormat, metrics::add);
        var expectLabels = referenceMap.replaceReference(req.metricId.getLabels());
        for (var metric : metrics) {
            if (metric.getLabels().size() != expectLabels.size()) {
                continue;
            }

            var labels = referenceMap.replaceReference(metric.getLabels());
            if (labels.equals(expectLabels)) {
                return metric;
            }
        }
        return null;
    }

    private CompletableFuture<Void> findResourcesAffectedBySelector() {
        return referenceMap.resolveReferenceAffectedBySelector(List.of(shard), toSelector(req.metricId), req.expiredAt)
                .thenRun(this::ensureNotExpired);
    }

    private Selectors toSelector(MetabaseMetricId metricId) {
        var selectors = Selectors.builder(metricId.getName(), metricId.getLabels().size());
        var it = metricId.getLabels().stream().iterator();
        while (it.hasNext()) {
            var label = it.next();
            selectors.add(Selector.exact(label.getKey(), label.getValue()));
        }
        return selectors.build();

    }

    private void ensureNotExpired() {
        if (req.expiredAt != 0 && System.currentTimeMillis() + 100 >= req.expiredAt) {
            throw Status.DEADLINE_EXCEEDED.asRuntimeException();
        }
    }

    private <T> T callInContext(Supplier<T> supplier) {
        var h = WhatThreadDoes.push(wtd);
        var previous = context.attach();
        try {
            return supplier.get();
        } finally {
            h.popSafely();
            context.detach(previous);
        }
    }

    private static record Request(ShardKey shardId, MetabaseMetricId metricId, boolean useNewFormat, long expiredAt) {
        public static Request of(ResolveOneRequest req) {
            var key = ShardKeyAndMetricKey.of(LabelConverter.protoToLabels(req.getLabelsList()));
            boolean useNewFormat = !req.getName().isEmpty();

            MetabaseMetricId metricKey = new MetabaseMetricId(req.getName(), key.getMetricKey());

            return new Request(key.getShardKey(), metricKey, useNewFormat, req.getDeadlineMillis());
        }
    }

}
