package ru.yandex.solomon.coremon.meta.service.handler;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import ru.yandex.solomon.coremon.meta.service.MetabaseShardImpl;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardResolver;
import ru.yandex.solomon.labels.protobuf.LabelConverter;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metabase.api.protobuf.TServerStatusRequest;
import ru.yandex.solomon.metabase.api.protobuf.TServerStatusResponse;
import ru.yandex.solomon.metabase.api.protobuf.TShardStatus;

/**
 * @author Vladimir Gordiychuk
 */
public class ServerStatusHandler {
    private final MetabaseShardResolver<MetabaseShardImpl> shardResolver;

    public ServerStatusHandler(MetabaseShardResolver<MetabaseShardImpl> shardResolver) {
        this.shardResolver = shardResolver;
    }

    public CompletableFuture<TServerStatusResponse> serverStatus(TServerStatusRequest request) {
        if (shardResolver.isLoading()) {
            return CompletableFuture.completedFuture(TServerStatusResponse.newBuilder()
                    .setStatus(EMetabaseStatusCode.NODE_UNAVAILABLE)
                    .setStatusMessage("Shard list not initialized yet")
                    .build());
        }

        var totalShardCount = shardResolver.getTotalShardCount();

        // race at this place not a problem, because client repeat request
        // after few seconds
        long hash = shardResolver.getShardKeysHash();
        if (request.getShardIdsHash() == hash) {
            return CompletableFuture.completedFuture(TServerStatusResponse.newBuilder()
                    .setStatus(EMetabaseStatusCode.OK)
                    .setShardIdsHash(hash)
                    .setTotalPartitionCountKnown(totalShardCount.isPresent())
                    .setTotalPartitionCount(totalShardCount.orElse(0))
                    .setInactivePartitionCount(shardResolver.getInactiveShardCount())
                    .build());
        }

        List<TShardStatus> statuses = shardResolver.getShards()
                .flatMap(shard -> {
                    List<TShardStatus> partitions = new ArrayList<>();
                    // temporary, remove after migration
                    int totalPartitions = shard.getNumPartitions() > 0 ? shard.getNumPartitions() : 1;
                    for (int i = 0; i < totalPartitions; i++) {
                        var partition = TShardStatus.newBuilder()
                                .setShardId(shard.getId())
                                .setNumId(shard.getNumId())
                                .setPartitionId(i)
                                .setTotalPartitions(totalPartitions)
                                .setGenerationId(10) // FIXIT hrustyashko@
                                .setReady(shard.isLoaded())
                                .setReadyWrite(shard.isReadyAcceptNewMetrics())
                                .setReadyRead(shard.isReadyRead())
                                .setMetricCount(shard.fileMetricsCount())
                                .setMetricLimit(shard.maxFileMetrics())
                                .addAllLabels(LabelConverter.labelsToProtoList(shard.getShardKey().toLabels()))
                                .build();
                        partitions.add(partition);
                    }
                    return partitions.stream();
                })
                .collect(Collectors.toList());

        TServerStatusResponse response = TServerStatusResponse.newBuilder()
                .setStatus(EMetabaseStatusCode.OK)
                .setShardIdsHash(hash)
                .setTotalPartitionCountKnown(totalShardCount.isPresent())
                .setTotalPartitionCount(totalShardCount.orElse(0))
                .setInactivePartitionCount(shardResolver.getInactiveShardCount())
                .addAllPartitionStatus(statuses)
                .build();

        return CompletableFuture.completedFuture(response);
    }
}
