package ru.yandex.solomon.coremon.meta.service.handler;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;

import com.google.common.collect.ImmutableSet;
import com.google.protobuf.TextFormat;
import io.grpc.Context;
import io.grpc.Status;

import ru.yandex.misc.thread.WhatThreadDoes;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.labels.LabelsBuilder;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardImpl;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardResolver;
import ru.yandex.solomon.coremon.meta.service.cloud.ReferenceMap;
import ru.yandex.solomon.coremon.meta.service.cloud.ReferenceResolver;
import ru.yandex.solomon.coremon.meta.service.cloud.ResourceFinder;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.protobuf.LabelConverter;
import ru.yandex.solomon.labels.protobuf.LabelSelectorConverter;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.query.ShardSelectors;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metabase.api.protobuf.TUniqueLabelsRequest;
import ru.yandex.solomon.metabase.api.protobuf.TUniqueLabelsResponse;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static ru.yandex.solomon.coremon.meta.service.handler.MetabaseShards.ensureSameProject;
import static ru.yandex.solomon.coremon.meta.service.handler.MetabaseShards.resolveReadyToReadShards;

/**
 * @author Vladimir Gordiychuk
 */
public class UniqueLabelsHandler {
    private final MetabaseShardResolver<MetabaseShardImpl> shardResolver;
    private final ResourceFinder resourceFinder;

    private String wtd;
    private Context context;
    private Req req;
    private Collection<MetabaseShardImpl> shards;
    private final ReferenceMap referenceMap;

    public UniqueLabelsHandler(MetabaseShardResolver<MetabaseShardImpl> shardResolver, ReferenceResolver referenceResolver, ResourceFinder resourceFinder) {
        this.shardResolver = shardResolver;
        this.resourceFinder = resourceFinder;
        this.referenceMap = new ReferenceMap(referenceResolver, resourceFinder);
    }

    public CompletableFuture<TUniqueLabelsResponse> uniqueLabels(TUniqueLabelsRequest request) {
        wtd = "Metabase#uniqueLabels " + TextFormat.shortDebugString(request);
        context = Context.current();
        return callInContext(() -> {
            req = new Req(request);
            shards = resolveReadyToReadShards(shardResolver, req.shardId, req.folderId, req.shardSelector);
            if (shards.isEmpty()) {
                return completedFuture(toResponse(Set.of()));
            }

            referenceMap.addReferences(shards);
            return findResourcesBySelectorReference()
                    .thenCompose(unused -> findResourcesByOtherReference(uniqueLabels()))
                    .thenApply(uniqueLabels -> callInContext(() -> toResponse(replaceReference(filterReplacedResource(uniqueLabels)))));
        });
    }

    private <T> T callInContext(Supplier<T> supplier) {
        var h = WhatThreadDoes.push(wtd);
        var previous = context.attach();
        try {
            return supplier.get();
        } finally {
            h.popSafely();
            context.detach(previous);
        }
    }

    private CompletableFuture<Void> findResourcesBySelectorReference() {
        return referenceMap.resolveReferenceAffectedBySelector(shards, req.metricSelector, req.expiredAt);
    }

    private Set<UniqueLabels> uniqueLabels() {
        return callInContext(() -> {
            var selectors = referenceMap.metricSelectors(req.metricSelector);
            var builder = new UniqueLabels.Builder(referenceMap.referenceLabels(), req.requestedNames);
            Set<UniqueLabels> uniqueLabels = new HashSet<>();
            for (MetabaseShardImpl shard : shards) {
                ensureNotExpired();
                var commonLabels = toLabels(shard.getShardKey());
                shard.searchLabels(selectors, 0, 0, req.useNewFormat, labels -> {
                    builder.clear();
                    builder.addCommon(commonLabels);
                    builder.addLabels(labels);
                    uniqueLabels.add(builder.build());
                });
            }
            return uniqueLabels;
        });
    }

    private Labels toLabels(ShardKey shardKey) {
        if (req.includeShard) {
            LabelsBuilder builder = new LabelsBuilder(3);
            shardKey.forEachLabel((key, value) -> {
                if (req.requestedNames.contains(key)) {
                    builder.add(key, value);
                }
            });
            return builder.build();
        } else {
            return Labels.of();
        }
    }

    private CompletableFuture<Set<UniqueLabels>> findResourcesByOtherReference(Set<UniqueLabels> uniqueLabels) {
        Set<String> labelToResolve = referenceMap.unresolvedReferenceLabel();
        if (labelToResolve.isEmpty()) {
            return completedFuture(uniqueLabels);
        }

        Set<String> unresolvedResourceIds = new HashSet<>();
        for (var unique : uniqueLabels) {
            unresolvedResourceIds.addAll(unique.referenceIds);
        }

        if (unresolvedResourceIds.isEmpty()) {
            return completedFuture(uniqueLabels);
        }

        String projectId = ensureSameProject(shards);
        return resourceFinder.resolve(projectId, unresolvedResourceIds, req.expiredAt)
                .thenApply(map -> {
                    for (var label : labelToResolve) {
                        referenceMap.addResource(label, map);
                    }
                    return uniqueLabels;
                });
    }

    public Set<Labels> filterReplacedResource(Set<UniqueLabels> uniqueLabels) {
        Set<Labels> result = new HashSet<>(uniqueLabels.size());
        uniqueLoop:
        for (var unique : uniqueLabels) {
            for (var resourceId : unique.referenceIds) {
                var resource = referenceMap.getResourceById(resourceId);
                if (resource != null && resource.replaced) {
                    continue uniqueLoop;
                }
            }

            result.add(unique.labels);
        }

        return result;
    }

    public Set<Labels> replaceReference(Set<Labels> uniqueLabels) {
        Set<Labels> result = new HashSet<>(uniqueLabels.size());
        LabelsBuilder labelsBuilder = new LabelsBuilder(req.requestedNames.size());
        for (var labels : uniqueLabels) {
            labelsBuilder.clear();
            labels.forEach(label -> {
                var resource = referenceMap.getResourceById(label.getKey(), label.getValue());
                if (resource == null) {
                    labelsBuilder.add(label);
                } else {
                    labelsBuilder.add(label.getKey(), resource.getNameOrId());
                }
            });

            result.add(labelsBuilder.build());
        }
        return result;
    }

    private TUniqueLabelsResponse toResponse(Set<Labels> uniqueLabels) {
        var result = TUniqueLabelsResponse.newBuilder()
                .setStatus(EMetabaseStatusCode.OK);

        for (Labels labels : uniqueLabels) {
            LabelConverter.addLabels(result.addLabelListsBuilder(), labels);
        }

        return result.build();
    }

    private void ensureNotExpired() {
        if (req.expiredAt != 0 && System.currentTimeMillis() + 100 >= req.expiredAt) {
            throw Status.DEADLINE_EXCEEDED.asRuntimeException();
        }
    }

    private static class Req {
        private final int shardId;
        private final boolean useNewFormat;
        private final String folderId;
        private final Selectors shardSelector;
        private final Selectors metricSelector;
        private final Set<String> requestedNames;
        private final boolean includeShard;
        private final long expiredAt;

        public Req(TUniqueLabelsRequest request) {
            shardId = request.getShardId();
            useNewFormat = request.hasNewSelectors();
            folderId = request.getFolderId();

            Selectors selector = useNewFormat
                    ? LabelSelectorConverter.protoToSelectors(request.getNewSelectors())
                    : LabelSelectorConverter.protoToSelectors(request.getSelectorsList());

            shardSelector = ShardSelectors.onlyShardKey(selector);
            metricSelector = ShardSelectors.withoutShardKey(selector);

            requestedNames = ImmutableSet.copyOf(request.getNamesList());
            includeShard = requestedNames.stream().anyMatch(LabelKeys::isShardKeyPart);
            expiredAt = request.getDeadlineMillis();
        }
    }

    private static record UniqueLabels(Labels labels, Set<String> referenceIds) {

        private static class Builder {
            private final Set<String> referenceLabels;
            private final Set<String> requestedNames;
            private final List<String> referenceIds;
            private final LabelsBuilder labelsBuilder;

            public Builder(Set<String> referenceLabels, Set<String> requestedNames) {
                this.referenceLabels = referenceLabels;
                this.requestedNames = requestedNames;
                this.referenceIds = new ArrayList<>(referenceLabels.size());
                this.labelsBuilder = new LabelsBuilder(requestedNames.size());
            }

            public void clear() {
                labelsBuilder.clear();
                referenceIds.clear();
            }

            public void addCommon(Labels labels) {
                labelsBuilder.addAll(labels);
            }

            public void addLabels(Labels labels) {
                labels.forEach(label -> {
                    if (referenceLabels.contains(label.getKey())) {
                        referenceIds.add(label.getValue());
                    }

                    if (requestedNames.contains(label.getKey())) {
                        labelsBuilder.add(label);
                    }
                });
            }

            public UniqueLabels build() {
                if (referenceLabels.isEmpty()) {
                    return new UniqueLabels(labelsBuilder.build(), Set.of());
                } else {
                    return new UniqueLabels(labelsBuilder.build(), new HashSet<>(referenceIds));
                }
            }
        }
    }
}
