package ru.yandex.solomon.coremon.meta.ttl;

import java.util.HashMap;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.ConcurrentLinkedQueue;

import javax.annotation.Nullable;

import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;

import ru.yandex.solomon.core.conf.ShardKeyAndId;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.FileCoremonMetric;
import ru.yandex.solomon.coremon.meta.MetricsCollection;
import ru.yandex.solomon.name.resolver.client.Resource;
import ru.yandex.solomon.util.collection.CloseableIterator;

import static ru.yandex.solomon.coremon.meta.CoremonMetric.UNKNOWN_LAST_POINT_SECONDS;


/**
 * @author Sergey Polovko
 */
public class Batcher implements AutoCloseable {
    private static final int MAX_BATCH_SIZE = 10_000;

    private final ShardKeyAndId key;
    private final CloseableIterator<? extends CoremonMetric> metricIt;
    private final Set<String> references;
    private final UnknownReferenceTracker referenceTracker;
    private final int size;
    private int skipped;
    private int unknownReference;
    private final int maxBatchSize;
    private final int expireTimeSeconds;

    private final Int2ObjectMap<LoadMetaBatch> loadMetaBatches = new Int2ObjectOpenHashMap<>();
    private final Int2ObjectMap<DeleteBatch> deleteBatches = new Int2ObjectOpenHashMap<>();
    private final Map<String, Resource> resourceById = new HashMap<>();
    @Nullable
    private LoadResourceBatch loadResourceBatch;

    private final Queue<Batch> updates = new ConcurrentLinkedQueue<>();


    public Batcher(ShardKeyAndId key, MetricsCollection<? extends CoremonMetric> metrics, int expireTimeSeconds, Set<String> references, UnknownReferenceTracker referenceTracker) {
        this(key, metrics, MAX_BATCH_SIZE, expireTimeSeconds, references, referenceTracker);
    }

    public Batcher(ShardKeyAndId key, MetricsCollection<? extends CoremonMetric> metrics, int maxBatchSize, int expireTimeSeconds, Set<String> references, UnknownReferenceTracker referenceTracker) {
        this.key = key;
        this.metricIt = metrics.iterator();
        this.size = metrics.size();
        this.maxBatchSize = maxBatchSize;
        this.expireTimeSeconds = expireTimeSeconds;
        this.references = references;
        this.referenceTracker = referenceTracker;
    }

    public int size() {
        return size;
    }

    public int getMaxBatchSize() {
        return maxBatchSize;
    }

    public int getSkipped() {
        return skipped;
    }

    public int getUnknownReference() {
        return unknownReference;
    }

    @Nullable
    public Batch nextBatch() {
        // (1) drain all last updates
        var updateBatch = processUpdates();
        if (updateBatch != null) {
            return updateBatch;
        }

        // (2) drain collection
        var processBatch = processNextMetrics();
        if (processBatch != null) {
            return processBatch;
        }

        // (3) return load meta batch if any
        var loadBatch = nextLoadBatch();
        if (loadBatch != null) {
            return loadBatch;
        }

        // (4) return load resource batch if any
        var loadResourceBatch = nextLoadResourceBatch();
        if (loadResourceBatch != null) {
            return loadResourceBatch;
        }

        // (5) return delete batch if any
        return nextDeleteBatch();
    }

    private Batch processUpdates() {
        Batch update;
        while ((update = updates.peek()) != null) {
            if (update.isEmpty()) {
                updates.poll();
            }

            final Batch batch;
            if (update instanceof Batcher.LoadMetaBatch) {
                batch = processLoadMetaUpdate((LoadMetaBatch) update);
            } else if (update instanceof Batcher.LoadResourceBatch) {
                batch = processLoadResourceUpdate((LoadResourceBatch) update);
            } else {
                String message = "unknown batch type: " + update.getClass();
                throw new UnsupportedOperationException(message);
            }

            if (batch != null) {
                return batch;
            }
        }

        return null;
    }

    private Batch processNextMetrics() {
        while (metricIt.hasNext()) {
            try (final CoremonMetric metric = metricIt.next()) {
                final int lastPointSeconds = metric.getLastPointSeconds();

                if (lastPointSeconds == UNKNOWN_LAST_POINT_SECONDS || lastPointSeconds <= expireTimeSeconds) {
                    // also load metadata if the last point's time <= expire time,
                    // to double check what we gonna delete
                    final Batch batch = loadMetaBatches.computeIfAbsent(metric.getShardId(), LoadMetaBatch::new);
                    batch.add(new FileCoremonMetric(metric));

                    if (batch.size() >= maxBatchSize) {
                        loadMetaBatches.remove(metric.getShardId());
                        return batch;
                    }
                } else if (!references.isEmpty()) {
                    var batch = deleteReplacedReference(metric);
                    if (batch != null) {
                        return batch;
                    }
                } else {
                    skipped++;
                }
            }
        }

        return null;
    }

    private Batch processLoadMetaUpdate(LoadMetaBatch update) {
        FileCoremonMetric metric;
        while ((metric = update.pop()) != null) {
            final int lastPointSeconds = metric.getLastPointSeconds();
            if ((lastPointSeconds != UNKNOWN_LAST_POINT_SECONDS && lastPointSeconds <= expireTimeSeconds)) {
                // here we get true last point time from Stockpile
                // and can safely delete this metric
                var batch = addDelete(metric);
                if (batch != null) {
                    return batch;
                }
            } else if (!references.isEmpty()) {
                var batch = deleteReplacedReference(metric);
                if (batch != null) {
                    return batch;
                }
            } else {
                skipped++;
            }
        }

        return null;
    }

    private Batch processLoadResourceUpdate(LoadResourceBatch update) {
        resourceById.putAll(update.resourceById);
        FileCoremonMetric metric;
        while ((metric = update.pop()) != null) {
            var batch = deleteReplacedReference(metric);
            if (batch != null) {
                return batch;
            }
        }
        return null;
    }

    private Batch deleteReplacedReference(CoremonMetric metric) {
        var labels = metric.getLabels();
        boolean hasUnresolved = false;
        for (int index = 0; index < labels.size(); index++) {
            var label = labels.at(index);
            if (!references.contains(label.getKey())) {
                continue;
            }

            var resourceId = label.getValue();
            if (resourceById.containsKey(resourceId)) {
                var resource = resourceById.get(resourceId);
                if (resource == null) {
                    unknownReference++;
                    referenceTracker.unknownReference(key, resourceId, label.getKey(), metric);
                } else if (resource.replaced && resource.deletedAt != 0) {
                    return addDelete(FileCoremonMetric.of(metric));
                }
            } else {
                hasUnresolved = true;
                if (loadResourceBatch == null) {
                    loadResourceBatch = new LoadResourceBatch();
                }
                loadResourceBatch.addResourceId(resourceId);
            }
        }

        if (hasUnresolved) {
            loadResourceBatch.add(FileCoremonMetric.of(metric));
            if (loadResourceBatch.size() > maxBatchSize) {
                var batch = loadResourceBatch;
                loadResourceBatch = null;
                return batch;
            }
        } else {
            skipped++;
        }

        return null;
    }

    @Nullable
    private Batch addDelete(FileCoremonMetric metric) {
        DeleteBatch deleteBatch = deleteBatches.computeIfAbsent(metric.getShardId(), DeleteBatch::new);
        deleteBatch.add(metric);
        if (deleteBatch.size() >= maxBatchSize) {
            deleteBatches.remove(metric.getShardId());
            return deleteBatch;
        }

        return null;
    }

    @Nullable
    private LoadResourceBatch nextLoadResourceBatch() {
        var batch = loadResourceBatch;
        if (batch != null) {
            loadResourceBatch = null;
        }
        return batch;
    }

    @Nullable
    private LoadMetaBatch nextLoadBatch() {
        var loadIt = loadMetaBatches.values().iterator();
        if (loadIt.hasNext()) {
            var batch = loadIt.next();
            loadIt.remove();
            return batch;
        }
        return null;
    }

    @Nullable
    private DeleteBatch nextDeleteBatch() {
        var deleteIt = deleteBatches.values().iterator();
        if (deleteIt.hasNext()) {
            var batch = deleteIt.next();
            deleteIt.remove();
            return batch;
        }
        return null;
    }

    public void update(Batch batch) {
        updates.offer(batch);
    }

    @Override
    public void close() {
        metricIt.close();
    }

    /**
     * Batch of metrics to be deleted.
     */
    public static final class DeleteBatch extends StockpileBatch {
        public DeleteBatch(int stockpileShardId) {
            super(stockpileShardId);
        }
    }

    /**
     * Batch of metrics that need to load additional metadata.
     */
    public static final class LoadMetaBatch extends StockpileBatch {
        public LoadMetaBatch(int stockpileShardId) {
            super(stockpileShardId);
        }
    }

    public static final class LoadResourceBatch extends Batch {
        private final Map<String, Resource> resourceById = new HashMap<>();

        public void addResourceId(String resourceId) {
            resourceById.put(resourceId, null);
        }

        public Set<String> resourceIds() {
            return resourceById.keySet();
        }

        public void addResource(Resource resource) {
            resourceById.put(resource.resourceId, resource);
        }
    }
}
