package ru.yandex.solomon.coremon.meta.ttl;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import ru.yandex.misc.ExceptionUtils;
import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.coremon.meta.ttl.tasks.FailedTask;
import ru.yandex.solomon.coremon.meta.ttl.tasks.FinishedTask;
import ru.yandex.solomon.coremon.meta.ttl.tasks.RunningTask;
import ru.yandex.solomon.coremon.meta.ttl.tasks.Task;
import ru.yandex.solomon.coremon.meta.ttl.tasks.TaskStats;
import ru.yandex.solomon.staffOnly.html.HtmlWriterWithCommonLibraries;
import ru.yandex.solomon.staffOnly.manager.ManagerController;
import ru.yandex.solomon.staffOnly.manager.find.NamedObjectId;
import ru.yandex.solomon.util.time.DurationUtils;
import ru.yandex.solomon.util.time.InstantUtils;


/**
 * @author Sergey Polovko
 */
final class DeletionManagerWww {
    private DeletionManagerWww() {}

    public static void tasksInfo(HtmlWriterWithCommonLibraries hw, Map<Integer, Task> tasks, SolomonConfWithContext conf) {
        final class Row {
            private final int numId;
            private final String shardId;
            private final Task task;

            private Row(int numId, String shardId, Task task) {
                this.numId = numId;
                this.shardId = shardId;
                this.task = task;
            }
        }

        List<Row> rows = tasks.entrySet().stream()
            .map(e -> {
                var shard = conf.getShardByNumIdOrNull(e.getKey());
                if (shard == null) {
                    return null;
                }
                return new Row(e.getKey(), shard.getId(), e.getValue());
            })
            .filter(Objects::nonNull)
            .sorted(Comparator.<Row>comparingLong(r -> r.task.getCreatedAtMillis()).reversed())
            .collect(Collectors.toList());

        hw.h2("Tasks Per Shard");
        hw.tableTable(() -> {
            hw.trThs("CreatedAt", "NumId", "ShardId", "Status", "Duration", "Stats", "Details");

            for (Row row : rows) {
                final int numId = row.numId;
                final String unsignedNumId = Integer.toUnsignedString(numId);
                final String shardId = row.shardId;
                final Task task = row.task;
                final TaskStats stats = task.getStats();

                hw.tr(() -> {
                    // CreatedAt
                    hw.tdText(InstantUtils.formatToMillis(task.getCreatedAtMillis()));

                    // NumId
                    hw.td(() -> {
                        String shardHref = ManagerController.namedObjectLink(new NamedObjectId("ru.yandex.solomon.coremon.CoremonShard", unsignedNumId));
                        hw.aHref(shardHref, unsignedNumId);
                    });

                    // ShardId
                    hw.td(() -> {
                        String shardHref = ManagerController.namedObjectLink(new NamedObjectId("ru.yandex.solomon.coremon.CoremonShard", unsignedNumId));
                        hw.aHref(shardHref, shardId);
                    });

                    // Status
                    hw.td(() -> {
                        if (task instanceof RunningTask) {
                            hw.label("info", "RUNNING");
                        } else if (task instanceof FinishedTask) {
                            hw.label("success", "FINISHED");
                        } else if (task instanceof FailedTask) {
                            hw.label("danger", "FAILED");
                        } else {
                            hw.label("default", "UNKNOWN");
                        }
                    });

                    // Duration
                    hw.tdText(DurationUtils.formatDurationMillis(stats.getDurationMillis()));

                    // Stats
                    hw.td(() -> {
                        hw.preText(formatStats(stats));
                    });

                    // Details
                    hw.td(() -> {
                        if (task instanceof RunningTask) {
                            hw.divWithClass("row", () -> {
                                RunningTask runningTask = (RunningTask) task;
                                hw.preText(String.format(
                                    "InFlight [metaLoad: %d, resourceLoad: %d, delete: %d]",
                                    runningTask.getMetaLoadInFlight(),
                                    runningTask.getResourceLoadInFlight(),
                                    runningTask.getDeleteInFlight()));
                                hw.divWithClass("row", () -> {
                                    hw.progress(stats.getProgress());
                                });
                            });
                        } else if (task instanceof FailedTask) {
                            FailedTask failedTask = (FailedTask) task;
                            hw.divWithClass("row", () -> {
                                hw.write(InstantUtils.formatToMillis(failedTask.getExceptionAtMillis()));
                            });
                            hw.divWithClass("row", () -> {
                                hw.preText(ExceptionUtils.getStackTrace(failedTask.getException()));
                            });
                        }
                    });
                });
            }
        });
    }

    private static String formatStats(TaskStats stats) {
        String deleted = String.format(
                "deleted %d (%d -> %d, -%.1f%%)",
                stats.getDeletedMetrics(),
                stats.getTotalMetrics(),
                stats.getTotalMetrics() - stats.getDeletedMetrics(),
                stats.getTotalMetrics() == 0
                        ? 0.0
                        : 100. * stats.getDeletedMetrics() / stats.getTotalMetrics());
        if (stats.getUnknownReference() == 0) {
            return deleted;
        }

        return String.format("%s\nunknown reference %d", deleted, stats.getUnknownReference());
    }
}
