package ru.yandex.solomon.coremon.meta.ttl;

import java.time.Instant;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import com.google.common.base.Strings;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;
import org.springframework.http.MediaType;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.solomon.auth.http.HttpAuthenticator;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.core.conf.ShardConfDetailed;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.coremon.meta.service.ShardLocator;
import ru.yandex.solomon.coremon.meta.ttl.tasks.Task;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.SelectorType;
import ru.yandex.solomon.staffOnly.RootLink;
import ru.yandex.solomon.staffOnly.html.AHref;
import ru.yandex.solomon.staffOnly.manager.ManagerWriterContext;
import ru.yandex.solomon.staffOnly.manager.find.NamedObjectId;
import ru.yandex.solomon.staffOnly.manager.table.Column;
import ru.yandex.solomon.staffOnly.manager.table.Table;
import ru.yandex.solomon.staffOnly.manager.table.TableRecord;

import static ru.yandex.solomon.staffOnly.manager.ManagerController.namedObjectLink;

/**
 * @author Vladimir Gordiychuk
 */
@RestController
@Import({
        ManagerWriterContext.class,
        DeletionManager.class
})
public class UnknownReferenceWww {
    @Autowired
    private HttpAuthenticator authenticator;
    @Autowired
    private InternalAuthorizer authorizer;
    @Autowired
    private ManagerWriterContext contect;
    @Autowired
    private DeletionManager manager;
    @Autowired
    private ShardLocator shards;
    @Autowired
    private SolomonConfHolder confHolder;

    @Bean
    public RootLink unknownReferencesLink() {
        return new RootLink("/unknown-reference", "Unknown references");
    }

    @RequestMapping(value = "/unknown-reference", produces = MediaType.TEXT_HTML_VALUE)
    public CompletableFuture<String> localShards(
            @RequestParam(value = "sortBy", defaultValue = "1") int sortBy,
            @RequestParam(value = "filterById", defaultValue = "") String filterById,
            ServerHttpRequest request)
    {
        return authenticator.authenticate(request)
                .thenCompose(authSubject -> authorizer.authorize(authSubject))
                .thenApply(account -> makeContent(sortBy, filterById));
    }

    private String makeContent(int sortBy, String filterById) {
        var filter = makeFilter(filterById);
        var records = manager.tasksByShardNumId.entrySet()
                .stream()
                .filter(entry -> {
                    if (!shards.isLocal(entry.getKey())) {
                        return false;
                    }

                    var stats = entry.getValue().getStats();
                    return stats.getUnknownReference() != 0;
                })
                .map(entry -> {
                    var shard = confHolder.getConfOrThrow().getShardByNumIdOrNull(entry.getKey());
                    if (shard == null || !shard.isCorrect()) {
                        return null;
                    }

                    return Record.of(shard.getConfOrThrow(), entry.getValue());
                })
                .filter(record -> record != null && filter.test(record))
                .sorted(Comparator.comparingLong(o -> o.createdAt))
                .collect(Collectors.toList());

        List<Column<Record>> columns = makeColumns();
        return new Table<>("Shards with unknown references(" + DataSize.shortString(records.size()) + ")", contect, columns, records, sortBy).genString();
    }

    private Predicate<Record> makeFilter(String filterById) {
        var selector = makeSelector(filterById);
        return record -> selector.match(record.id);
    }

    private Selector makeSelector(String filterById) {
        if (Strings.isNullOrEmpty(filterById)) {
            return Selector.any("id");
        }

        if (filterById.startsWith("!")) {
            return SelectorType.NOT_GLOB.create("id", filterById.substring(1));
        }

        return SelectorType.GLOB.create("id", filterById);
    }

    private List<Column<Record>> makeColumns() {
        return List.of(
                Column.of(
                        "NumId",
                        r -> {
                            String shardId = Integer.toUnsignedString(r.numId);
                            return new AHref(namedObjectLink(new NamedObjectId("ru.yandex.solomon.coremon.CoremonShard", shardId)), shardId);
                        },
                        (o1, o2) -> Integer.compareUnsigned(o1.numId, o2.numId)),
                Column.of(
                        "ShardId",
                        r -> {
                            return new AHref(namedObjectLink(new NamedObjectId("ru.yandex.solomon.coremon.CoremonShard", r.id)), r.id);
                        },
                        Comparator.comparing(r -> r.id)),
                Column.of("Metrics",
                        r -> DataSize.shortString(r.totalMetrics),
                        Comparator.comparingInt(o2 -> o2.totalMetrics)
                ),
                Column.of("Unknown reference",
                        r -> DataSize.shortString(r.unknownReferenceMetrics),
                        Comparator.comparingInt(o2 -> o2.unknownReferenceMetrics)
                ),
                Column.of("Last Check",
                        r -> Instant.ofEpochMilli(r.createdAt),
                        Comparator.comparingLong(o2 -> o2.createdAt)
                ));
    }

    private static class Record implements TableRecord {
        long createdAt;
        int numId;
        String id;
        int totalMetrics;
        int unknownReferenceMetrics;

        public static Record of(ShardConfDetailed shard, Task task) {
            var r = new Record();

            r.createdAt = task.getCreatedAtMillis();
            r.numId = shard.getNumId();
            r.id = shard.getId();
            var stats = task.getStats();
            r.totalMetrics = stats.getTotalMetrics();
            r.unknownReferenceMetrics = stats.getUnknownReference();
            return r;
        }
    }
}
