package ru.yandex.solomon.coremon.meta.ttl.tasks;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;

import it.unimi.dsi.fastutil.longs.Long2IntOpenHashMap;

import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.ttl.Batch;
import ru.yandex.solomon.coremon.meta.ttl.Batcher;
import ru.yandex.solomon.coremon.meta.ttl.Batcher.LoadResourceBatch;
import ru.yandex.solomon.coremon.meta.ttl.Deleter;
import ru.yandex.solomon.coremon.meta.ttl.MetaLoader;
import ru.yandex.solomon.coremon.meta.ttl.ResourceLoader;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.solomon.util.time.InstantUtils;
import ru.yandex.stockpile.api.MetricMeta;


/**
 * @author Sergey Polovko
 */
public final class RunningTask implements Task {

    private final long createdAtMillis;
    private final AsyncActorRunner actorRunner;

    private final int numId;
    private final String projectId;
    private final Batcher batcher;
    private final MetaLoader metaLoader;
    private final Deleter deleter;
    private final ResourceLoader resourceLoader;

    private final AtomicInteger metaLoadInFlight = new AtomicInteger();
    private final AtomicInteger resourceLoadInFlight = new AtomicInteger();
    private final AtomicInteger deleteInFlight = new AtomicInteger();
    private final AtomicInteger deletedMetrics = new AtomicInteger();

    private final AtomicReference<CompletableFuture<?>> nextAsyncOp = new AtomicReference<>(CompletableFuture.completedFuture(null));

    public RunningTask(
            String projectId,
            int numId,
            Batcher batcher,
            MetaLoader metaLoader,
            ResourceLoader resourceLoader,
            Deleter deleter,
            Executor executor,
            int maxAsyncOperationsPerTask)
    {
        this.projectId = projectId;
        this.numId = numId;
        this.batcher = batcher;
        this.metaLoader = metaLoader;
        this.resourceLoader = resourceLoader;
        this.deleter = deleter;

        this.createdAtMillis = System.currentTimeMillis();
        this.actorRunner = new AsyncActorRunner(this::body, executor, maxAsyncOperationsPerTask);
    }

    public CompletableFuture<Void> start() {
        return actorRunner.start();
    }

    public void stop() {
        actorRunner.stop();
    }

    private CompletableFuture<?> body() {
        Batch batch = batcher.nextBatch();
        if (batch == null) {
            var future = nextAsyncOp.get();

            // we have no batches to process right now, but there is some async
            // operation in flight, so we must to wait it and return here again
            if (metaLoadInFlight.get() > 0 || deleteInFlight.get() > 0 || resourceLoadInFlight.get() > 0) {
                return future;
            }

            batch = batcher.nextBatch();
            if (batch == null) {
                // return DONE_MARKER iff there is no non completed async operations right now
                return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
            }
        }

        if (batch instanceof Batcher.LoadMetaBatch) {
            return loadMeta((Batcher.LoadMetaBatch) batch);
        } else if (batch instanceof Batcher.DeleteBatch) {
            return delete((Batcher.DeleteBatch) batch);
        } else if (batch instanceof Batcher.LoadResourceBatch) {
            return loadResource((LoadResourceBatch) batch);
        } else {
            String message = "unknown batch type: " + batch.getClass();
            return CompletableFuture.failedFuture(new RuntimeException(message));
        }
    }

    private CompletableFuture<?> loadMeta(Batcher.LoadMetaBatch batch) {
        metaLoadInFlight.incrementAndGet();
        return metaLoader.loadMeta(batch)
            .thenAccept(metas -> {
                Long2IntOpenHashMap timestamps = new Long2IntOpenHashMap(metas.size());
                timestamps.defaultReturnValue(0);

                for (MetricMeta meta : metas) {
                    timestamps.put(meta.getLocalId(), InstantUtils.millisecondsToSeconds(meta.getLastTsMillis()));
                }

                for (int i = 0; i < batch.size(); i++) {
                    CoremonMetric metric = batch.getMetric(i);
                    metric.setLastPointSeconds(timestamps.get(metric.getLocalId()));
                }

                batcher.update(batch);
            })
            .whenComplete((r, t) -> {
                metaLoadInFlight.decrementAndGet();
                completeAsyncOp();
            });
    }

    private CompletableFuture<?> loadResource(LoadResourceBatch batch) {
        resourceLoadInFlight.incrementAndGet();
        return resourceLoader.loadResources(projectId, batch)
                .thenAccept(resources -> {
                    for (var resource : resources) {
                        batch.addResource(resource);
                    }

                    batcher.update(batch);
                })
                .whenComplete((r, t) -> {
                    resourceLoadInFlight.decrementAndGet();
                    completeAsyncOp();
                });
    }

    private CompletableFuture<?> delete(Batcher.DeleteBatch batch) {
        deleteInFlight.incrementAndGet();
        return deleter.delete(numId, batch)
            .whenComplete((r, t) -> {
                deleteInFlight.decrementAndGet();
                if (t == null) {
                    deletedMetrics.addAndGet(batch.size());
                }
                completeAsyncOp();
            });
    }

    private void completeAsyncOp() {
        var future = this.nextAsyncOp.getAndSet(new CompletableFuture<>());
        future.complete(null);
    }

    @Override
    public TaskStats getStats() {
        long durationMillis = System.currentTimeMillis() - createdAtMillis;
        int total = batcher.size();
        int processed = batcher.getSkipped() + deletedMetrics.get();
        return new TaskStats(batcher.size(), deletedMetrics.get(), batcher.getUnknownReference(), durationMillis, 100. * processed / total);
    }

    @Override
    public long getCreatedAtMillis() {
        return createdAtMillis;
    }

    public int getMetaLoadInFlight() {
        return metaLoadInFlight.get();
    }

    public int getDeleteInFlight() {
        return deleteInFlight.get();
    }

    public int getResourceLoadInFlight() {
        return resourceLoadInFlight.get();
    }

    public FinishedTask toFinished() {
        return new FinishedTask(createdAtMillis, getStats());
    }

    public FailedTask toFailed(Throwable t) {
        return new FailedTask(createdAtMillis, getStats(), t);
    }
}
