package ru.yandex.solomon.coremon.meta;

import java.util.Iterator;
import java.util.List;
import java.util.stream.Collectors;

import org.junit.Assert;
import org.junit.Test;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.util.time.InstantUtils;

import static ru.yandex.solomon.coremon.meta.CoremonMetricHelper.assertEquals;


/**
 * @author Sergey Polovko
 */
@SuppressWarnings("Duplicates")
public class CoremonMetricArrayTest {

    @Test
    public void empty() {
        try (CoremonMetricArray a = new CoremonMetricArray(0)) {
            // basic contract
            Assert.assertEquals(0, a.size());
            Assert.assertEquals(0, a.capacity());
            Assert.assertTrue(a.isEmpty());

            // labels iterator
            Iterator<Labels> it = a.labelsIterator();
            Assert.assertFalse(it.hasNext());
            Assert.assertFalse(it.hasNext()); // idempotent

            // stream
            List<CoremonMetric> metrics = a.stream().collect(Collectors.toList());
            Assert.assertTrue(metrics.isEmpty());
        }
    }

    @Test
    public void constructedFromArray() {
        CoremonMetric[] expected = {
            new FileCoremonMetric(1, 2, Labels.of("a", "b"), 100, MetricType.DGAUGE),
            new FileCoremonMetric(3, 4, Labels.of("c", "d"), 200, MetricType.COUNTER),
            new FileCoremonMetric(5, 6, Labels.of("e", "f"), 300, MetricType.RATE),
        };

        try (CoremonMetricArray a = new CoremonMetricArray(expected)) {
            // basic contract
            Assert.assertEquals(3, a.size());
            Assert.assertEquals(3, a.capacity());
            Assert.assertFalse(a.isEmpty());

            // labels iterator
            Iterator<Labels> it = a.labelsIterator();
            Assert.assertTrue(it.hasNext());
            Assert.assertEquals(Labels.of("a", "b"), it.next());
            Assert.assertTrue(it.hasNext());
            Assert.assertEquals(Labels.of("c", "d"), it.next());
            Assert.assertTrue(it.hasNext());
            Assert.assertEquals(Labels.of("e", "f"), it.next());
            Assert.assertFalse(it.hasNext());
            Assert.assertFalse(it.hasNext()); // idempotent

            // stream
            CoremonMetric[] metrics = a.stream().toArray(CoremonMetric[]::new);
            Assert.assertEquals(3, metrics.length);
            assertEquals(expected[0], metrics[0]);
            assertEquals(expected[1], metrics[1]);
            assertEquals(expected[2], metrics[2]);

            // get labels
            Assert.assertEquals(Labels.of("a", "b"), a.getLabels(0));
            Assert.assertEquals(Labels.of("c", "d"), a.getLabels(1));
            Assert.assertEquals(Labels.of("e", "f"), a.getLabels(2));

            // get
            assertEquals(expected[0], a.get(0));
            assertEquals(expected[1], a.get(1));
            assertEquals(expected[2], a.get(2));
        }
    }

    @Test
    public void copyConstructed() {
        CoremonMetric[] expected = {
            new FileCoremonMetric(1, 2, Labels.of("a", "b"), 100, MetricType.DGAUGE),
            new FileCoremonMetric(3, 4, Labels.of("c", "d"), 200, MetricType.COUNTER),
            new FileCoremonMetric(5, 6, Labels.of("e", "f"), 300, MetricType.RATE),
        };

        try (CoremonMetricArray a = new CoremonMetricArray(expected)) {
            try (CoremonMetricArray copyA = new CoremonMetricArray(a)) {
                // basic contract
                Assert.assertEquals(3, copyA.size());
                Assert.assertEquals(3, copyA.capacity());
                Assert.assertFalse(copyA.isEmpty());

                // labels iterator
                Iterator<Labels> it = copyA.labelsIterator();
                Assert.assertTrue(it.hasNext());
                Assert.assertEquals(Labels.of("a", "b"), it.next());
                Assert.assertTrue(it.hasNext());
                Assert.assertEquals(Labels.of("c", "d"), it.next());
                Assert.assertTrue(it.hasNext());
                Assert.assertEquals(Labels.of("e", "f"), it.next());
                Assert.assertFalse(it.hasNext());
                Assert.assertFalse(it.hasNext()); // idempotent

                // stream
                CoremonMetric[] metrics = copyA.stream().toArray(CoremonMetric[]::new);
                Assert.assertEquals(3, metrics.length);
                assertEquals(expected[0], metrics[0]);
                assertEquals(expected[1], metrics[1]);
                assertEquals(expected[2], metrics[2]);

                // get labels
                Assert.assertEquals(Labels.of("a", "b"), copyA.getLabels(0));
                Assert.assertEquals(Labels.of("c", "d"), copyA.getLabels(1));
                Assert.assertEquals(Labels.of("e", "f"), copyA.getLabels(2));

                // get
                assertEquals(expected[0], copyA.get(0));
                assertEquals(expected[1], copyA.get(1));
                assertEquals(expected[2], copyA.get(2));
            }

            // basic contract
            Assert.assertEquals(3, a.size());
            Assert.assertEquals(3, a.capacity());
            Assert.assertFalse(a.isEmpty());

            // get
            assertEquals(expected[0], a.get(0));
            assertEquals(expected[1], a.get(1));
            assertEquals(expected[2], a.get(2));
        }
    }

    @Test
    public void addExplicit() {
        try (CoremonMetricArray a = new CoremonMetricArray(0)) {
            a.add(1, 2, Labels.of("a", "b"), 100, MetricType.DGAUGE);
            a.add(3, 4, Labels.of("c", "d"), 200, MetricType.COUNTER);
            a.add(5, 6, Labels.of("e", "f"), 300, MetricType.RATE);

            // basic contract
            Assert.assertEquals(3, a.size());
            Assert.assertEquals(10, a.capacity());
            Assert.assertFalse(a.isEmpty());

            // labels iterator
            Iterator<Labels> it = a.labelsIterator();
            Assert.assertTrue(it.hasNext());
            Assert.assertEquals(Labels.of("a", "b"), it.next());
            Assert.assertTrue(it.hasNext());
            Assert.assertEquals(Labels.of("c", "d"), it.next());
            Assert.assertTrue(it.hasNext());
            Assert.assertEquals(Labels.of("e", "f"), it.next());
            Assert.assertFalse(it.hasNext());
            Assert.assertFalse(it.hasNext()); // idempotent

            CoremonMetric[] expected = {
                new FileCoremonMetric(1, 2, Labels.of("a", "b"), 100, MetricType.DGAUGE),
                new FileCoremonMetric(3, 4, Labels.of("c", "d"), 200, MetricType.COUNTER),
                new FileCoremonMetric(5, 6, Labels.of("e", "f"), 300, MetricType.RATE),
            };

            // stream
            CoremonMetric[] metrics = a.stream().toArray(CoremonMetric[]::new);
            Assert.assertEquals(3, metrics.length);
            assertEquals(expected[0], metrics[0]);
            assertEquals(expected[1], metrics[1]);
            assertEquals(expected[2], metrics[2]);

            // get labels
            Assert.assertEquals(Labels.of("a", "b"), a.getLabels(0));
            Assert.assertEquals(Labels.of("c", "d"), a.getLabels(1));
            Assert.assertEquals(Labels.of("e", "f"), a.getLabels(2));

            // get
            assertEquals(expected[0], a.get(0));
            assertEquals(expected[1], a.get(1));
            assertEquals(expected[2], a.get(2));
        }
    }

    @Test
    public void addMetrics() {
        CoremonMetric[] expected = {
            new FileCoremonMetric(1, 2, Labels.of("a", "b"), 100, MetricType.DGAUGE),
            new FileCoremonMetric(3, 4, Labels.of("c", "d"), 200, MetricType.COUNTER),
            new FileCoremonMetric(5, 6, Labels.of("e", "f"), 300, MetricType.RATE),
        };

        try (CoremonMetricArray a = new CoremonMetricArray(0)) {
            a.add(expected[0]);
            a.add(expected[1]);
            a.add(expected[2]);

            // basic contract
            Assert.assertEquals(3, a.size());
            Assert.assertEquals(10, a.capacity());
            Assert.assertFalse(a.isEmpty());

            // labels iterator
            Iterator<Labels> it = a.labelsIterator();
            Assert.assertTrue(it.hasNext());
            Assert.assertEquals(Labels.of("a", "b"), it.next());
            Assert.assertTrue(it.hasNext());
            Assert.assertEquals(Labels.of("c", "d"), it.next());
            Assert.assertTrue(it.hasNext());
            Assert.assertEquals(Labels.of("e", "f"), it.next());
            Assert.assertFalse(it.hasNext());
            Assert.assertFalse(it.hasNext()); // idempotent

            // stream
            CoremonMetric[] metrics = a.stream().toArray(CoremonMetric[]::new);
            Assert.assertEquals(3, metrics.length);
            assertEquals(expected[0], metrics[0]);
            assertEquals(expected[1], metrics[1]);
            assertEquals(expected[2], metrics[2]);

            // get labels
            Assert.assertEquals(Labels.of("a", "b"), a.getLabels(0));
            Assert.assertEquals(Labels.of("c", "d"), a.getLabels(1));
            Assert.assertEquals(Labels.of("e", "f"), a.getLabels(2));

            // get
            assertEquals(expected[0], a.get(0));
            assertEquals(expected[1], a.get(1));
            assertEquals(expected[2], a.get(2));
        }
    }

    @Test
    public void addAll() {
        CoremonMetric[] metricsA = {
            new FileCoremonMetric(1, 2, Labels.of("a", "b"), 100, MetricType.DGAUGE),
            new FileCoremonMetric(3, 4, Labels.of("c", "d"), 200, MetricType.COUNTER),
            new FileCoremonMetric(5, 6, Labels.of("e", "f"), 300, MetricType.RATE),
        };

        CoremonMetric[] metricsB = {
            new FileCoremonMetric(7, 8, Labels.of("g", "h"), 400, MetricType.HIST),
            new FileCoremonMetric(9, 10, Labels.of("i", "g"), 500, MetricType.HIST_RATE),
            new FileCoremonMetric(11, 12, Labels.of("k", "l"), 600, MetricType.DSUMMARY),
            new FileCoremonMetric(13, 14, Labels.of("m", "n"), 700, MetricType.ISUMMARY),
        };

        try (CoremonMetricArray a = new CoremonMetricArray(metricsA);
             CoremonMetricArray b = new CoremonMetricArray(metricsB))
        {
            a.addAll(b);

            // basic contract
            Assert.assertEquals(7, a.size());
            Assert.assertEquals(7, a.capacity());
            Assert.assertFalse(a.isEmpty());

            // labels iterator
            Iterator<Labels> it = a.labelsIterator();
            Assert.assertTrue(it.hasNext());
            Assert.assertEquals(Labels.of("a", "b"), it.next());
            Assert.assertTrue(it.hasNext());
            Assert.assertEquals(Labels.of("c", "d"), it.next());
            Assert.assertTrue(it.hasNext());
            Assert.assertEquals(Labels.of("e", "f"), it.next());
            Assert.assertTrue(it.hasNext());
            Assert.assertEquals(Labels.of("g", "h"), it.next());
            Assert.assertTrue(it.hasNext());
            Assert.assertEquals(Labels.of("i", "g"), it.next());
            Assert.assertTrue(it.hasNext());
            Assert.assertEquals(Labels.of("k", "l"), it.next());
            Assert.assertTrue(it.hasNext());
            Assert.assertEquals(Labels.of("m", "n"), it.next());
            Assert.assertFalse(it.hasNext());
            Assert.assertFalse(it.hasNext()); // idempotent

            // stream
            CoremonMetric[] metrics = a.stream().toArray(CoremonMetric[]::new);
            Assert.assertEquals(7, metrics.length);
            for (int i = 0; i < 3; i++) {
                assertEquals(metricsA[i], metrics[i]);
            }
            for (int i = 0; i < 4; i++) {
                assertEquals(metricsB[i], metrics[i + 3]);
            }

            // get labels
            Assert.assertEquals(Labels.of("a", "b"), a.getLabels(0));
            Assert.assertEquals(Labels.of("c", "d"), a.getLabels(1));
            Assert.assertEquals(Labels.of("e", "f"), a.getLabels(2));
            Assert.assertEquals(Labels.of("g", "h"), a.getLabels(3));
            Assert.assertEquals(Labels.of("i", "g"), a.getLabels(4));
            Assert.assertEquals(Labels.of("k", "l"), a.getLabels(5));
            Assert.assertEquals(Labels.of("m", "n"), a.getLabels(6));

            // get
            assertEquals(metricsA[0], a.get(0));
            assertEquals(metricsA[1], a.get(1));
            assertEquals(metricsA[2], a.get(2));
            assertEquals(metricsB[0], a.get(3));
            assertEquals(metricsB[1], a.get(4));
            assertEquals(metricsB[2], a.get(5));
            assertEquals(metricsB[3], a.get(6));
        }
    }

    @Test
    public void shrinkToFit() {
        CoremonMetricArray a = new CoremonMetricArray(0);
        a.add(1, 2, Labels.of("a", "b"), 100, MetricType.DGAUGE);
        a.add(2, 3, Labels.of("c", "d"), 200, MetricType.COUNTER);
        a.add(4, 5, Labels.of("e", "f"), 300, MetricType.RATE);

        Assert.assertEquals(3, a.size());
        Assert.assertEquals(10, a.capacity());

        a.shrinkToFit();

        Assert.assertEquals(3, a.capacity());
    }

    @Test
    public void multipleBuffers() {
        CoremonMetric[] metricsA = {
            new FileCoremonMetric(1, 2, Labels.of("a", "b"), 100, MetricType.DGAUGE),
            new FileCoremonMetric(3, 4, Labels.of("c", "d"), 200, MetricType.COUNTER),
            new FileCoremonMetric(5, 6, Labels.of("e", "f"), 300, MetricType.RATE),
        };

        CoremonMetric[] metricsB = {
            new FileCoremonMetric(7, 8, Labels.of("g", "h"), 400, MetricType.HIST),
            new FileCoremonMetric(9, 10, Labels.of("i", "g"), 500, MetricType.HIST_RATE),
            new FileCoremonMetric(11, 12, Labels.of("k", "l"), 600, MetricType.DSUMMARY),
            new FileCoremonMetric(13, 14, Labels.of("m", "n"), 700, MetricType.ISUMMARY),
        };

        CoremonMetricArray a = new CoremonMetricArray(metricsA.length, 5);
        for (CoremonMetric s : metricsA) {
            a.add(s);
        }
        CoremonMetricArray b = new CoremonMetricArray(metricsB);
        try (a; b) {
            a.addAll(b);

            // basic contract
            Assert.assertEquals(7, a.size());
            Assert.assertEquals(7, a.capacity());
            Assert.assertFalse(a.isEmpty());

            // labels iterator
            Iterator<Labels> it = a.labelsIterator();
            Assert.assertTrue(it.hasNext());
            Assert.assertEquals(Labels.of("a", "b"), it.next());
            Assert.assertTrue(it.hasNext());
            Assert.assertEquals(Labels.of("c", "d"), it.next());
            Assert.assertTrue(it.hasNext());
            Assert.assertEquals(Labels.of("e", "f"), it.next());
            Assert.assertTrue(it.hasNext());
            Assert.assertEquals(Labels.of("g", "h"), it.next());
            Assert.assertTrue(it.hasNext());
            Assert.assertEquals(Labels.of("i", "g"), it.next());
            Assert.assertTrue(it.hasNext());
            Assert.assertEquals(Labels.of("k", "l"), it.next());
            Assert.assertTrue(it.hasNext());
            Assert.assertEquals(Labels.of("m", "n"), it.next());
            Assert.assertFalse(it.hasNext());
            Assert.assertFalse(it.hasNext()); // idempotent

            // stream
            CoremonMetric[] metrics = a.stream().toArray(CoremonMetric[]::new);
            Assert.assertEquals(7, metrics.length);
            for (int i = 0; i < 3; i++) {
                assertEquals(metricsA[i], metrics[i]);
            }
            for (int i = 0; i < 4; i++) {
                assertEquals(metricsB[i], metrics[i + 3]);
            }

            // get labels
            Assert.assertEquals(Labels.of("a", "b"), a.getLabels(0));
            Assert.assertEquals(Labels.of("c", "d"), a.getLabels(1));
            Assert.assertEquals(Labels.of("e", "f"), a.getLabels(2));
            Assert.assertEquals(Labels.of("g", "h"), a.getLabels(3));
            Assert.assertEquals(Labels.of("i", "g"), a.getLabels(4));
            Assert.assertEquals(Labels.of("k", "l"), a.getLabels(5));
            Assert.assertEquals(Labels.of("m", "n"), a.getLabels(6));

            // get
            assertEquals(metricsA[0], a.get(0));
            assertEquals(metricsA[1], a.get(1));
            assertEquals(metricsA[2], a.get(2));
            assertEquals(metricsB[0], a.get(3));
            assertEquals(metricsB[1], a.get(4));
            assertEquals(metricsB[2], a.get(5));
            assertEquals(metricsB[3], a.get(6));
        }
    }

    @Test
    public void addAllInSplitedBuffers() {
        final int chunkSize = 100;
        final int numberOfChunks = 10;
        final int maxBufferCapacity = chunkSize * numberOfChunks / 2;

        try (CoremonMetricArray chunk = new CoremonMetricArray(chunkSize)) {
            int createdAtSeconds = InstantUtils.currentTimeSeconds();
            MetricType[] types = MetricType.values();

            for (int i = 1; i <= chunkSize; i++) {
                chunk.add(1, i, Labels.of("s", Integer.toString(i)), createdAtSeconds, types[i % types.length]);
            }

            try (CoremonMetricArray a = new CoremonMetricArray(chunkSize * numberOfChunks, maxBufferCapacity)) {
                for (int i = 0; i < numberOfChunks; i++) {
                    a.addAll(chunk);
                }

                Assert.assertEquals(chunkSize * numberOfChunks, a.size());
                for (int i = 0; i < a.size(); i++) {
                    int id = (i % chunkSize) + 1;
                    Assert.assertEquals(Labels.of("s", Integer.toString(id)), a.getLabels(i));
                    Assert.assertEquals(id, a.getLocalId(i));
                    Assert.assertEquals(createdAtSeconds, a.getCreatedAtSeconds(i));
                    Assert.assertEquals(types[id % types.length], a.getType(i));
                }
            }
        }
    }
}
