package ru.yandex.solomon.coremon.meta;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.Iterators;
import com.google.common.collect.Streams;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.coremon.meta.file.FileMetricsCollection;
import ru.yandex.solomon.labels.query.Selectors;


/**
 * @author Sergey Polovko
 */
public class FileMetricsCollectionTest {

    private static final int[] LEVELS_SIZES = { 1, 2, 10 };
    private ExecutorService executor;

    @Before
    public void setUp() {
        executor = Executors.newFixedThreadPool(2);
    }

    @After
    public void tearDown() throws Exception {
        executor.shutdown();
        Assert.assertTrue(executor.awaitTermination(5, TimeUnit.SECONDS));
    }

    @Test
    public void getOrNull() {
        FileMetricsCollection collection = new FileMetricsCollection(
            "test", executor, LEVELS_SIZES, CoremonMetricHelper.metrics(
                Labels.of("s", "a"), Labels.of("s", "b", "t", "c"
            )));

        Assert.assertEquals(2, collection.size());

        Assert.assertNotNull(collection.getOrNull(Labels.of("s", "a")));
        Assert.assertNotNull(collection.getOrNull(Labels.of("s", "b", "t", "c")));

        Assert.assertNull(collection.getOrNull(Labels.of("s", "b")));
        Assert.assertNull(collection.getOrNull(Labels.of("s", "c")));
    }

    @Test
    public void has() {
        FileMetricsCollection collection = new FileMetricsCollection(
            "test", executor, LEVELS_SIZES, CoremonMetricHelper.metrics(
            Labels.of("s", "a"), Labels.of("s", "b", "t", "c"
            )));

        Assert.assertEquals(2, collection.size());

        Assert.assertTrue(collection.has(Labels.of("s", "a")));
        Assert.assertTrue(collection.has(Labels.of("s", "b", "t", "c")));

        Assert.assertFalse(collection.has(Labels.of("s", "b")));
        Assert.assertFalse(collection.has(Labels.of("s", "c")));
    }

    @Test
    public void putAll() {
        AwaitableFileMetricsCollection collection = new AwaitableFileMetricsCollection(
            executor, LEVELS_SIZES, CoremonMetricHelper.metrics(Labels.of("s", "a"), Labels.of("s", "b", "t", "c")));
        Assert.assertEquals(2, collection.size());
        Assert.assertEquals(0, collection.levelSize(0));
        Assert.assertEquals(0, collection.levelSize(1));
        Assert.assertEquals(2, collection.levelSize(2));

        try (CoremonMetricArray metrics = CoremonMetricHelper.metrics(Labels.of("s", "d"), Labels.of("s", "e"))) {
            collection.putAll(metrics);
            collection.awaitActDone();
        }

        Assert.assertEquals(4, collection.size());
        Assert.assertEquals(0, collection.levelSize(0));
        Assert.assertEquals(2, collection.levelSize(1));
        Assert.assertEquals(2, collection.levelSize(2));

        Assert.assertNotNull(collection.getOrNull(Labels.of("s", "d")));
        Assert.assertNotNull(collection.getOrNull(Labels.of("s", "e")));

        collection.putAll(CoremonMetricHelper.metrics(
            Labels.of("s", "e"),
            Labels.of("s", "d"),
            Labels.of("s", "f")));
        collection.awaitActDone();

        Assert.assertEquals(5, collection.size());
        Assert.assertEquals(1, collection.levelSize(0));
        Assert.assertEquals(2, collection.levelSize(1));
        Assert.assertEquals(2, collection.levelSize(2));

        Assert.assertNotNull(collection.getOrNull(Labels.of("s", "f")));

        collection.putAll(CoremonMetricHelper.metrics(Labels.of("s", "g")));
        collection.awaitActDone();

        Assert.assertEquals(6, collection.size());
        Assert.assertEquals(0, collection.levelSize(0));
        Assert.assertEquals(4, collection.levelSize(1));
        Assert.assertEquals(2, collection.levelSize(2));

        Assert.assertNotNull(collection.getOrNull(Labels.of("s", "g")));

        collection.putAll(CoremonMetricHelper.metrics(Labels.of("s", "h")));
        collection.awaitActDone();

        Assert.assertEquals(7, collection.size());
        Assert.assertEquals(1, collection.levelSize(0));
        Assert.assertEquals(0, collection.levelSize(1));
        Assert.assertEquals(6, collection.levelSize(2));

        Assert.assertNotNull(collection.getOrNull(Labels.of("s", "h")));
    }

    @Test
    public void removeAll() {
        // fill all levels
        AwaitableFileMetricsCollection collection = new AwaitableFileMetricsCollection(
            executor, LEVELS_SIZES,
            CoremonMetricHelper.metrics(
                Labels.of("s", "a"),
                Labels.of("s", "b", "t", "c"),
                Labels.of("s", "c"),
                Labels.of("s", "d"),
                Labels.of("s", "e"),
                Labels.of("s", "f")));

        collection.putAll(CoremonMetricHelper.metrics(Labels.of("s", "g"), Labels.of("s", "h")));
        collection.awaitActDone();

        collection.putAll(CoremonMetricHelper.metrics(Labels.of("s", "i")));
        collection.awaitActDone();

        Assert.assertEquals(9, collection.size());
        Assert.assertEquals(1, collection.levelSize(0));
        Assert.assertEquals(2, collection.levelSize(1));
        Assert.assertEquals(6, collection.levelSize(2));

        Labels l1 = Labels.of("s", "i");
        Labels l2 = Labels.of("s", "h");
        Labels l3 = Labels.of("s", "d");

        Assert.assertNotNull(collection.getOrNull(l1));
        Assert.assertNotNull(collection.getOrNull(l2));
        Assert.assertNotNull(collection.getOrNull(l3));

        // remove one metric from each level (first, last and middle)
        collection.removeAll(Arrays.asList(l1, l2, l3));
        collection.awaitActDone();

        Assert.assertEquals(6, collection.size());
        Assert.assertEquals(0, collection.levelSize(0));
        Assert.assertEquals(1, collection.levelSize(1));
        Assert.assertEquals(5, collection.levelSize(2));

        Assert.assertNull(collection.getOrNull(l1));
        Assert.assertNull(collection.getOrNull(l2));
        Assert.assertNull(collection.getOrNull(l3));
    }

    @Test
    public void searchWithCount() {
        // fill all levels
        AwaitableFileMetricsCollection collection = new AwaitableFileMetricsCollection(
            executor, LEVELS_SIZES,
            CoremonMetricHelper.metrics(
                Labels.of("s", "a"),
                Labels.of("s", "b", "t", "c"),
                Labels.of("s", "c"),
                Labels.of("s", "d"),
                Labels.of("s", "e"),
                Labels.of("s", "f")));

        collection.putAll(CoremonMetricHelper.metrics(Labels.of("s", "g"), Labels.of("s", "h")));
        collection.awaitActDone();

        collection.putAll(CoremonMetricHelper.metrics(Labels.of("s", "i")));
        collection.awaitActDone();

        Assert.assertEquals(9, collection.size());
        Assert.assertEquals(1, collection.levelSize(0));
        Assert.assertEquals(2, collection.levelSize(1));
        Assert.assertEquals(6, collection.levelSize(2));

        // search one metric from each level
        List<FileCoremonMetric> result = new ArrayList<>();
        int totalMetrics = collection.searchMetrics(
            Selectors.parse("s=i|a|h"),
            s -> result.add(new FileCoremonMetric(s)));
        Assert.assertEquals(3, result.size());
        Assert.assertEquals(3, totalMetrics);

        Map<Labels, CoremonMetric> metrics = result.stream()
            .collect(Collectors.toMap(CoremonMetric::getLabels, Function.identity()));
        Assert.assertEquals(3, metrics.size());
        Assert.assertNotNull(metrics.get(Labels.of("s", "i")));
        Assert.assertNotNull(metrics.get(Labels.of("s", "a")));
        Assert.assertNotNull(metrics.get(Labels.of("s", "h")));
    }

    @Test
    public void searchIterator() {
        // fill all levels
        var collection = new AwaitableFileMetricsCollection(
            executor, LEVELS_SIZES,
            CoremonMetricHelper.metrics(
                Labels.of("s", "a"),
                Labels.of("s", "b", "t", "c"),
                Labels.of("s", "c"),
                Labels.of("s", "d"),
                Labels.of("s", "e"),
                Labels.of("s", "f")));

        collection.putAll(CoremonMetricHelper.metrics(Labels.of("s", "g"), Labels.of("s", "h")));
        collection.awaitActDone();

        collection.putAll(CoremonMetricHelper.metrics(Labels.of("s", "i")));
        collection.awaitActDone();

        Assert.assertEquals(9, collection.size());
        Assert.assertEquals(1, collection.levelSize(0));
        Assert.assertEquals(2, collection.levelSize(1));
        Assert.assertEquals(6, collection.levelSize(2));

        // search one metric from each level
        var result = collection.searchIterator(Selectors.parse("s=i|a|h"));
        Assert.assertEquals(3, result.count());

        Map<Labels, CoremonMetric> metrics = Streams.stream(result.iterator())
            .collect(Collectors.toMap(CoremonMetric::getLabels, Function.identity()));
        Assert.assertEquals(3, metrics.size());
        Assert.assertNotNull(metrics.get(Labels.of("s", "i")));
        Assert.assertNotNull(metrics.get(Labels.of("s", "a")));
        Assert.assertNotNull(metrics.get(Labels.of("s", "h")));
    }

    @Test
    public void emptySearchIterator() {
        var collection = new AwaitableFileMetricsCollection(
            executor, LEVELS_SIZES,
            CoremonMetricHelper.metrics(
                Labels.of("s", "a"),
                Labels.of("s", "b", "t", "c"),
                Labels.of("s", "c"),
                Labels.of("s", "d"),
                Labels.of("s", "e"),
                Labels.of("s", "f")));

        var result = collection.searchIterator(Selectors.parse("foo=bar"));

        Assert.assertEquals(0, result.count());
        Assert.assertEquals(0, Iterators.size(result.iterator()));
    }

    @Test
    public void emptySearchIteratorWhenStopped() {
        var collection = new AwaitableFileMetricsCollection(
            executor, LEVELS_SIZES,
            CoremonMetricHelper.metrics(
                Labels.of("s", "a"),
                Labels.of("s", "b", "t", "c"),
                Labels.of("s", "c"),
                Labels.of("s", "d"),
                Labels.of("s", "e"),
                Labels.of("s", "f")));

        collection.close();
        collection.awaitActDone();

        var result = collection.searchIterator(Selectors.parse("s=a"));

        Assert.assertEquals(0, result.count());
        Assert.assertEquals(0, Iterators.size(result.iterator()));
    }

    @Test
    public void iterable() {
        // empty
        {
            FileMetricsCollection collection = new FileMetricsCollection(
                "test", executor, LEVELS_SIZES, CoremonMetricHelper.metrics());

            try (var it = collection.iterator()) {
                Assert.assertFalse(it.hasNext());
                Assert.assertFalse(it.hasNext()); // is idempotent
            }
        }

        // initially filled
        {
            FileMetricsCollection collection = new FileMetricsCollection(
                "test", executor, LEVELS_SIZES, CoremonMetricHelper.metrics(
                    Labels.of("a", "1"),
                    Labels.of("b", "2"),
                    Labels.of("c", "3")
                ));

            try (var it = collection.iterator()) {
                Assert.assertTrue(it.hasNext());
                Assert.assertTrue(it.hasNext()); // is idempotent
                Assert.assertEquals(it.next().getLabels(), Labels.of("a", "1"));
                Assert.assertEquals(it.next().getLabels(), Labels.of("b", "2"));
                Assert.assertEquals(it.next().getLabels(), Labels.of("c", "3"));
                Assert.assertFalse(it.hasNext());
                Assert.assertFalse(it.hasNext()); // is idempotent
            }
        }

        // filled at runtime
        {
            AwaitableFileMetricsCollection collection = new AwaitableFileMetricsCollection(
                executor, LEVELS_SIZES, CoremonMetricHelper.metrics());

            Set<Labels> expected = new HashSet<>();

            for (int i = 0; i < 1000; i++) {
                Labels labels = Labels.of("a", Integer.toString(i));
                Assert.assertTrue(expected.add(labels));
                collection.putAll(CoremonMetricHelper.metrics(labels));
                collection.awaitActDone();

                Set<Labels> actual = new HashSet<>();
                collection.forEach(s -> {
                    actual.add(s.getLabels());
                });

                Assert.assertEquals(expected, actual);
            }
        }
    }
}
