package ru.yandex.solomon.coremon.meta.db;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;
import java.util.stream.IntStream;

import javax.annotation.Nullable;

import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.LabelAllocator;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.labels.string.StringLabelAllocator;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.CoremonMetricArray;
import ru.yandex.solomon.coremon.meta.CoremonMetricHelper;
import ru.yandex.solomon.coremon.meta.FileCoremonMetric;
import ru.yandex.solomon.util.time.InstantUtils;

import static java.util.Collections.shuffle;
import static java.util.Comparator.comparing;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static org.apache.commons.lang3.RandomStringUtils.randomAlphanumeric;
import static org.hamcrest.CoreMatchers.containsString;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;
import static ru.yandex.misc.concurrent.CompletableFutures.join;
import static ru.yandex.misc.concurrent.CompletableFutures.unwrapCompletionException;

/**
 * @author Stanislav Kashirin
 */
public abstract class DeletedMetricsDaoTest {

    private static final Logger logger = LoggerFactory.getLogger(DeletedMetricsDaoTest.class);

    private static final LabelAllocator LABEL_ALLOCATOR = StringLabelAllocator.SELF;
    private static final AtomicInteger numIdSeq = new AtomicInteger(0);

    private final int numId = numIdSeq.incrementAndGet();

    protected abstract DeletedMetricsDao getDao();

    @Test
    public void readWhenOperationHasNoMetrics() {
        // arrange
        var operationId = operationId();
        var otherOperationId = operationId();
        assertTrue(readMetrics(operationId, numId).isEmpty());
        assertTrue(readMetrics(otherOperationId, numId).isEmpty());

        var m1 = new FileCoremonMetric(
            14,
            10,
            Labels.of("x", "y", "a", "b"),
            InstantUtils.parseToSeconds("2015-10-12T13:14:10Z"),
            MetricType.RATE);
        var m2 = new FileCoremonMetric(
            13,
            12,
            Labels.of("x", "y", "a", "c"),
            InstantUtils.parseToSeconds("2015-10-12T13:14:12Z"),
            MetricType.DGAUGE);

        writeMetrics(otherOperationId, numId, m1, m2);

        // act
        var read = readMetricsOrderByLocalId(operationId, numId);
        var count = countMetrics(operationId, numId);

        // assert
        assertMetricsEqual(List.of(), read);
        assertEquals(read.size(), count);
    }

    @Test
    public void readWhenOperationHasSinglePageOfMetrics() {
        // arrange
        var operationId = operationId();
        assertTrue(readMetrics(operationId, numId).isEmpty());

        var m1 = new FileCoremonMetric(
            14,
            10,
            Labels.of("x", "y", "a", "b"),
            InstantUtils.parseToSeconds("2015-10-12T13:14:10Z"),
            MetricType.RATE);
        var m2 = new FileCoremonMetric(
            13,
            12,
            Labels.of("x", "y", "a", "c"),
            InstantUtils.parseToSeconds("2015-10-12T13:14:12Z"),
            MetricType.DGAUGE);

        writeMetrics(operationId, numId, m1, m2);

        // act
        var read = readMetricsOrderByLocalId(operationId, numId);
        var count = countMetrics(operationId, numId);

        // assert
        assertMetricsEqual(List.of(m1, m2), read);
        assertEquals(read.size(), count);
    }

    @Test
    public void readWhenOperationSpansMultipleShards() {
        // arrange
        var operationId = operationId();
        var numId1 = numId;
        var numId2 = numId1 + 1000;

        assertTrue(readMetrics(operationId, numId1).isEmpty());
        assertTrue(readMetrics(operationId, numId2).isEmpty());

        var m1 = new FileCoremonMetric(
            10,
            14,
            Labels.of("x", "y", "a", "b"),
            InstantUtils.parseToSeconds("2015-10-12T13:14:10Z"),
            MetricType.RATE);

        var m2 = new FileCoremonMetric(
            13,
            -12,
            Labels.of("x", "y", "a", "c"),
            InstantUtils.parseToSeconds("2015-10-12T13:14:12Z"),
            MetricType.DGAUGE);

        var m3 = new FileCoremonMetric(
            177,
            12342348,
            Labels.of("x", "y", "a", "d"),
            InstantUtils.parseToSeconds("2015-10-12T13:14:14Z"),
            MetricType.COUNTER);

        writeMetrics(operationId, numId1, m1, m2);
        writeMetrics(operationId, numId2, m3);

        // act
        var shard1 = readMetrics(operationId, numId1);
        var shard2 = readMetrics(operationId, numId2);
        var count1 = countMetrics(operationId, numId1);
        var count2 = countMetrics(operationId, numId2);

        // assert
        assertMetricsEqual(List.of(m1, m2), shard1);
        assertMetricsEqual(List.of(m3), shard2);
        assertEquals(shard1.size(), count1);
        assertEquals(shard2.size(), count2);
    }

    @Test
    public void readWhenMultipleOperationsWithinSingleShard() {
        // arrange
        var operationId1 = operationId();
        var operationId2 = operationId();
        assertTrue(readMetrics(operationId1, numId).isEmpty());
        assertTrue(readMetrics(operationId2, numId).isEmpty());

        var m1 = new FileCoremonMetric(
            10,
            14,
            Labels.of("x", "y", "a", "b"),
            InstantUtils.parseToSeconds("2015-10-12T13:14:10Z"),
            MetricType.RATE);

        var m2 = new FileCoremonMetric(
            13,
            -12,
            Labels.of("x", "y", "a", "c"),
            InstantUtils.parseToSeconds("2015-10-12T13:14:12Z"),
            MetricType.DGAUGE);

        var m3 = new FileCoremonMetric(
            177,
            12342348,
            Labels.of("x", "y", "a", "d"),
            InstantUtils.parseToSeconds("2015-10-12T13:14:14Z"),
            MetricType.COUNTER);

        writeMetrics(operationId1, numId, m1, m2);
        writeMetrics(operationId2, numId, m3);

        // act
        var operation1 = readMetrics(operationId1, numId);
        var operation2 = readMetrics(operationId2, numId);
        var count1 = countMetrics(operationId1, numId);
        var count2 = countMetrics(operationId2, numId);

        // assert
        assertMetricsEqual(List.of(m1, m2), operation1);
        assertMetricsEqual(List.of(m3), operation2);
        assertEquals(operation1.size(), count1);
        assertEquals(operation2.size(), count2);
    }

    @Test
    public void readWhenOperationHasMultiplePagesOfMetrics() {
        // arrange
        var operationId = operationId();
        var baseSeconds = InstantUtils.parseToSeconds("2015-10-12T13:14:10Z");
        var labelNames = IntStream.range(0, 2)
            .mapToObj(x -> "label_" + x)
            .collect(toList());

        var metrics = IntStream.range(0, 3750)
            .<CoremonMetric>mapToObj(i -> {
                var value = String.valueOf(i);
                var labels = labelNames.stream()
                    .collect(collectingAndThen(toMap(identity(), any -> value), Labels::of));

                return new FileCoremonMetric(13, i + 1, labels, baseSeconds + i, MetricType.RATE);
            })
            .collect(toList());
        shuffle(metrics);

        for (var chunk : Lists.partition(metrics, 500)) {
            writeMetrics(operationId, numId, chunk);
        }

        // act
        Labels lastKey = null;
        List<CoremonMetric> page;
        var pages = new ArrayList<List<CoremonMetric>>();
        while (!(page = readMetrics(operationId, numId, 1000, lastKey)).isEmpty()) {
            pages.add(page);
            lastKey = Iterables.getLast(page).getLabels();
        }

        var count = countMetrics(operationId, numId);

        // assert
        assertNotEquals(0, pages.size());
        assertNotEquals(1, pages.size());

        var expected = metrics.stream()
            .sorted(CoremonMetricHelper::compare)
            .collect(toList());

        var actual = pages.stream()
            .flatMap(Collection::stream)
            .sorted(CoremonMetricHelper::compare)
            .collect(toList());

        assertMetricsEqual(expected, actual);
        assertEquals(actual.size(), count);
    }

    @Test
    public void writeWhenMetricsOverlap() {
        // arrange
        var operationId = operationId();
        var baseSeconds = InstantUtils.parseToSeconds("2015-10-12T13:14:10Z");
        var labelNames = IntStream.range(0, 2)
            .mapToObj(x -> "label_" + x)
            .collect(toList());

        var metrics = IntStream.range(0, 12000)
            .<CoremonMetric>mapToObj(i -> {
                var value = String.valueOf(i);
                var labels = labelNames.stream()
                    .collect(collectingAndThen(toMap(identity(), any -> value), Labels::of));

                return new FileCoremonMetric(13, i + 1, labels, baseSeconds + i, MetricType.RATE);
            })
            .collect(toList());
        shuffle(metrics);

        var firstMetrics = Lists.partition(metrics, 6000).get(0);
        var overlappingMetrics = Lists.partition(metrics, 4000).get(1);
        var lastMetrics = Lists.partition(metrics, 4000).get(2);

        // act
        writeMetrics(operationId, numId, firstMetrics);
        writeMetrics(operationId, numId, overlappingMetrics);
        writeMetrics(operationId, numId, lastMetrics);

        Labels lastKey = null;
        List<CoremonMetric> page;
        var pages = new ArrayList<List<CoremonMetric>>();
        while (!(page = readMetrics(operationId, numId, 1000, lastKey)).isEmpty()) {
            pages.add(page);
            lastKey = Iterables.getLast(page).getLabels();
        }

        var count = countMetrics(operationId, numId);

        // assert
        var expected = metrics.stream()
            .sorted(CoremonMetricHelper::compare)
            .collect(toList());

        var actual = pages.stream()
            .flatMap(Collection::stream)
            .sorted(CoremonMetricHelper::compare)
            .collect(toList());

        assertMetricsEqual(expected, actual);
        assertEquals(actual.size(), count);
    }

    @Test
    public void useLargeBuffers() {
        // arrange
        final var large = 10000;

        var operationId = operationId();
        var baseSeconds = InstantUtils.parseToSeconds("2021-12-06T22:23:33Z");
        var labelNames = IntStream.range(0, Labels.MAX_LABELS_COUNT)
            .mapToObj(x -> "label_" + x + "X".repeat(25))
            .collect(toList());

        var metrics = IntStream.range(0, large)
            .<CoremonMetric>mapToObj(i -> {
                var value = String.valueOf(i);
                var labels = labelNames.stream()
                    .collect(collectingAndThen(toMap(identity(), any -> value), Labels::of));

                return new FileCoremonMetric(13, i + 1, labels, baseSeconds + i, MetricType.RATE);
            })
            .collect(toList());
        shuffle(metrics);

        // act
        writeMetrics(operationId, numId, metrics);
        var read = readMetrics(operationId, numId, large, null);
        var count = countMetrics(operationId, numId);

        // assert
        var expected = metrics.stream()
            .sorted(CoremonMetricHelper::compare)
            .collect(toList());

        var actual = read.stream()
            .sorted(CoremonMetricHelper::compare)
            .collect(toList());

        assertMetricsEqual(expected, actual);
        assertEquals(large, count);
    }

    @Test
    public void deleteWhenOperationHasNoMetrics() {
        // arrange
        var operationId = operationId();
        var otherOperationId = operationId();
        assertTrue(readMetrics(operationId, numId).isEmpty());
        assertTrue(readMetrics(otherOperationId, numId).isEmpty());

        var m1 = new FileCoremonMetric(
            14,
            10,
            Labels.of("x", "y", "a", "b"),
            InstantUtils.parseToSeconds("2015-10-12T13:14:10Z"),
            MetricType.RATE);
        var m2 = new FileCoremonMetric(
            13,
            12,
            Labels.of("x", "y", "a", "c"),
            InstantUtils.parseToSeconds("2015-10-12T13:14:12Z"),
            MetricType.DGAUGE);

        writeMetrics(otherOperationId, numId, m1, m2);

        // act
        deleteMetrics(operationId, numId, List.of(m1.getLabels(), m2.getLabels()));
        var count = countMetrics(operationId, numId);

        var otherRead = readMetricsOrderByLocalId(otherOperationId, numId);
        var otherCount = countMetrics(otherOperationId, numId);

        // assert
        assertEquals(0, count);

        assertMetricsEqual(List.of(m1, m2), otherRead);
        assertEquals(otherRead.size(), otherCount);
    }

    @Test
    public void deleteSomeOfOperationMetrics() {
        // arrange
        var operationId = operationId();
        var m1 = new FileCoremonMetric(
            13,
            10,
            Labels.of("x", "y", "a", "b"),
            InstantUtils.parseToSeconds("2015-10-12T13:14:10Z"),
            MetricType.RATE);
        var m2 = new FileCoremonMetric(
            14,
            11,
            Labels.of("x", "y", "a", "c"),
            InstantUtils.parseToSeconds("2016-11-12T13:14:11Z"),
            MetricType.RATE);
        var m3 = new FileCoremonMetric(
            14,
            12,
            Labels.of("x", "y", "a", "d"),
            InstantUtils.parseToSeconds("2016-11-12T13:14:12Z"),
            MetricType.RATE);

        writeMetrics(operationId, numId, m1, m2, m3);

        // act
        deleteMetrics(operationId, numId, List.of(m1.getLabels(), m3.getLabels()));

        // assert
        var actual = readMetrics(operationId, numId);
        var count = countMetrics(operationId, numId);
        assertMetricsEqual(List.of(m2), actual);
        assertEquals(actual.size(), count);
    }

    @Test
    public void deleteAllOperationMetrics() {
        // arrange
        var operationId = operationId();
        var baseSeconds = InstantUtils.parseToSeconds("2015-10-12T13:14:10Z");
        var labelNames = IntStream.range(0, 2)
            .mapToObj(x -> "label_" + x)
            .collect(toList());

        var metrics = IntStream.range(0, 3000)
            .<CoremonMetric>mapToObj(i -> {
                var value = String.valueOf(i);
                var labels = labelNames.stream()
                    .collect(collectingAndThen(toMap(identity(), any -> value), Labels::of));

                return new FileCoremonMetric(13, i + 1, labels, baseSeconds + i, MetricType.RATE);
            })
            .collect(toList());
        shuffle(metrics);

        for (var chunk : Lists.partition(metrics, 500)) {
            writeMetrics(operationId, numId, chunk);
        }

        var allLabels = metrics.stream().map(CoremonMetric::getLabels).collect(toList());

        // act
        deleteMetrics(operationId, numId, allLabels);

        // assert
        assertMetricsEqual(List.of(), readMetrics(operationId, numId));
        assertEquals(0, countMetrics(operationId, numId));
    }

    @Test
    public void deleteWhenOperationSpansMultipleShards() {
        // arrange
        var operationId = operationId();
        var numId1 = numId;
        var numId2 = numId1 + 1000;

        assertTrue(readMetrics(operationId, numId1).isEmpty());
        assertTrue(readMetrics(operationId, numId2).isEmpty());

        var m1 = new FileCoremonMetric(
            10,
            14,
            Labels.of("x", "y", "a", "b"),
            InstantUtils.parseToSeconds("2015-10-12T13:14:10Z"),
            MetricType.RATE);

        var m2 = new FileCoremonMetric(
            13,
            -12,
            Labels.of("x", "y", "a", "c"),
            InstantUtils.parseToSeconds("2015-10-12T13:14:12Z"),
            MetricType.DGAUGE);

        var m3 = new FileCoremonMetric(
            177,
            12342348,
            Labels.of("x", "y", "a", "d"),
            InstantUtils.parseToSeconds("2015-10-12T13:14:14Z"),
            MetricType.COUNTER);

        writeMetrics(operationId, numId1, m1, m2);
        writeMetrics(operationId, numId2, m3);

        // act
        deleteMetrics(operationId, numId1, List.of(m2.getLabels()));
        var shard1AfterDelete1 = readMetrics(operationId, numId1);
        var shard2AfterDelete1 = readMetrics(operationId, numId2);
        var count1AfterDelete1 = countMetrics(operationId, numId1);
        var count2AfterDelete1 = countMetrics(operationId, numId2);

        deleteMetrics(operationId, numId2, List.of(m3.getLabels()));
        var shard1AfterDelete2 = readMetrics(operationId, numId1);
        var shard2AfterDelete2 = readMetrics(operationId, numId2);
        var count1AfterDelete2 = countMetrics(operationId, numId1);
        var count2AfterDelete2 = countMetrics(operationId, numId2);

        // assert
        assertMetricsEqual(List.of(m1), shard1AfterDelete1);
        assertMetricsEqual(List.of(m3), shard2AfterDelete1);
        assertEquals(shard1AfterDelete1.size(), count1AfterDelete1);
        assertEquals(shard2AfterDelete1.size(), count2AfterDelete1);

        assertMetricsEqual(List.of(m1), shard1AfterDelete2);
        assertMetricsEqual(List.of(), shard2AfterDelete2);
        assertEquals(shard1AfterDelete2.size(), count1AfterDelete2);
        assertEquals(shard2AfterDelete2.size(), count2AfterDelete2);
    }

    @Test
    public void deleteWhenMultipleOperationsWithinSingleShard() {
        // arrange
        var operationId1 = operationId();
        var operationId2 = operationId();
        assertTrue(readMetrics(operationId1, numId).isEmpty());
        assertTrue(readMetrics(operationId2, numId).isEmpty());

        var m1 = new FileCoremonMetric(
            10,
            14,
            Labels.of("x", "y", "a", "b"),
            InstantUtils.parseToSeconds("2015-10-12T13:14:10Z"),
            MetricType.RATE);

        var m2 = new FileCoremonMetric(
            13,
            -12,
            Labels.of("s", "a", "m", "e"),
            InstantUtils.parseToSeconds("2015-10-12T13:14:12Z"),
            MetricType.DGAUGE);

        var m3 = new FileCoremonMetric(
            177,
            12342348,
            Labels.of("s", "a", "m", "e"),
            InstantUtils.parseToSeconds("2015-10-12T13:14:14Z"),
            MetricType.COUNTER);

        writeMetrics(operationId1, numId, m1, m2);
        writeMetrics(operationId2, numId, m3);

        // act
        deleteMetrics(operationId1, numId, List.of(m1.getLabels()));
        var operation1AfterDelete1 = readMetrics(operationId1, numId);
        var operation2AfterDelete1 = readMetrics(operationId2, numId);
        var count1AfterDelete1 = countMetrics(operationId1, numId);
        var count2AfterDelete1 = countMetrics(operationId2, numId);

        deleteMetrics(operationId2, numId, List.of(m3.getLabels()));
        var operation1AfterDelete2 = readMetrics(operationId1, numId);
        var operation2AfterDelete2 = readMetrics(operationId2, numId);
        var count1AfterDelete2 = countMetrics(operationId1, numId);
        var count2AfterDelete2 = countMetrics(operationId2, numId);

        // assert
        assertMetricsEqual(List.of(m2), operation1AfterDelete1);
        assertMetricsEqual(List.of(m3), operation2AfterDelete1);
        assertEquals(operation1AfterDelete1.size(), count1AfterDelete1);
        assertEquals(operation2AfterDelete1.size(), count2AfterDelete1);

        assertMetricsEqual(List.of(m2), operation1AfterDelete2);
        assertMetricsEqual(List.of(), operation2AfterDelete2);
        assertEquals(operation1AfterDelete2.size(), count1AfterDelete2);
        assertEquals(operation2AfterDelete2.size(), count2AfterDelete2);
    }

    @Test
    public void deleteBatchWhenOperationHasNoMetrics() {
        // arrange
        var operationId = operationId();
        var otherOperationId = operationId();
        assertTrue(readMetrics(operationId, numId).isEmpty());
        assertTrue(readMetrics(otherOperationId, numId).isEmpty());

        var m1 = new FileCoremonMetric(
            14,
            10,
            Labels.of("x", "y", "a", "b"),
            InstantUtils.parseToSeconds("2015-10-12T13:14:10Z"),
            MetricType.RATE);
        var m2 = new FileCoremonMetric(
            13,
            12,
            Labels.of("x", "y", "a", "c"),
            InstantUtils.parseToSeconds("2015-10-12T13:14:12Z"),
            MetricType.DGAUGE);

        writeMetrics(otherOperationId, numId, m1, m2);

        // act
        var deleted = deleteBatchMetrics(operationId, numId);
        var count = countMetrics(operationId, numId);

        var otherRead = readMetricsOrderByLocalId(otherOperationId, numId);
        var otherCount = countMetrics(otherOperationId, numId);

        // assert
        assertEquals(0, deleted);
        assertEquals(0, count);

        assertMetricsEqual(List.of(m1, m2), otherRead);
        assertEquals(otherRead.size(), otherCount);
    }

    @Test
    public void deleteBatchSomeOfOperationMetrics() {
        // arrange
        var operationId = operationId();
        var baseSeconds = InstantUtils.parseToSeconds("2015-10-12T13:14:10Z");
        var labelNames = IntStream.range(0, 2)
            .mapToObj(x -> "label_" + x)
            .collect(toList());

        var metrics = IntStream.range(0, 7777)
            .<CoremonMetric>mapToObj(i -> {
                var value = String.valueOf(i);
                var labels = labelNames.stream()
                    .collect(collectingAndThen(toMap(identity(), any -> value), Labels::of));

                return new FileCoremonMetric(13, i + 1, labels, baseSeconds + i, MetricType.RATE);
            })
            .collect(toList());
        shuffle(metrics);

        for (var chunk : Lists.partition(metrics, 500)) {
            writeMetrics(operationId, numId, chunk);
        }

        // act
        var deleted = deleteBatchMetrics(operationId, numId);
        var read = readMetrics(operationId, numId, 7777, null);
        var count = countMetrics(operationId, numId);

        // assert
        assertNotEquals(0, read.size());
        assertEquals(read.size(), count);
        assertEquals(metrics.size(), count + deleted);
    }

    @Test
    public void deleteBatchAllOperationMetrics() {
        // arrange
        var operationId = operationId();
        var baseSeconds = InstantUtils.parseToSeconds("2015-10-12T13:14:10Z");
        var labelNames = IntStream.range(0, 2)
            .mapToObj(x -> "label_" + x)
            .collect(toList());

        var metrics = IntStream.range(0, 7777)
            .<CoremonMetric>mapToObj(i -> {
                var value = String.valueOf(i);
                var labels = labelNames.stream()
                    .collect(collectingAndThen(toMap(identity(), any -> value), Labels::of));

                return new FileCoremonMetric(13, i + 1, labels, baseSeconds + i, MetricType.RATE);
            })
            .collect(toList());
        shuffle(metrics);

        for (var chunk : Lists.partition(metrics, 500)) {
            writeMetrics(operationId, numId, chunk);
        }

        // act
        long deleted;
        do {
            deleted = deleteBatchMetrics(operationId, numId);
        } while (deleted != 0);

        var read = readMetrics(operationId, numId, 7777, null);
        var count = countMetrics(operationId, numId);

        // assert
        assertEquals(List.of(), read);
        assertEquals(0, count);
    }

    @Test
    public void deleteBatchWhenOperationSpansMultipleShards() {
        // arrange
        var operationId = operationId();
        var numId1 = numId;
        var numId2 = numId1 + 1000;

        assertTrue(readMetrics(operationId, numId1).isEmpty());
        assertTrue(readMetrics(operationId, numId2).isEmpty());

        var m1 = new FileCoremonMetric(
            10,
            14,
            Labels.of("x", "y", "a", "b"),
            InstantUtils.parseToSeconds("2015-10-12T13:14:10Z"),
            MetricType.RATE);

        var m2 = new FileCoremonMetric(
            13,
            -12,
            Labels.of("x", "y", "a", "c"),
            InstantUtils.parseToSeconds("2015-10-12T13:14:12Z"),
            MetricType.DGAUGE);

        var m3 = new FileCoremonMetric(
            177,
            12342348,
            Labels.of("x", "y", "a", "d"),
            InstantUtils.parseToSeconds("2015-10-12T13:14:14Z"),
            MetricType.COUNTER);

        writeMetrics(operationId, numId1, m1, m2);
        writeMetrics(operationId, numId2, m3);

        // act
        deleteBatchMetrics(operationId, numId1);
        var shard1AfterDelete1 = readMetrics(operationId, numId1);
        var shard2AfterDelete1 = readMetrics(operationId, numId2);
        var count1AfterDelete1 = countMetrics(operationId, numId1);
        var count2AfterDelete1 = countMetrics(operationId, numId2);

        deleteBatchMetrics(operationId, numId2);
        var shard1AfterDelete2 = readMetrics(operationId, numId1);
        var shard2AfterDelete2 = readMetrics(operationId, numId2);
        var count1AfterDelete2 = countMetrics(operationId, numId1);
        var count2AfterDelete2 = countMetrics(operationId, numId2);

        // assert
        assertMetricsEqual(List.of(), shard1AfterDelete1);
        assertMetricsEqual(List.of(m3), shard2AfterDelete1);
        assertEquals(shard1AfterDelete1.size(), count1AfterDelete1);
        assertEquals(shard2AfterDelete1.size(), count2AfterDelete1);

        assertMetricsEqual(List.of(), shard1AfterDelete2);
        assertMetricsEqual(List.of(), shard2AfterDelete2);
        assertEquals(shard1AfterDelete2.size(), count1AfterDelete2);
        assertEquals(shard2AfterDelete2.size(), count2AfterDelete2);
    }

    @Test
    public void deleteBatchWhenMultipleOperationsWithinSingleShard() {
        // arrange
        var operationId1 = operationId();
        var operationId2 = operationId();
        assertTrue(readMetrics(operationId1, numId).isEmpty());
        assertTrue(readMetrics(operationId2, numId).isEmpty());

        var m1 = new FileCoremonMetric(
            10,
            14,
            Labels.of("x", "y", "a", "b"),
            InstantUtils.parseToSeconds("2015-10-12T13:14:10Z"),
            MetricType.RATE);

        var m2 = new FileCoremonMetric(
            13,
            -12,
            Labels.of("s", "a", "m", "e"),
            InstantUtils.parseToSeconds("2015-10-12T13:14:12Z"),
            MetricType.DGAUGE);

        var m3 = new FileCoremonMetric(
            177,
            12342348,
            Labels.of("s", "a", "m", "e"),
            InstantUtils.parseToSeconds("2015-10-12T13:14:14Z"),
            MetricType.COUNTER);

        writeMetrics(operationId1, numId, m1, m2);
        writeMetrics(operationId2, numId, m3);

        // act
        deleteBatchMetrics(operationId1, numId);
        var operation1AfterDelete1 = readMetrics(operationId1, numId);
        var operation2AfterDelete1 = readMetrics(operationId2, numId);
        var count1AfterDelete1 = countMetrics(operationId1, numId);
        var count2AfterDelete1 = countMetrics(operationId2, numId);

        deleteBatchMetrics(operationId2, numId);
        var operation1AfterDelete2 = readMetrics(operationId1, numId);
        var operation2AfterDelete2 = readMetrics(operationId2, numId);
        var count1AfterDelete2 = countMetrics(operationId1, numId);
        var count2AfterDelete2 = countMetrics(operationId2, numId);

        // assert
        assertMetricsEqual(List.of(), operation1AfterDelete1);
        assertMetricsEqual(List.of(m3), operation2AfterDelete1);
        assertEquals(operation1AfterDelete1.size(), count1AfterDelete1);
        assertEquals(operation2AfterDelete1.size(), count2AfterDelete1);

        assertMetricsEqual(List.of(), operation1AfterDelete2);
        assertMetricsEqual(List.of(), operation2AfterDelete2);
        assertEquals(operation1AfterDelete2.size(), count1AfterDelete2);
        assertEquals(operation2AfterDelete2.size(), count2AfterDelete2);
    }

    @Test
    public void emptyBulkUpsertIsIllegal() {
        // arrange
        var operationId = operationId();

        // act
        try {
            writeMetrics(operationId(), numId, List.of());
        } catch (Exception ex) {
            var cause = unwrapCompletionException(ex);
            assertThat(cause.getMessage(), containsString("bulkUpsert"));
            assertThat(cause.getMessage(), containsString("empty"));
        }

        // assert
        assertEquals(List.of(), readMetrics(operationId, numId));
    }

    private void writeMetrics(String operationId, int numId, Collection<CoremonMetric> metrics) {
        writeMetrics(operationId, numId, metrics.toArray(CoremonMetric[]::new));
    }

    private void writeMetrics(String operationId, int numId, CoremonMetric... metrics) {
        try (CoremonMetricArray ms = new CoremonMetricArray(metrics)) {
            writeMetrics(operationId, numId, ms);
        }
    }

    private void writeMetrics(String operationId, int numId, CoremonMetricArray metrics) {
        timedRoutine("Write", () -> getDao().bulkUpsert(operationId, numId, metrics));
    }

    private List<CoremonMetric> readMetrics(String operationId, int numId) {
        return readMetrics(operationId, numId, 1000, null);
    }

    private List<CoremonMetric> readMetrics(
        String operationId,
        int numId,
        int limit,
        @Nullable Labels lastKey)
    {
        try (var metrics = new CoremonMetricArray(limit)) {
            timedRoutine("Read", () -> getDao().find(operationId, numId, limit, lastKey, metrics, LABEL_ALLOCATOR));
            return metrics.stream()
                .<CoremonMetric>map(FileCoremonMetric::new)
                .collect(toList());
        }
    }

    private long countMetrics(String operationId, int numId) {
        return timedRoutine("Count", () -> getDao().count(operationId, numId));
    }

    private void deleteMetrics(String operationId, int numId, Collection<Labels> keys) {
        timedRoutine("Delete", () -> getDao().delete(operationId, numId, keys));
    }

    private long deleteBatchMetrics(String operationId, int numId) {
        return timedRoutine("Delete batch", () -> getDao().deleteBatch(operationId, numId));
    }

    private List<CoremonMetric> readMetricsOrderByLocalId(String operationId, int numId) {
        return readMetrics(operationId, numId).stream()
            .sorted(comparing(CoremonMetric::getLocalId))
            .collect(toList());
    }

    private static String operationId() {
        return randomAlphanumeric(8);
    }

    private static <T> T timedRoutine(String routineName, Supplier<CompletableFuture<T>> routine) {
        var startTimeMillis = System.currentTimeMillis();
        var result = join(routine.get());
        logger.info("{}: {}ms", routineName, System.currentTimeMillis() - startTimeMillis);
        return result;
    }

    private static void assertMetricsEqual(List<CoremonMetric> expected, List<CoremonMetric> actual) {
        assertEquals(expected.size(), actual.size());
        for (int i = 0; i < expected.size(); i++) {
            CoremonMetricHelper.assertEquals(expected.get(i), actual.get(i));
        }
    }
}
