package ru.yandex.solomon.coremon.meta.db;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.OptionalLong;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.CoremonMetricArray;
import ru.yandex.solomon.coremon.meta.CoremonMetricHelper;
import ru.yandex.solomon.coremon.meta.FileCoremonMetric;
import ru.yandex.solomon.util.time.InstantUtils;
import ru.yandex.stockpile.client.shard.StockpileMetricId;

import static ru.yandex.solomon.coremon.meta.CoremonMetricHelper.assertEquals;


/**
 * @author Stepan Koltsov
 */
public abstract class MetricsDaoTest {

    protected MetricsDao metricsDao;

    protected abstract void initClientCreateSchema(String dbName);

    @Rule
    public TestName testName = new TestName();

    @Before
    public void before() throws Exception {
        initClientCreateSchema(getClass().getSimpleName() + "_" + testName.getMethodName());
    }

    @After
    public abstract void closeClient();

    @Test
    public void testFindMetrics() {
        Assert.assertTrue(findMetrics().isEmpty());

        CoremonMetric r1 = new FileCoremonMetric(
                14,
                10,
                Labels.of("x", "y", "a", "b"),
                InstantUtils.parseToSeconds("2015-10-12T13:14:10Z"),
                MetricType.RATE);
        CoremonMetric r2 = new FileCoremonMetric(
                13,
                12,
                Labels.of("x", "y", "a", "c"),
                InstantUtils.parseToSeconds("2015-10-12T13:14:12Z"),
                MetricType.DGAUGE);

        try (CoremonMetricArray inserted = new CoremonMetricArray(r1, r2)) {
            replaceMetrics(inserted);

            List<CoremonMetric> read = findMetricsOrderById();
            Assert.assertEquals(inserted.size(), read.size());
            for (int i = 0; i < inserted.size(); i++) {
                assertEquals(inserted.get(i), read.get(i));
            }
        }
    }

    @Test
    public void findStockpileIds() {
        Assert.assertTrue(findMetrics().isEmpty());

        CoremonMetric r1 = new FileCoremonMetric(
                10,
                14,
                Labels.of("x", "y", "a", "b"),
                InstantUtils.parseToSeconds("2015-10-12T13:14:10Z"),
                MetricType.RATE);
        CoremonMetric r2 = new FileCoremonMetric(
                13,
                -12,
                Labels.of("x", "y", "a", "c"),
                InstantUtils.parseToSeconds("2015-10-12T13:14:12Z"),
                MetricType.DGAUGE);

        try (CoremonMetricArray metrics = new CoremonMetricArray(r1, r2)) {
            replaceMetrics(metrics);
        }

        List<StockpileMetricId> written = Stream.of(r1, r2)
            .map(s -> new StockpileMetricId(s.getShardId(), s.getLocalId()))
            .sorted()
            .collect(Collectors.toList());
        List<StockpileMetricId> read = findMetrics()
            .stream()
            .map(metric -> new StockpileMetricId(metric.getShardId(), metric.getLocalId()))
            .sorted()
            .collect(Collectors.toList());
        Assert.assertEquals(read, written);
    }

    @Test
    public void updateMode() {
        CoremonMetric r1 = new FileCoremonMetric(
                13,
                10,
                Labels.of("x", "y", "a", "b"),
                InstantUtils.parseToSeconds("2015-10-12T13:14:10Z"),
                MetricType.RATE);
        CoremonMetric r2 = new FileCoremonMetric(
                14,
                12,
                Labels.of("x", "y", "a", "c"),
                InstantUtils.parseToSeconds("2015-10-12T13:14:12Z"),
                MetricType.DGAUGE);

        try (CoremonMetricArray metrics = new CoremonMetricArray(r1, r2)) {
            replaceMetrics(metrics);
        }

        Assert.assertEquals(1, updateModeForMetric(MetricType.DGAUGE, 10));
        Assert.assertEquals(1, updateModeForMetric(MetricType.RATE, 12));

        List<CoremonMetric> read = findMetricsOrderById();

        List<CoremonMetric> expected = List.of(
            new FileCoremonMetric(
                    13,
                    10,
                    Labels.of("x", "y", "a", "b"),
                    InstantUtils.parseToSeconds("2015-10-12T13:14:10Z"),
                    MetricType.DGAUGE),
            new FileCoremonMetric(
                    14,
                    12,
                    Labels.of("x", "y", "a", "c"),
                    InstantUtils.parseToSeconds("2015-10-12T13:14:12Z"),
                    MetricType.RATE)
        );

        Assert.assertEquals(expected, read);
    }

    private int updateModeForMetric(MetricType type, int id) {
        CoremonMetric[] updatedMetrics = findMetrics().stream()
            .filter(metric -> (int) metric.getLocalId() == id)
            .filter(metric -> metric.getType() != type)
            .map(metric -> new FileCoremonMetric(
                metric.getShardId(),
                metric.getLocalId(),
                metric.getLabels(),
                metric.getCreatedAtSeconds(),
                type))
            .toArray(CoremonMetric[]::new);

        try (CoremonMetricArray metrics = new CoremonMetricArray(updatedMetrics)) {
            replaceMetrics(metrics);
        }
        return updatedMetrics.length;
    }

    @Test
    public void deleteMetricRecords() throws Exception {
        CoremonMetric r1 = new FileCoremonMetric(
            13,
            10,
            Labels.of("x", "y", "a", "b"),
            InstantUtils.parseToSeconds("2015-10-12T13:14:10Z"),
            MetricType.RATE);

        CoremonMetric r2 = new FileCoremonMetric(
            14,
            11,
            Labels.of("x", "y", "a", "c"),
            InstantUtils.parseToSeconds("2016-11-12T13:14:10Z"),
            MetricType.RATE);

        try (CoremonMetricArray metrics = new CoremonMetricArray(r1, r2)) {
            replaceMetrics(metrics);
        }

        metricsDao.deleteMetrics(List.of(r1.getLabels())).join();
        List<CoremonMetric> found = findMetrics();
        Assert.assertEquals(1, found.size());
        CoremonMetricHelper.assertEquals(r2, found.get(0));
    }

    @Test
    public void deleteMetricBatch() {
        int nowSeconds = InstantUtils.parseToSeconds("2015-10-12T13:14:10Z");

        String[] labelNames = IntStream.range(0, Labels.MAX_LABELS_COUNT)
                .mapToObj(x -> "label" + x + "_name_looooooooooooooong")
                .toArray(String[]::new);

        int metricsCount = 15_000;
        try (CoremonMetricArray metrics = new CoremonMetricArray(metricsCount)) {
            for (int i = 1; i <= metricsCount; i++) {
                int iFinal = i;
                Map<String, String> labels = Arrays.stream(labelNames)
                        .collect(Collectors.toMap(name -> name, name -> String.valueOf(iFinal)));
                metrics.add(13, i, Labels.of(labels), nowSeconds, MetricType.RATE);
            }

            replaceMetrics(metrics);

            long removed = 0;
            long lastCount;
            do {
                lastCount = deleteMetrics();
                removed += lastCount;
            } while (lastCount > 0);

            List<CoremonMetric> found = findMetrics();
            Assert.assertEquals(metrics.size(), (int) removed);
            Assert.assertEquals(0, lastCount);
            Assert.assertEquals(List.of(), found);
        }
    }

    @Test
    public void largeMetabaseSupport() {
        int nowSeconds = InstantUtils.parseToSeconds("2015-10-12T13:14:10Z");

        String[] labelNames = IntStream.range(0, Labels.MAX_LABELS_COUNT)
            .mapToObj(x -> "label" + x + "_name_looooooooooooooong")
            .toArray(String[]::new);

        try (CoremonMetricArray metrics = new CoremonMetricArray(5000)) {
            for (int i = 1; i <= 5000; i++) {
                int iFinal = i;
                Map<String, String> labels = Arrays.stream(labelNames)
                    .collect(Collectors.toMap(name -> name, name -> String.valueOf(iFinal)));
                metrics.add(13, i, Labels.of(labels), nowSeconds, MetricType.RATE);
            }

            replaceMetrics(metrics);
            List<CoremonMetric> found = findMetrics();

            List<CoremonMetric> saved = metrics.stream()
                .sorted(CoremonMetricHelper::compare)
                .collect(Collectors.toList());

            found.sort(CoremonMetricHelper::compare);

            Assert.assertEquals(saved.size(), found.size());
            for (int i = 0; i < found.size(); i++) {
                CoremonMetricHelper.assertEquals(saved.get(i), found.get(i));
            }
        }
    }

    @Test
    public void insertMNoMergeSmall() {
        Labels labels = Labels.of("x", "y", "a", "b");
        int shardId = 13;
        long localId = 10;

        CoremonMetric r1 = new FileCoremonMetric(
                shardId,
                localId,
                labels,
                InstantUtils.parseToSeconds("2015-10-12T13:14:10Z"),
                MetricType.RATE);

        try (CoremonMetricArray metrics = new CoremonMetricArray(r1)) {
            replaceMetrics(metrics);
        }
        List<CoremonMetric> found1 = findMetrics();
        Assert.assertEquals(1, found1.size());
        CoremonMetricHelper.assertEquals(r1, found1.get(0));

        CoremonMetric r2 = new FileCoremonMetric(
                shardId,
                localId,
                labels,
                InstantUtils.parseToSeconds("2015-10-12T13:14:10Z"),
                MetricType.COUNTER);

        try (CoremonMetricArray metrics = new CoremonMetricArray(r2)) {
            replaceMetrics(metrics);
        }
        List<CoremonMetric> found2 = findMetrics();
        Assert.assertEquals(1, found2.size());
        CoremonMetricHelper.assertEquals(r2, found2.get(0));
    }

    @Test
    public void insertMetricsNoMergeLarge() {
        int nowSeconds = InstantUtils.parseToSeconds("2015-10-12T13:14:10Z");

        try (CoremonMetricArray metrics = new CoremonMetricArray(5000)) {
            String[] labelNames = IntStream.range(0, Labels.MAX_LABELS_COUNT)
                .mapToObj(x -> "label" + x + "_name_looooooooooooooong")
                .toArray(String[]::new);

            for (int i = 1; i <= 5000; i++) {
                int iFinal = i;
                Map<String, String> labels = Arrays.stream(labelNames)
                    .collect(Collectors.toMap(name -> name, name -> String.valueOf(iFinal)));
                metrics.add(13, i, Labels.of(labels), nowSeconds, MetricType.RATE);
            }

            replaceMetrics(metrics);
            List<CoremonMetric> found = findMetrics();

            List<CoremonMetric> saved = metrics.stream()
                .sorted(CoremonMetricHelper::compare)
                .collect(Collectors.toList());
            found.sort(CoremonMetricHelper::compare);

            Assert.assertEquals(saved.size(), found.size());
            for (int i = 0; i < found.size(); i++) {
                CoremonMetricHelper.assertEquals(saved.get(i), found.get(i));
            }
        }
    }

    private void replaceMetrics(CoremonMetricArray metrics) {
        long startTimeMillis = System.currentTimeMillis();
        metricsDao.replaceMetrics(metrics).join();
        System.out.println("Write: " + (System.currentTimeMillis() - startTimeMillis) + "ms");
    }

    private List<CoremonMetric> findMetrics() {
        long startTimeMillis = System.currentTimeMillis();
        List<CoremonMetric> result = new ArrayList<>();
        metricsDao.findMetrics(chunk -> {
            for (int i = 0; i < chunk.size(); i++) {
                result.add(new FileCoremonMetric(chunk.get(i)));
            }
        }, OptionalLong.empty()).join();
        System.out.println("Read: " + (System.currentTimeMillis() - startTimeMillis) + "ms");
        return result;
    }

    private long deleteMetrics() {
        long startTimeMillis = System.currentTimeMillis();
        long count = CompletableFutures.join(metricsDao.deleteMetricsBatch());
        System.out.println("Delete: " + (System.currentTimeMillis() - startTimeMillis) + "ms");
        return count;
    }

    private List<CoremonMetric> findMetricsOrderById() {
        return findMetrics().stream()
            .map(FileCoremonMetric::new)
            .sorted(Comparator.comparing(CoremonMetric::getLocalId))
            .collect(Collectors.toList());
    }
}
