package ru.yandex.solomon.coremon.meta.db;

import java.util.IntSummaryStatistics;
import java.util.concurrent.ForkJoinPool;

import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.metrics.client.StockpileClientStub;
import ru.yandex.stockpile.client.shard.StockpileShardId;

import static org.hamcrest.Matchers.greaterThanOrEqualTo;
import static org.hamcrest.Matchers.lessThanOrEqualTo;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertThat;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileMetricIdProviderHashTest {

    private StockpileClientStub stockpile;
    private ShardIdToAtomicInteger stats = new ShardIdToAtomicInteger();
    private StockpileMetricIdProviderHash provider;

    @Before
    public void setUp() {
        stockpile = new StockpileClientStub(ForkJoinPool.commonPool());
        stats = new ShardIdToAtomicInteger();
        provider = new StockpileMetricIdProviderHash(stats, stockpile);
    }

    @After
    public void tearDown() {
        stockpile.close();
    }

    @Test
    public void hash() {
        stockpile.setShardCount(4096);
        String key = "host";
        for (int i = 1; i <= 150_000_000; i++) {
            String hostname = "n" + i + "-sas";
            var labels = Labels.of(key, hostname);

            int shardId = provider.shardId(labels);
            StockpileShardId.validate(shardId);
        }

        IntSummaryStatistics actual = stats.stream().summaryStatistics();
        assertEquals(150_000_000, actual.getSum()); // total metrics

        double ideal = (double) actual.getSum() / (double) stockpile.getTotalShardsCount();

        System.out.println("avg: " + actual.getAverage());
        System.out.println("min: " + actual.getMin());
        System.out.println("max: " + actual.getMax());
        System.out.println("ideal: " + ideal);

        double avgDiff = Math.abs(ideal - actual.getAverage()) / ideal;
        System.out.println("avgDiff: " + String.format("%.5f", avgDiff));
        Assert.assertTrue(avgDiff < 0.0001);  // less than 0.01%

        double minDiff = Math.abs(ideal - actual.getMin()) / ideal;
        Assert.assertTrue(minDiff < 0.1);   // less than 10%

        double maxDiff = Math.abs(ideal - actual.getMax()) / ideal;
        Assert.assertTrue(maxDiff < 0.1);   // less than 10%
    }

    @Test
    public void changeNumberOfShards() {
        stockpile.setShardCount(32);
        String key = "host";
        for (int i = 1; i <= 100_000; i++) {
            String hostname = String.format("n-%06d-sas", i);
            var labels = Labels.of(key, hostname);

            int shardId = provider.shardId(labels);
            StockpileShardId.validate(shardId);
            assertThat(shardId, greaterThanOrEqualTo(1));
            assertThat(shardId, lessThanOrEqualTo(32));
        }

        stockpile.setShardCount(128);
        for (int i = 1; i <= 100_000; i++) {
            String hostname = String.format("n-%06d-sas", i);
            var labels = Labels.of(key, hostname);

            int shardId = provider.shardId(labels);
            StockpileShardId.validate(shardId);
            assertThat(shardId, greaterThanOrEqualTo(1));
            assertThat(shardId, lessThanOrEqualTo(128));
        }

        IntSummaryStatistics actual = stats.stream().summaryStatistics();
        assertEquals(200_000, actual.getSum()); // total metrics
        assertNotEquals(0, actual.getMin());

        double ideal = (double) actual.getSum() / (double) stockpile.getTotalShardsCount();

        System.out.println("avg: " + actual.getAverage());
        System.out.println("min: " + actual.getMin());
        System.out.println("max: " + actual.getMax());
        System.out.println("ideal: " + ideal);
    }
}
