package ru.yandex.solomon.coremon.meta.db.ydb;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.OptionalLong;
import java.util.concurrent.atomic.AtomicInteger;

import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.junit.Assert;
import org.junit.ClassRule;
import org.junit.Test;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.labels.string.StringLabelAllocator;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.CoremonMetricArray;
import ru.yandex.solomon.coremon.meta.CoremonMetricHelper;
import ru.yandex.solomon.coremon.meta.FileCoremonMetric;
import ru.yandex.solomon.coremon.meta.db.MetricsDao;
import ru.yandex.solomon.coremon.meta.db.MetricsDaoStats;
import ru.yandex.solomon.coremon.meta.db.MetricsDaoTest;
import ru.yandex.solomon.kikimr.LocalKikimr;
import ru.yandex.solomon.kikimr.YdbHelper;
import ru.yandex.solomon.util.time.InstantUtils;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertTrue;
import static ru.yandex.misc.concurrent.CompletableFutures.join;

/**
 * @author Sergey Polovko
 */
public class YdbMetricsHugeTableDaoTest extends MetricsDaoTest {

    @ClassRule
    public static final LocalKikimr localKikimr = new LocalKikimr();
    private YdbHelper ydb;

    private static final AtomicInteger cnt = new AtomicInteger(0);

    @Override
    protected void initClientCreateSchema(String dbName) {
        ydb = new YdbHelper(localKikimr, dbName);

        metricsDao = new YdbMetricsHugeTableDao(
            ydb.getTableClient(),
            cnt.incrementAndGet(),
            ydb.getRootPath(),
            new MetricsDaoStats(MetricRegistry.root()),
            StringLabelAllocator.SELF);
        metricsDao.createSchema().join();
    }


    @Override
    public void closeClient() {
        ydb.close();
    }

    @Test
    public void sameTableButDifferentShardIds() {
        var metricsDao2 = new YdbMetricsHugeTableDao(
            ydb.getTableClient(),
            cnt.get() + 64,
            ydb.getRootPath(),
            new MetricsDaoStats(MetricRegistry.root()),
            StringLabelAllocator.SELF);

        assertEquals(((YdbMetricsHugeTableDao) metricsDao).getTablePath(), metricsDao2.getTablePath());
        assertNotEquals(((YdbMetricsHugeTableDao) metricsDao).getShardId(), metricsDao2.getShardId());

        assertTrue(findMetrics(metricsDao).isEmpty());
        assertTrue(findMetrics(metricsDao2).isEmpty());

        var s1 = new FileCoremonMetric(
            10,
            14,
            Labels.of("x", "y", "a", "b"),
            InstantUtils.parseToSeconds("2015-10-12T13:14:10Z"),
            MetricType.RATE);

        var s2 = new FileCoremonMetric(
            13,
            -12,
            Labels.of("x", "y", "a", "c"),
            InstantUtils.parseToSeconds("2015-10-12T13:14:12Z"),
            MetricType.DGAUGE);

        var s3 = new FileCoremonMetric(
            177,
            12342348,
            Labels.of("x", "y", "a", "d"),
            InstantUtils.parseToSeconds("2015-10-12T13:14:14Z"),
            MetricType.COUNTER);

        // (1) replace & find metrics
        try (CoremonMetricArray metrics = new CoremonMetricArray(s1, s2)) {
            join(metricsDao.replaceMetrics(metrics));
        }
        try (CoremonMetricArray metrics = new CoremonMetricArray(s3)) {
            join(metricsDao2.replaceMetrics(metrics));
        }
        assertEquals(List.of(s1, s2), findMetrics(metricsDao));
        assertEquals(2L, metricsDao.getMetricCount().join().longValue());
        assertEquals(List.of(s3), findMetrics(metricsDao2));
        assertEquals(1L, metricsDao2.getMetricCount().join().longValue());

        // (2) delete metric from first shard
        join(metricsDao.deleteMetrics(List.of(s2.getLabels())));
        assertEquals(List.of(s1), findMetrics(metricsDao));
        assertEquals(1L, metricsDao.getMetricCount().join().longValue());
        assertEquals(List.of(s3), findMetrics(metricsDao2));
        assertEquals(1L, metricsDao2.getMetricCount().join().longValue());

        // (3) delete metrics from second shard
        join(metricsDao2.deleteMetrics(List.of(s3.getLabels())));
        assertEquals(List.of(s1), findMetrics(metricsDao));
        assertEquals(1L, metricsDao.getMetricCount().join().longValue());
        assertEquals(List.of(), findMetrics(metricsDao2));
        assertEquals(0L, metricsDao2.getMetricCount().join().longValue());
    }

    @Test
    public void replaceMetrics_insertNew() {
        var metricsMeta = new YdbMetricsHugeTableDao(
                ydb.getTableClient(),
                cnt.get() + 1,
                ydb.getRootPath(),
                new MetricsDaoStats(MetricRegistry.root()),
                StringLabelAllocator.SELF);
        metricsMeta.createSchema().join();

        var s1 = new FileCoremonMetric(
                8912,
                18,
                Labels.of("a", "z", "w", "ty"),
                InstantUtils.parseToSeconds("2021-05-30T11:22:12Z"),
                MetricType.COUNTER);
        var s2 = new FileCoremonMetric(
                9055,
                22,
                Labels.of("a", "a1"),
                InstantUtils.parseToSeconds("2021-05-30T11:22:12Z"),
                MetricType.HIST);

        List<CoremonMetric> actualMetrics = new ArrayList<>();
        try (CoremonMetricArray metrics = new CoremonMetricArray(s1, s2)) {
            metricsMeta.replaceMetrics(metrics).thenApply(m -> {
                for (int i = 0; i < m.size(); i++) {
                    actualMetrics.add(new FileCoremonMetric(m.get(i)));
                }
                return null;
            }).join();
        }
        MatcherAssert.assertThat(actualMetrics, Matchers.containsInAnyOrder(s2, s1));

        var expectedMetricsNum = metricsMeta.getMetricCount().join().longValue();
        var s2New = new FileCoremonMetric(
                9055,
                22,
                Labels.of("a", "a1"),
                InstantUtils.parseToSeconds("2021-05-30T11:22:12Z"),
                MetricType.RATE);
        List<CoremonMetric> updatedMetrics = new ArrayList<>();
        try (CoremonMetricArray metrics = new CoremonMetricArray(s1, s2New)) {
            metricsMeta.replaceMetrics(metrics).thenApply(m -> {
                for (int i = 0; i < m.size(); i++) {
                    updatedMetrics.add(new FileCoremonMetric(m.get(i)));
                }
                return null;
            }).join();
        }
        MatcherAssert.assertThat(updatedMetrics, Matchers.containsInAnyOrder(s2New));
        assertEquals(expectedMetricsNum, metricsMeta.getMetricCount().join().longValue());

        var s1Same = new FileCoremonMetric(
                500,
                974,
                Labels.of("a", "z", "w", "ty"),
                InstantUtils.parseToSeconds("2021-05-30T11:22:12Z"),
                MetricType.COUNTER);
        List<CoremonMetric> expectedEmptyList = new ArrayList<>();
        try (CoremonMetricArray metrics = new CoremonMetricArray(s1Same)) {
            metricsMeta.replaceMetrics(metrics).thenApply(m -> {
                for (int i = 0; i < m.size(); i++) {
                    expectedEmptyList.add(new FileCoremonMetric(m.get(i)));
                }
                return null;
            }).join();
        }
        MatcherAssert.assertThat(expectedEmptyList, Matchers.empty());
    }

    @Test
    public void findWithCount() {
        CoremonMetric r1 = new FileCoremonMetric(
                14,
                10,
                Labels.of("x", "y", "a", "b"),
                InstantUtils.parseToSeconds("2015-10-12T13:14:10Z"),
                MetricType.RATE);
        CoremonMetric r2 = new FileCoremonMetric(
                13,
                12,
                Labels.of("x", "y", "a", "c"),
                InstantUtils.parseToSeconds("2015-10-12T13:14:12Z"),
                MetricType.DGAUGE);

        try (CoremonMetricArray inserted = new CoremonMetricArray(r1, r2)) {
            CompletableFutures.join(metricsDao.replaceMetrics(inserted));

            List<CoremonMetric> found = new ArrayList<>();
            metricsDao.findMetrics(chunk -> {
                for (int i = 0; i < chunk.size(); i++) {
                    found.add(new FileCoremonMetric(chunk.get(i)));
                }
            }, OptionalLong.of(10)).join();

            found.sort(Comparator.comparing(CoremonMetric::getLocalId));

            Assert.assertEquals(inserted.size(), found.size());
            for (int i = 0; i < inserted.size(); i++) {
                CoremonMetricHelper.assertEquals(inserted.get(i), found.get(i));
            }
        }
    }

    private static List<CoremonMetric> findMetrics(MetricsDao metricsDao) {
        List<CoremonMetric> result = new ArrayList<>();
        metricsDao.findMetrics(chunk -> {
            for (int i = 0; i < chunk.size(); i++) {
                result.add(new FileCoremonMetric(chunk.get(i)));
            }
        }, OptionalLong.empty()).join();
        return result;
    }
}
