package ru.yandex.solomon.coremon.meta.file;

import java.util.Arrays;
import java.util.Collections;

import org.junit.Assert;
import org.junit.Test;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.CoremonMetricArray;

import static ru.yandex.solomon.coremon.meta.CoremonMetricHelper.metrics;


/**
 * @author Sergey Polovko
 */
public class LevelTest {

    @Test
    public void update() {
        // empty level
        {
            Level level = new Level();
            Level updatedLevel = level.update(metrics(
                Labels.of("a", "b")
            ));

            Assert.assertNotSame(level, updatedLevel);

            // initial level not changed
            Assert.assertEquals(0, level.size());
            Assert.assertEquals(1, level.refCnt());

            Assert.assertEquals(1, updatedLevel.size());
            Assert.assertEquals(1, updatedLevel.refCnt());
            Assert.assertNotNull(updatedLevel.getMetricByLabels(Labels.of("a", "b")));

            Assert.assertTrue(level.release());
            Assert.assertTrue(updatedLevel.release());
        }

        // filled level
        {
            Level level = new Level(metrics(
                Labels.of("a", "b")
            ));
            Level updatedLevel = level.update(metrics(
                Labels.of("c", "d")
            ));

            Assert.assertNotSame(level, updatedLevel);

            // initial level not changed
            Assert.assertEquals(1, level.size());
            Assert.assertEquals(1, level.refCnt());

            Assert.assertEquals(2, updatedLevel.size());
            Assert.assertEquals(1, updatedLevel.refCnt());
            Assert.assertTrue(updatedLevel.has(Labels.of("a", "b")));
            Assert.assertTrue(updatedLevel.has(Labels.of("c", "d")));

            Assert.assertTrue(level.release());
            Assert.assertTrue(updatedLevel.release());
        }

        // empty metrics
        {
            Level level = new Level(metrics(
                Labels.of("a", "b")
            ));
            Level updatedLevel = level.update(new CoremonMetricArray(0));

            // initial level not changed
            Assert.assertEquals(1, level.size());
            Assert.assertEquals(1, level.refCnt());

            Assert.assertSame(level, updatedLevel);
            Assert.assertTrue(level.release());
        }
    }

    @Test
    public void mergeWith() {
        // left is empty
        {
            Level left = new Level();
            Level right = new Level(metrics(
                Labels.of("a", "b")
            ));
            Level merged = left.mergeWith(right);

            // left not changed
            Assert.assertEquals(0, left.size());
            Assert.assertEquals(1, left.refCnt());
            // right not changed
            Assert.assertEquals(1, right.size());
            Assert.assertEquals(1, right.refCnt());

            Assert.assertSame(merged, right);
            Assert.assertTrue(left.release());
            Assert.assertTrue(right.release());
        }

        // right is empty
        {
            Level left = new Level(metrics(
                Labels.of("a", "b")
            ));
            Level right = new Level();
            Level merged = left.mergeWith(right);

            // left not changed
            Assert.assertEquals(1, left.size());
            Assert.assertEquals(1, left.refCnt());
            // right not changed
            Assert.assertEquals(0, right.size());
            Assert.assertEquals(1, right.refCnt());

            Assert.assertSame(merged, left);
            Assert.assertTrue(left.release());
            Assert.assertTrue(right.release());
        }

        // both are not empty
        {
            Level left = new Level(metrics(
                Labels.of("a", "b")
            ));
            Level right = new Level(metrics(
                Labels.of("c", "d")
            ));
            Level merged = left.mergeWith(right);

            // left not changed
            Assert.assertEquals(1, left.size());
            Assert.assertEquals(1, left.refCnt());
            // right not changed
            Assert.assertEquals(1, right.size());
            Assert.assertEquals(1, right.refCnt());

            Assert.assertNotSame(merged, left);
            Assert.assertNotSame(merged, right);
            Assert.assertEquals(2, merged.size());
            Assert.assertEquals(1, merged.refCnt());
            Assert.assertTrue(merged.has(Labels.of("a", "b")));
            Assert.assertTrue(merged.has(Labels.of("c", "d")));

            Assert.assertTrue(left.release());
            Assert.assertTrue(right.release());
            Assert.assertTrue(merged.release());
        }
    }

    @Test
    public void remove() {
        Labels[] even = {
            Labels.of("s", "0"), Labels.of("s", "2"),
            Labels.of("s", "4"), Labels.of("s", "6"),
            Labels.of("s", "8"),
        };

        Labels[] odd = {
            Labels.of("s", "1"), Labels.of("s", "3"),
            Labels.of("s", "5"), Labels.of("s", "7"),
            Labels.of("s", "9"),
        };

        Level level = new Level(metrics(
            Labels.of("s", "0"), Labels.of("s", "1"), Labels.of("s", "2"), Labels.of("s", "3"),
            Labels.of("s", "4"), Labels.of("s", "5"), Labels.of("s", "6"), Labels.of("s", "7"),
            Labels.of("s", "8"), Labels.of("s", "9")
        ));

        // remove non exists metric
        {
            Level updatedLevel = level.remove(Collections.singletonList(new RemoveRequest(Labels.of("a", "b"))));

            // initial level not changed
            Assert.assertEquals(10, level.size());
            Assert.assertEquals(1, level.refCnt());

            Assert.assertSame(level, updatedLevel);
        }

        // remove odd
        {
            Level updatedLevel = level.remove(Arrays.asList(
                new RemoveRequest(odd[0]),
                new RemoveRequest(odd[1], odd[2], odd[3]),
                new RemoveRequest(odd[4])
            ));

            // initial level not changed
            Assert.assertEquals(10, level.size());
            Assert.assertEquals(1, level.refCnt());

            Assert.assertNotSame(level, updatedLevel);
            Assert.assertEquals(5, updatedLevel.size());
            Assert.assertEquals(1, updatedLevel.refCnt());

            // must found
            for (Labels l : even) {
                Assert.assertTrue(updatedLevel.has(l));
            }

            // must not found
            for (Labels l : odd) {
                Assert.assertFalse(updatedLevel.has(l));
            }
        }

        // remove even
        {
            Level updatedLevel = level.remove(Arrays.asList(
                new RemoveRequest(even[0], even[1]),
                new RemoveRequest(even[2], even[3]),
                new RemoveRequest(even[4])
            ));

            // initial level not changed
            Assert.assertEquals(10, level.size());
            Assert.assertEquals(1, level.refCnt());

            Assert.assertNotSame(level, updatedLevel);
            Assert.assertEquals(5, updatedLevel.size());
            Assert.assertEquals(1, updatedLevel.refCnt());

            // must found
            for (Labels l : odd) {
                Assert.assertTrue(updatedLevel.has(l));
            }

            // must not found
            for (Labels l : even) {
                Assert.assertFalse(updatedLevel.has(l));
            }
        }
    }


    @Test
    public void retainMetricsArray() {
        // get metric before level release
        {
            Level level = new Level(metrics(Labels.of("s", "1"), Labels.of("s", "2")));
            CoremonMetricArray metrics = level.getMetrics();

            Assert.assertEquals(1, level.refCnt());
            Assert.assertEquals(1, metrics.refCnt());

            CoremonMetric metric = level.getMetricByIndex(0);
            Assert.assertNotNull(metric);
            Assert.assertEquals(Labels.of("s", "1"), metric.getLabels());

            // obtained metric retains only metrics array
            Assert.assertEquals(1, level.refCnt());
            Assert.assertEquals(2, metrics.refCnt());

            // releasing level decrement metrics array refCnt
            Assert.assertTrue(level.release());;
            Assert.assertEquals(0, level.refCnt());
            Assert.assertEquals(1, metrics.refCnt());

            // after releasing level we can access metric's data
            Assert.assertEquals(Labels.of("s", "1"), metric.getLabels());
            metric.setLastPointSeconds(1234567890);
            Assert.assertEquals(1234567890, metric.getLastPointSeconds());

            // closing metric will destroy underlying array
            metric.close();
            Assert.assertEquals(0, metrics.refCnt());
        }

        // get metric after level release
        {
            Level level = new Level(metrics(Labels.of("s", "1"), Labels.of("s", "2")));
            CoremonMetricArray metrics = level.getMetrics();

            Assert.assertEquals(1, level.refCnt());
            Assert.assertEquals(1, metrics.refCnt());

            level.release();
            Assert.assertEquals(0, level.refCnt());
            Assert.assertEquals(0, metrics.refCnt());

            CoremonMetric metric = level.getMetricByIndex(0);
            Assert.assertNull(metric);
        }
    }

    @Test
    public void nullableAutoCloseable() {
        // null
        {
            Level level = new Level(metrics(Labels.of("s", "1"), Labels.of("s", "2")));
            CoremonMetricArray metrics = level.getMetrics();

            Assert.assertEquals(1, level.refCnt());
            Assert.assertEquals(1, metrics.refCnt());

            try (CoremonMetric s = level.getMetricByIndex(2)) {
                Assert.assertNull(s);
                Assert.assertEquals(1, level.refCnt());
                Assert.assertEquals(1, metrics.refCnt());
            }

            Assert.assertEquals(1, level.refCnt());
            Assert.assertEquals(1, metrics.refCnt());

            level.release();

            Assert.assertEquals(0, level.refCnt());
            Assert.assertEquals(0, metrics.refCnt());
        }

        // nonnull
        {
            Level level = new Level(metrics(Labels.of("s", "1"), Labels.of("s", "2")));
            CoremonMetricArray metrics = level.getMetrics();

            Assert.assertEquals(1, level.refCnt());
            Assert.assertEquals(1, metrics.refCnt());

            try (CoremonMetric s = level.getMetricByIndex(0)) {
                Assert.assertNotNull(s);
                Assert.assertEquals(1, level.refCnt());
                Assert.assertEquals(2, metrics.refCnt());
            }

            Assert.assertEquals(1, level.refCnt());
            Assert.assertEquals(1, metrics.refCnt());

            level.release();

            Assert.assertEquals(0, level.refCnt());
            Assert.assertEquals(0, metrics.refCnt());
        }
    }
}
