package ru.yandex.solomon.coremon.meta.file;

import java.util.Collections;

import org.junit.Assert;
import org.junit.Test;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.coremon.meta.CoremonMetricArray;
import ru.yandex.solomon.util.labelStats.LabelValuesStats;

import static ru.yandex.solomon.coremon.meta.CoremonMetricHelper.metrics;


/**
 * @author Sergey Polovko
 */
public class LevelsArrayTest {

    private static final int[] TEST_LEVELS_SIZES = { 1, 2, 4, 8 };

    @Test
    public void emptyArray() {
        LevelsArray a = new LevelsArray(TEST_LEVELS_SIZES, new CoremonMetricArray(0));
        assertLevelsSize(a, 0, 0, 0, 0);
        assertLevelsRefs(a, 1, 1, 1, 1);
        Assert.assertEquals(0, a.size());
        Assert.assertEquals(1, a.refCnt());
        Assert.assertTrue(a.release());
    }

    @Test
    public void updateZeroLevel() {
        // empty metrics
        {
            LevelsArray a = new LevelsArray(TEST_LEVELS_SIZES, new CoremonMetricArray(0));
            LevelsArray b = a.updateZeroLevel(metrics());
            Assert.assertSame(a, b);
        }

        // non empty metrics
        {
            LevelsArray a = new LevelsArray(TEST_LEVELS_SIZES, new CoremonMetricArray(0));
            LevelsArray b = a.updateZeroLevel(metrics(Labels.of("a", "b")));
            Assert.assertNotSame(a, b);

            // a not changed
            assertLevelsSize(a, 0, 0, 0, 0);
            assertLevelsRefs(a, 1, 2, 2, 2);  // but levels >1 are shared with b
            Assert.assertEquals(0, a.size());
            Assert.assertEquals(1, a.refCnt());

            // b has new metrics
            assertLevelsSize(b, 1, 0, 0, 0);
            assertLevelsRefs(b, 1, 2, 2, 2);
            Assert.assertEquals(1, b.size());
            Assert.assertEquals(1, b.refCnt());
            Assert.assertTrue(b.has(Labels.of("a", "b")));

            Assert.assertTrue(a.release());
            assertLevelsRefs(b, 1, 1, 1, 1);
            Assert.assertTrue(b.release());
        }

        // non empty levels
        {
            LevelsArray a = new LevelsArray(TEST_LEVELS_SIZES, metrics(Labels.of("a", "b")));
            LevelsArray b = a.updateZeroLevel(metrics(Labels.of("c", "d")));
            Assert.assertNotSame(a, b);

            // a not changed
            assertLevelsSize(a, 0, 0, 0, 1);
            assertLevelsRefs(a, 1, 2, 2, 2);  // but levels >1 are shared with b
            Assert.assertEquals(1, a.size());
            Assert.assertEquals(1, a.refCnt());

            // b has new metrics
            assertLevelsSize(b, 1, 0, 0, 1);
            assertLevelsRefs(b, 1, 2, 2, 2);
            Assert.assertEquals(2, b.size());
            Assert.assertEquals(1, b.refCnt());
            Assert.assertTrue(b.has(Labels.of("a", "b")));
            Assert.assertTrue(b.has(Labels.of("c", "d")));

            Assert.assertTrue(a.release());
            assertLevelsRefs(b, 1, 1, 1, 1);
            Assert.assertTrue(b.release());
        }
    }

    @Test
    public void remove() {
        // empty removals
        {
            LevelsArray a = new LevelsArray(TEST_LEVELS_SIZES, metrics(Labels.of("a", "b")));
            LevelsArray b = a.remove(Collections.emptyList());

            // a not changed
            assertLevelsSize(a, 0, 0, 0, 1);
            assertLevelsRefs(a, 1, 1, 1, 1);
            Assert.assertEquals(1, a.size());
            Assert.assertEquals(1, a.refCnt());

            Assert.assertSame(a, b);
            Assert.assertTrue(a.release());
        }

        // remove non exist metrics
        {
            LevelsArray a = new LevelsArray(TEST_LEVELS_SIZES, metrics(Labels.of("a", "b")));
            LevelsArray b = a.remove(Collections.singletonList(new RemoveRequest(Labels.of("c", "d"))));

            // a not changed
            assertLevelsSize(a, 0, 0, 0, 1);
            assertLevelsRefs(a, 1, 1, 1, 1);
            Assert.assertEquals(1, a.size());
            Assert.assertEquals(1, a.refCnt());

            Assert.assertSame(a, b);
            Assert.assertTrue(a.release());
        }

        // remove non exist metrics
        {
            LevelsArray a = new LevelsArray(TEST_LEVELS_SIZES, metrics(Labels.of("a", "b"), Labels.of("c", "d")));
            LevelsArray b = a.remove(Collections.singletonList(new RemoveRequest(Labels.of("c", "d"))));

            // a not changed
            assertLevelsSize(a, 0, 0, 0, 2);
            assertLevelsRefs(a, 2, 2, 2, 1);  // non modified levels are shared with b
            Assert.assertEquals(2, a.size());
            Assert.assertEquals(1, a.refCnt());

            Assert.assertNotSame(a, b);

            // b has no removed metric
            assertLevelsSize(b, 0, 0, 0, 1);
            assertLevelsRefs(b, 2, 2, 2, 1);
            Assert.assertEquals(1, b.size());
            Assert.assertEquals(1, b.refCnt());
            Assert.assertTrue(b.has(Labels.of("a", "b")));
            Assert.assertFalse(b.has(Labels.of("c", "d")));

            Assert.assertTrue(a.release());
            assertLevelsRefs(b, 1, 1, 1, 1);
            Assert.assertTrue(b.release());
        }
    }

    @Test
    public void mergeOverfullLevel() {
        LevelsArray a = new LevelsArray(TEST_LEVELS_SIZES, new CoremonMetricArray(0));
        assertLevelsRefs(a, 1, 1, 1, 1);

        // no overfull levels
        {
            LevelsArray b = a.updateZeroLevel(metrics(Labels.of("a", "b")));
            LevelsArray c = b.mergeOverfullLevel();

            // b not changed
            assertLevelsSize(b, 1, 0, 0, 0);
            assertLevelsRefs(b, 1, 2, 2, 2);
            Assert.assertEquals(1, b.size());
            Assert.assertEquals(1, b.refCnt());

            Assert.assertSame(b, c);
            Assert.assertTrue(b.release());
            assertLevelsRefs(a, 1, 1, 1, 1);
        }

        // zero level is overfull & next level is empty
        {
            LevelsArray b = a.updateZeroLevel(metrics(Labels.of("a", "b"), Labels.of("c", "d")));
            LevelsArray c = b.mergeOverfullLevel();

            // b not changed
            assertLevelsSize(b, 2, 0, 0, 0);
            assertLevelsRefs(b, 2, 2, 3, 3);
            Assert.assertEquals(2, b.size());
            Assert.assertEquals(1, b.refCnt());

            Assert.assertNotSame(b, c);
            assertLevelsSize(c, 0, 2, 0, 0);
            assertLevelsRefs(c, 1, 2, 3, 3);
            Assert.assertEquals(2, c.size());
            Assert.assertEquals(1, c.refCnt());
            Assert.assertTrue(c.has(Labels.of("a", "b")));
            Assert.assertTrue(c.has(Labels.of("c", "d")));

            Assert.assertTrue(b.release());
            assertLevelsRefs(c, 1, 1, 2, 2);
            Assert.assertTrue(c.release());
            assertLevelsRefs(a, 1, 1, 1, 1);
        }

        // zero level is overfull & next level is not empty
        {
            LevelsArray d, e;
            {
                LevelsArray b = a.updateZeroLevel(metrics(Labels.of("a", "b"), Labels.of("c", "d")));
                LevelsArray c = b.mergeOverfullLevel();
                Assert.assertTrue(b.release());
                d = c.updateZeroLevel(metrics(Labels.of("e", "f"), Labels.of("g", "h")));
                Assert.assertTrue(c.release());
                e = d.mergeOverfullLevel();
            }

            // b not changed
            assertLevelsSize(d, 2, 2, 0, 0);
            assertLevelsRefs(d, 1, 1, 3, 3);
            Assert.assertEquals(4, d.size());
            Assert.assertEquals(1, d.refCnt());

            Assert.assertNotSame(d, e);
            assertLevelsSize(e, 0, 4, 0, 0);
            assertLevelsRefs(e, 1, 1, 3, 3);
            Assert.assertEquals(4, e.size());
            Assert.assertEquals(1, e.refCnt());
            Assert.assertTrue(e.has(Labels.of("a", "b")));
            Assert.assertTrue(e.has(Labels.of("c", "d")));
            Assert.assertTrue(e.has(Labels.of("e", "f")));
            Assert.assertTrue(e.has(Labels.of("g", "h")));

            Assert.assertTrue(d.release());
            assertLevelsRefs(e, 1, 1, 2, 2);
            Assert.assertTrue(e.release());
            assertLevelsRefs(a, 1, 1, 1, 1);
        }

        Assert.assertTrue(a.release());
    }

    @Test
    public void emptyLevels() {
        Labels labels = Labels.of("a", "b");

        LevelsArray a = LevelsArray.EMPTY.updateZeroLevel(metrics(labels));
        Assert.assertSame(a, LevelsArray.EMPTY);

        LevelsArray b = LevelsArray.EMPTY.remove(Collections.singletonList(new RemoveRequest(labels)));
        Assert.assertSame(b, LevelsArray.EMPTY);

        LevelsArray c = LevelsArray.EMPTY.mergeOverfullLevel();
        Assert.assertSame(c, LevelsArray.EMPTY);

        Assert.assertEquals(0, LevelsArray.EMPTY.size());
        Assert.assertEquals(0, LevelsArray.EMPTY.count());
        Assert.assertEquals(0, LevelsArray.EMPTY.searchIndexSize());
        Assert.assertEquals(Collections.emptySet(), LevelsArray.EMPTY.labelNames());
        Assert.assertEquals(new LabelValuesStats(), LevelsArray.EMPTY.labelStats(Collections.emptySet()));
    }

    private static void assertLevelsSize(LevelsArray a, int... sizes) {
        Assert.assertEquals(a.count(), sizes.length);
        for (int i = 0; i < a.count(); i++) {
            Assert.assertEquals("at pos " + i, sizes[i], a.levelSize(i));
        }
    }

    private static void assertLevelsRefs(LevelsArray a, int... refCnt) {
        Assert.assertEquals(a.count(), refCnt.length);
        for (int i = 0; i < a.count(); i++) {
            Assert.assertEquals("at pos " + i, refCnt[i], a.getLevel(i).refCnt());
        }
    }
}
