package ru.yandex.solomon.coremon.meta.gc;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import it.unimi.dsi.fastutil.longs.LongArrayList;
import it.unimi.dsi.fastutil.longs.LongOpenHashSet;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.FileCoremonMetric;
import ru.yandex.solomon.coremon.meta.db.memory.InMemoryMetricsDao;
import ru.yandex.solomon.labels.LabelsFormat;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.shard.StockpileShardId;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

/**
 * @author Vladimir Gordiychuk
 */
public class GcMetricsTaskTest {
    private static final RetryConfig RETRY_CONFIG = RetryConfig.DEFAULT
            .withDelay(1)
            .withNumRetries(Integer.MAX_VALUE);

    private InMemoryMetricsDao dao;
    private StockpileStub stockpile;

    @Before
    public void setUp() throws Exception {
        dao = new InMemoryMetricsDao();
        stockpile = new StockpileStub();
    }

    @Test
    public void gcEmptyShard() {
        long deleted = runTask();
        assertEquals(0L, deleted);
        assertEquals(Map.of(), stockpile.deleted);
    }

    @Test
    public void gcSmallShard() {
        List<CoremonMetric> metrics = Arrays.asList(metric("name=alice"));
        dao.add(metrics);

        var deleted = runTask();
        assertEquals(1, deleted);
        assertDeleted(metrics);
    }

    @Test
    public void gcHugeShard() {
        var metrics = IntStream.range(0, 100_000)
                .mapToObj(idx -> metric("name=idx-" + idx))
                .collect(Collectors.toList());
        dao.add(metrics);

        var deleted = runTask();
        assertEquals(metrics.size(), deleted);
        assertDeleted(metrics);
    }

    private long runTask() {
        var task = new GcMetricsTask(42, dao, stockpile, RETRY_CONFIG);
        return task.run().join();
    }

    private void assertDeleted(List<CoremonMetric> metrics) {
        for (var metric : metrics) {
            String metricStr = metric.toString();
            var localIds = stockpile.deleted.get(metric.getShardId());
            assertNotNull(metricStr, localIds);
            assertTrue(metricStr, localIds.contains(metric.getLocalId()));
        }
        assertArrayEquals(new Object[0], dao.metrics().toArray());
    }

    private CoremonMetric metric(String labels) {
        int shardId = StockpileShardId.random(10);
        long localId = StockpileLocalId.random();
        return new FileCoremonMetric(shardId, localId, LabelsFormat.parse(labels), MetricType.DGAUGE);
    }

    private static class StockpileStub implements StockpileDeleter {
        private final ConcurrentHashMap<Integer, LongOpenHashSet> deleted = new ConcurrentHashMap<>();

        @Override
        public CompletableFuture<Void> delete(int shardId, long[] localIds) {
            return CompletableFuture.supplyAsync(() -> {
                if (ThreadLocalRandom.current().nextBoolean()) {
                    throw new IllegalStateException("oops");
                }

                deleted.compute(shardId, (id, prev) -> {
                    if (prev == null) {
                        prev = new LongOpenHashSet();
                    }

                    prev.addAll(LongArrayList.wrap(localIds));
                    return prev;
                });
                return null;
            });
        }
    }

}
