package ru.yandex.solomon.coremon.meta.service;

import java.util.Arrays;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.coremon.meta.db.memory.InMemoryMetricsDaoFactory;
import ru.yandex.solomon.coremon.meta.service.cloud.EmptyReferenceResolver;
import ru.yandex.solomon.coremon.meta.service.cloud.EmptyResourceFinder;
import ru.yandex.solomon.labels.protobuf.LabelConverter;
import ru.yandex.solomon.metabase.api.protobuf.CreateManyRequest;
import ru.yandex.solomon.metabase.api.protobuf.CreateOneRequest;
import ru.yandex.solomon.metabase.api.protobuf.DeleteManyRequest;
import ru.yandex.solomon.metabase.api.protobuf.DeleteManyResponse;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metabase.api.protobuf.FindRequest;
import ru.yandex.solomon.metabase.api.protobuf.Metric;
import ru.yandex.solomon.metabase.api.protobuf.ResolveManyRequest;
import ru.yandex.solomon.metabase.api.protobuf.ResolveOneRequest;
import ru.yandex.solomon.metabase.api.protobuf.TLabelNamesRequest;
import ru.yandex.solomon.metabase.api.protobuf.TLabelValuesRequest;
import ru.yandex.solomon.metabase.api.protobuf.TUniqueLabelsRequest;
import ru.yandex.solomon.metrics.client.StockpileClientStub;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.protobuf.Selector;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;
import static ru.yandex.solomon.labels.protobuf.LabelConverter.labelsToProtoList;


/**
 * @author Vladimir Gordiychuk
 */
public class CrossShardMetabaseServiceNotReadyShardTest {
    private static final List<MetabaseShardConf> SHARDS = Arrays.asList(
            MetabaseShardConf.of("junk", "", "foo", "bar", 1, 1),
            MetabaseShardConf.of("junk", "", "man", "test", 2, 1),
            MetabaseShardConf.of("junk", "", "sas", "test", 3, 5)
    );

    private MetabaseService metabaseService;
    private ExecutorService engineExecutor;

    private Metric metric = Metric.newBuilder()
            .addAllLabels(labelsToProtoList(Labels.of(
                    "project", "junk",
                    "cluster", "foo",
                    "service", "bar",
                    "host", "kikimr-003",
                    "sensor", "upTime")))
            .setType(MetricType.COUNTER)
            .build();

    private static Selector selector(String key, String pattern) {
        return Selector.newBuilder().setKey(key).setPattern(pattern).build();
    }

    @Before
    public void setUp() {
        engineExecutor = Executors.newCachedThreadPool();

        StockpileClientStub stockpileForTest = new StockpileClientStub(engineExecutor);
        InMemoryMetricsDaoFactory metricDao = new InMemoryMetricsDaoFactory();
        metricDao.setReplayDelay(1, TimeUnit.HOURS);

        MetabaseShardResolverStub shardResolver = new MetabaseShardResolverStub(SHARDS, metricDao, stockpileForTest);
        metabaseService = new CrossShardMetabaseService(shardResolver, new EmptyReferenceResolver(), new EmptyResourceFinder());
    }

    @After
    public void tearDown() throws Exception {
        engineExecutor.shutdownNow();
    }

    @Test(expected = ShardNotReadyException.class)
    public void createOne() {
        metabaseService.createOne(CreateOneRequest.newBuilder()
                .setMetric(metric)
                .build())
                .join();
    }

    @Test(expected = ShardNotReadyException.class)
    public void createMany() {
        metabaseService.createMany(CreateManyRequest.newBuilder()
                .addAllCommonLabels(LabelConverter.labelsToProtoList(Labels.of("project", "junk", "cluster", "foo", "service", "bar")))
                .addMetrics(metric)
                .build())
                .join();
    }

    @Test(expected = ShardNotReadyException.class)
    public void resolveOne() {
        metabaseService.resolveOne(ResolveOneRequest.newBuilder()
                .addAllLabels(metric.getLabelsList())
                .build())
                .join();
    }

    @Test(expected = ShardNotReadyException.class)
    public void resolveMany() {
        metabaseService.resolveMany(ResolveManyRequest.newBuilder()
                .addAllCommonLabels(LabelConverter.labelsToProtoList(Labels.of("project", "junk", "cluster", "foo", "service", "bar")))
                .addListLabels(ru.yandex.solomon.model.protobuf.Labels.newBuilder()
                        .addAllLabels(metric.getLabelsList())
                        .build())
                .build())
                .join();
    }

    @Test(expected = ShardNotReadyException.class)
    public void deleteMany() {
        DeleteManyResponse response = metabaseService.deleteMany(DeleteManyRequest.newBuilder()
                .addMetrics(metric)
                .build())
                .join();
    }

    @Test
    public void find() {
        expectShardNotReadyException(() -> {
            var request = FindRequest.newBuilder()
                    .addSelectors(selector("service", "test"))
                    .build();
            metabaseService.find(request).join();
        });
    }

    @Test
    public void labelValues() {
        expectShardNotReadyException(() -> {
            var request = TLabelValuesRequest.newBuilder()
                    .addSelectors(selector("project", "junk"))
                    .addLabels("host")
                    .build();
            metabaseService.labelValues(request).join();
        });
    }

    @Test
    public void labelNames() {
        expectShardNotReadyException(() -> {
            var request = TLabelNamesRequest.newBuilder()
                    .addSelectors(selector("project", "junk"))
                    .addSelectors(selector("cluster", "foo"))
                    .addSelectors(selector("service", "bar"))
                    .build();
            metabaseService.labelNames(request).join();
        });
    }

    @Test
    public void uniqueLabels() {
        expectShardNotReadyException(() -> {
            var request = TUniqueLabelsRequest.newBuilder()
                    .addSelectors(selector("project", "junk"))
                    .addSelectors(selector("cluster", "foo"))
                    .addSelectors(selector("service", "bar"))
                    .addNames("host")
                    .build();
            metabaseService.uniqueLabels(request).join();
        });
    }

    private static void expectShardNotReadyException(Runnable fn) {
        try {
            fn.run();
            fail("expected ShardNotReadyException was not thrown");
        } catch (ShardNotReadyException e) {
            var status = GrpcMetabaseService.classifyError(e);
            assertEquals(EMetabaseStatusCode.SHARD_NOT_READY, status.code);
            assertTrue(Pattern.compile("shard .* is not ready").matcher(status.message).matches());
        }
    }
}
