package ru.yandex.solomon.coremon.meta.service;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CyclicBarrier;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.protobuf.ByteString;
import io.netty.buffer.ByteBufUtil;
import io.netty.buffer.UnpooledByteBufAllocator;
import org.hamcrest.CoreMatchers;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.lang.DefaultToString;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.core.ShardIsNotLocalException;
import ru.yandex.solomon.core.db.model.ShardState;
import ru.yandex.solomon.coremon.meta.db.memory.InMemoryMetricsDaoFactory;
import ru.yandex.solomon.coremon.meta.service.cloud.EmptyReferenceResolver;
import ru.yandex.solomon.coremon.meta.service.cloud.EmptyResourceFinder;
import ru.yandex.solomon.labels.protobuf.LabelConverter;
import ru.yandex.solomon.labels.protobuf.LabelSelectorConverter;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.metabase.api.protobuf.CreateManyRequest;
import ru.yandex.solomon.metabase.api.protobuf.CreateManyResponse;
import ru.yandex.solomon.metabase.api.protobuf.CreateOneRequest;
import ru.yandex.solomon.metabase.api.protobuf.CreateOneResponse;
import ru.yandex.solomon.metabase.api.protobuf.DeleteManyRequest;
import ru.yandex.solomon.metabase.api.protobuf.DeleteManyResponse;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metabase.api.protobuf.FindRequest;
import ru.yandex.solomon.metabase.api.protobuf.FindResponse;
import ru.yandex.solomon.metabase.api.protobuf.MetabaseMetricId;
import ru.yandex.solomon.metabase.api.protobuf.MetricNamesRequest;
import ru.yandex.solomon.metabase.api.protobuf.MetricNamesResponse;
import ru.yandex.solomon.metabase.api.protobuf.ResolveManyRequest;
import ru.yandex.solomon.metabase.api.protobuf.ResolveManyResponse;
import ru.yandex.solomon.metabase.api.protobuf.ResolveOneRequest;
import ru.yandex.solomon.metabase.api.protobuf.ResolveOneResponse;
import ru.yandex.solomon.metabase.api.protobuf.TLabelNamesRequest;
import ru.yandex.solomon.metabase.api.protobuf.TLabelNamesResponse;
import ru.yandex.solomon.metabase.api.protobuf.TLabelValues;
import ru.yandex.solomon.metabase.api.protobuf.TLabelValuesRequest;
import ru.yandex.solomon.metabase.api.protobuf.TLabelValuesResponse;
import ru.yandex.solomon.metabase.api.protobuf.TResolveLogsRequest;
import ru.yandex.solomon.metabase.api.protobuf.TResolveLogsResponse;
import ru.yandex.solomon.metabase.api.protobuf.TServerStatusRequest;
import ru.yandex.solomon.metabase.api.protobuf.TSliceOptions;
import ru.yandex.solomon.metabase.api.protobuf.TUniqueLabelsRequest;
import ru.yandex.solomon.metabase.api.protobuf.TUniqueLabelsResponse;
import ru.yandex.solomon.metabase.protobuf.LabelValidationFilter;
import ru.yandex.solomon.metrics.client.StockpileClientStub;
import ru.yandex.solomon.model.protobuf.Label;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.model.protobuf.Selectors;
import ru.yandex.solomon.slog.ResolvedLogMetricsIteratorImpl;
import ru.yandex.solomon.slog.ResolvedLogMetricsRecord;
import ru.yandex.solomon.slog.UnresolvedLogMetaBuilderImpl;
import ru.yandex.solomon.util.protobuf.ByteStrings;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.shard.StockpileShardId;

import static java.util.stream.Collectors.collectingAndThen;
import static org.hamcrest.CoreMatchers.allOf;
import static org.hamcrest.CoreMatchers.containsString;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.everyItem;
import static org.hamcrest.CoreMatchers.hasItem;
import static org.hamcrest.CoreMatchers.hasItems;
import static org.hamcrest.CoreMatchers.not;
import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.emptyIterable;
import static org.hamcrest.Matchers.iterableWithSize;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;
import static org.junit.Assume.assumeThat;
import static ru.yandex.solomon.coremon.meta.service.MetricMatchers.hasLabels;
import static ru.yandex.solomon.coremon.meta.service.MetricMatchers.hasMetric;
import static ru.yandex.solomon.coremon.meta.service.MetricMatchers.hasMetricId;
import static ru.yandex.solomon.labels.protobuf.LabelConverter.labelsToProtoList;
import static ru.yandex.stockpile.client.TestUtil.metricId;

/**
 * @author Vladimir Gordiychuk
 */
public class CrossShardMetabaseServiceTest {
    private static final List<MetabaseShardConf> SHARDS = Arrays.asList(
        MetabaseShardConf.of("junk", "", "foo", "bar", 1, 2),
        MetabaseShardConf.of("junk", "", "man", "test", 2, 0),
        MetabaseShardConf.of("junk", "", "sas", "test", 3, 1),
        MetabaseShardConf.of("junk", "", "sas", "invalid\ntest", 4, 3),
        MetabaseShardConf.of("junk", "", "foo", "service_with_sensor_name", "sensor", 5, 5),
        MetabaseShardConf.of("junk", "folder1", "foo", "service1", "sensor", 6, 5)
    );

    private MetabaseService metabaseService;
    private ExecutorService engineExecutor;
    private MetabaseShardResolverStub resolverStub;

    private static ru.yandex.solomon.model.protobuf.Selector selector(String key, String pattern) {
        return ru.yandex.solomon.model.protobuf.Selector.newBuilder().setKey(key).setPattern(pattern).build();
    }

    private static ru.yandex.solomon.model.protobuf.Selector selector(Selector selector) {
        return LabelSelectorConverter.selectorToProto(selector);
    }

    @Before
    public void setUp() {
        engineExecutor = Executors.newCachedThreadPool();

        StockpileClientStub stockpileForTest = new StockpileClientStub(engineExecutor);
        InMemoryMetricsDaoFactory metricDao = new InMemoryMetricsDaoFactory();

        resolverStub = new MetabaseShardResolverStub(SHARDS, metricDao, stockpileForTest);
        resolverStub.awaitShardsReady();
        metabaseService = new CrossShardMetabaseService(resolverStub, new EmptyReferenceResolver(), new EmptyResourceFinder());
    }

    @After
    public void tearDown() {
        resolverStub.close();
        engineExecutor.shutdownNow();
    }

    @Test
    public void createOne() {
        Labels labels = Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "bar",
                "host", "kikimr-003",
                "sensor", "upTime");

        var metric = ru.yandex.solomon.metabase.api.protobuf.Metric.newBuilder()
                .addAllLabels(labelsToProtoList(labels))
                .setType(ru.yandex.solomon.model.protobuf.MetricType.COUNTER)
                .build();

        CreateOneResponse response = syncCreateOne(CreateOneRequest.newBuilder()
                .setMetric(metric)
                .build());

        assertThat(response.getMetric().getMetricId().getShardId(), not(equalTo(0)));
        assertThat(response.getMetric().getMetricId().getLocalId(), not(equalTo(0)));
        assertThat(response.getMetric(), hasLabels(labels));
    }

    @Test
    public void serverStatus() {
        var one = metabaseService.serverStatus(TServerStatusRequest.newBuilder().build()).join();

        assertNotEquals(0, one.getShardIdsHash());
        int numShardsPartitions = SHARDS.stream()
                .map(shard -> shard.getNumPartitions() > 0 ? shard.getNumPartitions() : 1)
                .reduce(0, Integer::sum);
        assertEquals(numShardsPartitions, one.getPartitionStatusCount());

        assertFalse(one.getTotalPartitionCountKnown());

        var two = metabaseService.serverStatus(TServerStatusRequest.newBuilder()
            .setShardIdsHash(42)
            .build()).join();

        assertEquals(one, two);
        assertFalse(two.getTotalPartitionCountKnown());

        resolverStub.setTotalShardCount(SHARDS.size());

        var tree = metabaseService.serverStatus(TServerStatusRequest.newBuilder()
            .setShardIdsHash(one.getShardIdsHash())
            .build()).join();

        assertEquals(one.getShardIdsHash(), tree.getShardIdsHash());
        assertEquals(0, tree.getPartitionStatusCount());
        assertTrue(tree.getTotalPartitionCountKnown());
        assertEquals(SHARDS.size(), tree.getTotalPartitionCount());
    }

    @Test
    public void createOneInNewFormat() {
        Labels labels = Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "service_with_sensor_name",
            "host", "kikimr-003");

        var metric = ru.yandex.solomon.metabase.api.protobuf.Metric.newBuilder()
            .setName("upTime")
            .addAllLabels(labelsToProtoList(labels))
            .setType(ru.yandex.solomon.model.protobuf.MetricType.COUNTER)
            .build();

        CreateOneResponse response = syncCreateOne(CreateOneRequest.newBuilder()
            .setMetric(metric)
            .build());

        assertThat(response.getMetric().getMetricId().getShardId(), not(equalTo(0)));
        assertThat(response.getMetric().getMetricId().getLocalId(), not(equalTo(0)));
        assertThat(response.getMetric(), hasLabels(labels));
        assertThat(response.getMetric().getName(), equalTo("upTime"));
    }

    @Test(expected = RuntimeException.class)
    public void createOneWithoutMetricNameLabel() {
        Labels labels = Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "service_with_sensor_name",
                "host", "kikimr-003");

        var metric = ru.yandex.solomon.metabase.api.protobuf.Metric.newBuilder()
                .addAllLabels(labelsToProtoList(labels))
                .setType(ru.yandex.solomon.model.protobuf.MetricType.COUNTER)
                .build();

        syncCreateOne(CreateOneRequest.newBuilder()
                .setMetric(metric)
                .build());
    }

    @Test
    public void createMany() {
        Labels commonLabels = Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "bar",
                "host", "cluster");

        Labels idleTime = Labels.of("sensor", "idleTime");
        Labels useTime = Labels.of("sensor", "useTime");

        CreateManyResponse response = syncCreateMany(CreateManyRequest.newBuilder()
                .addAllCommonLabels(labelsToProtoList(commonLabels))
                .addMetrics(ru.yandex.solomon.metabase.api.protobuf.Metric.newBuilder()
                        .addAllLabels(labelsToProtoList(idleTime))
                        .setType(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE)
                        .build())
                .addMetrics(ru.yandex.solomon.metabase.api.protobuf.Metric.newBuilder()
                        .addAllLabels(labelsToProtoList(useTime))
                        .setType(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE)
                        .build())
                .build());

        assertThat(response.getMetricsCount(), equalTo(2));
        assertThat(response.getMetrics(0).getMetricId().getShardId(), not(equalTo(0)));
        assertThat(response.getMetrics(1).getMetricId().getShardId(), not(equalTo(0)));
        assertThat(response.getMetrics(0).getMetricId().getLocalId(), not(equalTo(response.getMetrics(1).getMetricId().getLocalId())));
        assertThat(response.getMetricsList(),
                allOf(
                        hasItem(hasLabels(idleTime.addAll(commonLabels))),
                        hasItem(hasLabels(useTime.addAll(commonLabels)))
                )
        );
    }

    @Test
    public void createManyInNewFormat() {
        Labels commonLabels = Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "service_with_sensor_name",
                "host", "cluster");

        CreateManyResponse response = syncCreateMany(CreateManyRequest.newBuilder()
                .addAllCommonLabels(labelsToProtoList(commonLabels))
                .addMetrics(ru.yandex.solomon.metabase.api.protobuf.Metric.newBuilder()
                        .setName("idleTime")
                        .setType(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE)
                        .build())
                .addMetrics(ru.yandex.solomon.metabase.api.protobuf.Metric.newBuilder()
                        .setName("useTime")
                        .setType(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE)
                        .build())
                .build());

        assertThat(response.getMetricsCount(), equalTo(2));
        assertThat(response.getMetrics(0).getMetricId().getShardId(), not(equalTo(0)));
        assertThat(response.getMetrics(1).getMetricId().getShardId(), not(equalTo(0)));
        assertThat(response.getMetrics(0).getMetricId().getLocalId(), not(equalTo(response.getMetrics(1).getMetricId().getLocalId())));

       assertThat(response.getMetricsList(),
           allOf(
               hasItem(hasMetric("idleTime", commonLabels)),
               hasItem(hasMetric("useTime", commonLabels))
           )
       );
    }

    @Test(expected = RuntimeException.class)
    public void createManyWithoutMetricName() {
        Labels commonLabels = Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "service_with_sensor_name",
                "host", "cluster");

        Labels idleTime = Labels.of("path", "idleTime");
        Labels useTime = Labels.of("path", "useTime");

        CreateManyResponse response = syncCreateMany(CreateManyRequest.newBuilder()
                .addAllCommonLabels(labelsToProtoList(commonLabels))
                .addMetrics(ru.yandex.solomon.metabase.api.protobuf.Metric.newBuilder()
                        .addAllLabels(labelsToProtoList(idleTime))
                        .setType(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE)
                        .build())
                .addMetrics(ru.yandex.solomon.metabase.api.protobuf.Metric.newBuilder()
                        .addAllLabels(labelsToProtoList(useTime))
                        .setType(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE)
                        .build())
                .build());
    }

    @Test
    public void resolveOne() {
        Labels labels = Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "kikimr-003",
            "sensor", "upTime");

        createMetric(1, 111, labels);

        createMetric(
                2, 222,
                Labels.of(
                        "project", "junk",
                        "cluster", "foo",
                        "service", "bar",
                        "host", "kikimr-006",
                        "sensor", "upTime"));

        ResolveOneResponse response = syncResolveOne(ResolveOneRequest.newBuilder()
                .addAllLabels(labelsToProtoList(labels))
                .build());

        assertThat(response.getMetric(),
                allOf(
                        hasMetricId(metricId(1, 111)),
                        hasLabels(labels)
                )
        );
    }

    @Test
    public void resolveOneInNewFormat() {
        Labels uptimeLabels = Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "service_with_sensor_name",
            "host", "kikimr-003");

        createMetric(1, 111, "upTime", uptimeLabels);

        createMetric(
            2, 222,
            "upTime",
            Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "service_with_sensor_name",
                "host", "kikimr-006"));

        ResolveOneResponse response = syncResolveOne(ResolveOneRequest.newBuilder()
            .setName("upTime")
            .addAllLabels(labelsToProtoList(uptimeLabels))
            .build());

        assertThat(response.getMetric(),
            allOf(
                hasMetricId(metricId(1, 111)),
                hasMetric("upTime", uptimeLabels)
            )
        );
    }

    @Test
    public void resolveMany() {
        Labels commonLabels = Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "bar",
                "host", "cluster");

        Labels idleTime = Labels.of("sensor", "idleTime");
        Labels idleTimeFull = idleTime.addAll(commonLabels);
        createMetric(1, 22, idleTimeFull);

        Labels useTime = Labels.of("sensor", "useTime");
        Labels useTimeFull = useTime.addAll(commonLabels);
        createMetric(3, 44, useTimeFull);

        Labels noise = Labels.of("sensor", "noiseTime");
        Labels noiseFull = noise.addAll(commonLabels);
        createMetric(5, 66, noiseFull);

        ResolveManyResponse response = syncResolveMany(ResolveManyRequest.newBuilder()
                .addAllCommonLabels(labelsToProtoList(commonLabels))
                .addListLabels(LabelConverter.labelsToProto(idleTime))
                .addListLabels(LabelConverter.labelsToProto(useTime))
                .addListLabels(LabelConverter.labelsToProto(Labels.of("sensor", "not_exists")))
                .build());

        assertThat(response.getMetricsCount(), equalTo(2));
        assertThat(response.getMetricsList(),
                allOf(
                        hasItem(allOf(
                                hasMetricId(metricId(1, 22)),
                                hasLabels(idleTimeFull)
                        )),
                        hasItem(allOf(
                                hasMetricId(metricId(3, 44)),
                                hasLabels(useTimeFull)
                        ))
                )
        );
    }

    @Test
    public void resolveManyInNewFormat() {
        Labels commonLabels = Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "service_with_sensor_name",
                "host", "cluster");

        createMetric(1, 22, "idleTime", commonLabels);
        createMetric(3, 44, "useTime", commonLabels);
        createMetric(5, 66, "noiseTime", commonLabels);

        ResolveManyResponse response = syncResolveMany(ResolveManyRequest.newBuilder()
            .addAllCommonLabels(labelsToProtoList(commonLabels))
            .addMetrics(MetabaseMetricId.newBuilder().setName("idleTime").build())
            .addMetrics(MetabaseMetricId.newBuilder().setName("useTime").build())
            .addMetrics(MetabaseMetricId.newBuilder().setName("not_exists").build())
            .build());

        assertThat(response.getMetricsCount(), equalTo(2));
        assertThat(response.getMetricsList(),
                allOf(
                        hasItem(allOf(
                                hasMetricId(metricId(1, 22)),
                                hasMetric("idleTime", commonLabels)
                        )),
                        hasItem(allOf(
                                hasMetricId(metricId(3, 44)),
                                hasMetric("useTime", commonLabels)
                        ))
                )
        );
    }

    @Test
    public void findInWriteOnly() {
        createMetric(1, 1, Labels.of(
                "project", "junk",
                "cluster", "sas",
                "service", "test",
                "host", "kikimr-003",
                "sensor", "upTime"));

        createMetric(1, 2, Labels.of(
                "project", "junk",
                "cluster", "man",
                "service", "test",
                "host", "kikimr-006",
                "sensor", "upTime"));

        createMetric(1, 3, Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "bar",
                "host", "cluster",
                "sensor", "upTime"));

        resolverStub.resolveShard(new ShardKey("junk", "foo", "bar"))
                .setShardState(ShardState.WRITE_ONLY);

        {
            var resp = metabaseService.find(FindRequest.newBuilder()
                    .addSelectors(selector("project", "junk"))
                    .addSelectors(selector("cluster", "man|sas"))
                    .addSelectors(selector("service", "test"))
                    .addSelectors(selector("host", "*"))
                    .addSelectors(selector("sensor", "upTime"))
                    .build())
                    .join();

            Assert.assertEquals(EMetabaseStatusCode.OK, resp.getStatus());
        }

        resolverStub.resolveShard(new ShardKey("junk", "man", "test"))
                .setShardState(ShardState.WRITE_ONLY);

        {
            var result = metabaseService.find(FindRequest.newBuilder()
                            .addSelectors(selector("project", "junk"))
                            .addSelectors(selector("cluster", "foo"))
                            .addSelectors(selector("service", "bar"))
                            .addSelectors(selector("host", "*"))
                            .addSelectors(selector("sensor", "upTime"))
                            .build())
                    .join().getMetricsList();
            assertEquals(List.of(), result);
        }

        resolverStub.resolveShard(new ShardKey("junk", "man", "test"))
                .setShardState(ShardState.ACTIVE);

        {
            var resp = metabaseService.find(FindRequest.newBuilder()
                    .addSelectors(selector("project", "junk"))
                    .addSelectors(selector("cluster", "man|sas"))
                    .addSelectors(selector("service", "test"))
                    .addSelectors(selector("host", "*"))
                    .addSelectors(selector("sensor", "upTime"))
                    .build())
                    .join();

            Assert.assertEquals(EMetabaseStatusCode.OK, resp.getStatus());
            System.out.println();
        }
    }

    @Test
    public void resolveInWriteOnly() {
        createMetric(1, 1, Labels.of(
                "project", "junk",
                "cluster", "sas",
                "service", "test",
                "host", "kikimr-003",
                "sensor", "upTime"));

        Labels needle = Labels.of(
                "project", "junk",
                "cluster", "man",
                "service", "test",
                "host", "kikimr-006",
                "sensor", "upTime");

        createMetric(1, 2, needle);

        createMetric(1, 3, Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "bar",
                "host", "cluster",
                "sensor", "upTime"));

        resolverStub.resolveShard(new ShardKey("junk", "foo", "bar"))
                .setShardState(ShardState.WRITE_ONLY);

        {
            var resp = metabaseService.resolveOne(ResolveOneRequest.newBuilder()
                    .addAllLabels(labelsToProtoList(needle))
                    .build())
                    .join();

            Assert.assertEquals(EMetabaseStatusCode.OK, resp.getStatus());
            Assert.assertEquals(1, resp.getMetric().getMetricId().getShardId());
            Assert.assertEquals(2, resp.getMetric().getMetricId().getLocalId());
        }

        resolverStub.resolveShard(new ShardKey("junk", "man", "test"))
                .setShardState(ShardState.WRITE_ONLY);

        {
            try {
                var resp = metabaseService.resolveOne(ResolveOneRequest.newBuilder()
                        .addAllLabels(labelsToProtoList(needle))
                        .build())
                        .join();

                assertEquals(EMetabaseStatusCode.SHARD_WRITE_ONLY, resp.getStatus());
            } catch (ShardWriteOnlyException e) {
                // ok
            }
        }

        resolverStub.resolveShard(new ShardKey("junk", "man", "test"))
                .setShardState(ShardState.ACTIVE);

        {
            var resp = metabaseService.resolveOne(ResolveOneRequest.newBuilder()
                    .addAllLabels(labelsToProtoList(needle))
                    .build())
                    .join();

            Assert.assertEquals(EMetabaseStatusCode.OK, resp.getStatus());
            Assert.assertEquals(1, resp.getMetric().getMetricId().getShardId());
            Assert.assertEquals(2, resp.getMetric().getMetricId().getLocalId());
        }
    }

    @Test
    public void findAndDelete() {
        Labels first = Labels.of(
            "project", "junk",
            "cluster", "sas",
            "service", "test",
            "host", "kikimr-003",
            "sensor", "upTime");

        createMetric(1, 1, first);

        Labels second = Labels.of(
            "project", "junk",
            "cluster", "man",
            "service", "test",
            "host", "kikimr-006",
            "sensor", "upTime");

        createMetric(1, 2, second);

        createMetric(
                1, 3,
                Labels.of(
                        "project", "junk",
                        "cluster", "foo",
                        "service", "bar",
                        "host", "cluster",
                        "sensor", "upTime"));

        FindResponse findResponse = syncFind(FindRequest.newBuilder()
                .addSelectors(selector("project", "junk"))
                .addSelectors(selector("cluster", "man|sas"))
                .addSelectors(selector("service", "test"))
                .addSelectors(selector("host", "*"))
                .addSelectors(selector("sensor", "upTime"))
                .build());

        DeleteManyResponse response = syncDeleteMany(DeleteManyRequest.newBuilder()
                .addAllMetrics(findResponse.getMetricsList())
                .build());

        assertThat(response.getMetricsCount(), equalTo(2));
        assertThat(response.getMetricsList(), allOf(
                hasItem(allOf(
                        hasMetricId(metricId(1, 1)),
                        hasLabels(first)
                )),
                hasItem(allOf(
                        hasMetricId(metricId(1, 2)),
                        hasLabels(second)
                ))
        ));
    }

    @Test
    public void findInNewFormat() {
        Labels first = Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "service_with_sensor_name",
            "host", "kikimr-003");

        createMetric(1, 1, "upTime", first);

        createMetric(
            1, 2,
            "upTime",
            Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "service_with_sensor_name",
                "host", "kikimr-006"));

        Selectors protoSelectors = Selectors.newBuilder()
            .setNameSelector("upTime")
            .addLabelSelectors(selector("project", "junk"))
            .addLabelSelectors(selector("cluster", "foo"))
            .addLabelSelectors(selector("service", "service_with_sensor_name"))
            .addLabelSelectors(selector("host", "kikimr-003"))
            .build();

        FindResponse response = syncFind(FindRequest.newBuilder()
            .setNewSelectors(protoSelectors)
            .build());

        assertThat(response.getMetricsCount(), equalTo(1));
        assertThat(response.getMetrics(0),
            allOf(
                hasMetricId(metricId(1, 1)),
                hasMetric("upTime", first)
            )
        );
    }

    @Test
    public void findWithFolderId() {
        Labels first = Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "service1",
            "host", "kikimr-003");

        createMetric(1, 1, "upTime", first);

        Selectors protoSelectors = Selectors.newBuilder()
            .setNameSelector("upTime")
            .addLabelSelectors(selector("project", "junk"))
            .addLabelSelectors(selector("cluster", "foo"))
            .addLabelSelectors(selector("service", "service1"))
            .addLabelSelectors(selector("host", "kikimr-003"))
            .build();

        FindResponse response = syncFind(FindRequest.newBuilder()
            .setFolderId("folder1")
            .setNewSelectors(protoSelectors)
            .build());

        assertThat(response.getMetricsCount(), equalTo(1));
        assertThat(response.getMetrics(0),
            allOf(
                hasMetricId(metricId(1, 1)),
                hasMetric("upTime", first)
            )
        );
    }

    @Test
    public void findAndDeleteInNewFormat() {
        Labels first = Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "service_with_sensor_name",
            "host", "kikimr-003");

        createMetric(1, 1, "upTime", first);

        Labels second = Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "service_with_sensor_name",
            "host", "kikimr-006");

        createMetric(1, 2, "upTime", second);

        createMetric(1, 3, "upTime", Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "service_with_sensor_name",
            "host", "cluster"));

        Selectors protoSelectors = Selectors.newBuilder()
            .setNameSelector("upTime")
            .addLabelSelectors(selector("project", "junk"))
            .addLabelSelectors(selector("cluster", "foo"))
            .addLabelSelectors(selector("service", "service_with_sensor_name"))
            .addLabelSelectors(selector("host", "kikimr-003|kikimr-006"))
            .build();

        FindResponse findResponse = syncFind(FindRequest.newBuilder()
            .setNewSelectors(protoSelectors)
            .build());

        DeleteManyResponse response = syncDeleteMany(DeleteManyRequest.newBuilder()
            .addAllMetrics(findResponse.getMetricsList())
            .build());

        assertThat(response.getMetricsCount(), equalTo(2));
        assertThat(response.getMetricsList(), allOf(
            hasItem(allOf(
                hasMetricId(metricId(1, 1)),
                hasMetric("upTime", first)
            )),
            hasItem(allOf(
                hasMetricId(metricId(1, 2)),
                hasMetric("upTime", second)
            ))
        ));
    }

    @Test
    public void exactlySearch() {
        Labels first = Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "kikimr-003",
            "sensor", "upTime");

        createMetric(1, 1, first);

        createMetric(
                1, 2,
                Labels.of(
                        "project", "junk",
                        "cluster", "foo",
                        "service", "bar",
                        "host", "kikimr-006",
                        "sensor", "upTime"));

        FindResponse response = syncFind(FindRequest.newBuilder()
                .addSelectors(selector("project", "junk"))
                .addSelectors(selector("cluster", "foo"))
                .addSelectors(selector("service", "bar"))
                .addSelectors(selector("host", "kikimr-003"))
                .addSelectors(selector("sensor", "upTime"))
                .build());

        assertThat(response.getMetricsCount(), equalTo(1));
        assertThat(response.getMetrics(0),
                allOf(
                        hasMetricId(metricId(1, 1)),
                        hasLabels(first)
                )
        );
    }

    @Test
    public void matchByPart() {
        createMetric(
                2, 1,
                Labels.of(
                        "project", "junk",
                        "cluster", "foo",
                        "service", "bar",
                        "host", "kikimr-003",
                        "sensor", "upTime"));

        Labels second = Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "kikimr-006",
            "sensor", "upTime");

        createMetric(2, 2, second);

        FindResponse response = syncFind(FindRequest.newBuilder()
                .addSelectors(selector("project", "junk"))
                .addSelectors(selector("host", "kikimr-006"))
                .build());

        assertThat(response.getMetricsCount(), equalTo(1));
        assertThat(response.getMetrics(0),
                allOf(
                        hasMetricId(metricId(2, 2)),
                        hasLabels(second)
                )
        );
    }

    @Test
    public void matchesSingleGlob() {
        Labels first = Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "kikimr-003",
            "sensor", "upTime");

        createMetric(1, 1, first);

        Labels second = Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "kikimr-006",
            "sensor", "upTime");

        createMetric(1, 2, second);

        createMetric(
            1, 3,
            Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "bar",
                "host", "cluster",
                "sensor", "upTime"));

        FindResponse response = syncFind(FindRequest.newBuilder()
            .addSelectors(selector("project", "junk"))
            .addSelectors(selector("host", "kikimr-*"))
            .build());

        assertThat(response.getMetricsCount(), equalTo(2));
        assertThat(response.getMetricsList(),
            allOf(
                hasItem(allOf(
                    hasMetricId(metricId(1, 2)),
                    hasLabels(second))),
                hasItem(allOf(
                    hasMetricId(metricId(1, 1)),
                    hasLabels(first)))
            )
        );
    }

    @Test
    public void matchesMultiGlob() {
        Labels first = Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "kikimr-003",
            "sensor", "upTime");

        createMetric(1, 1L, first);

        Labels second = Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "kikimr-006",
            "sensor", "upTime");

        createMetric(1, 2, second);

        createMetric(
            1, 3,
            Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "bar",
                "host", "cluster",
                "sensor", "upTime"));

        FindResponse response = syncFind(FindRequest.newBuilder()
                .addSelectors(selector("project", "junk"))
                .addSelectors(selector("host", "kikimr-003|kikimr-006"))
                .build());

        assertThat(response.getMetricsCount(), equalTo(2));
        assertThat(response.getMetricsList(),
                allOf(
                        hasItem(allOf(
                                hasMetricId(metricId(1, 2)),
                                hasLabels(second))),
                        hasItem(allOf(
                                hasMetricId(metricId(1, 1)),
                                hasLabels(first)))
                )
        );
    }

    @Test
    public void matchesAbsent() {
        createMetric(1, 1L, Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "kikimr-001",
            "sensor", "upTime"));

        createMetric(1, 2, Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "kikimr-002",
            "sensor", "upTime"));

        Labels third = Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "sensor", "count");

        createMetric(1, 3, third);

        FindResponse response = syncFind(FindRequest.newBuilder()
            .addSelectors(selector("project", "junk"))
            .addSelectors(selector(Selector.absent("host")))
            .build());

        assertThat(response.getMetricsCount(), equalTo(1));

        assertThat(response.getMetricsList(), hasItem(
            allOf(
                hasMetricId(metricId(1, 3)),
                hasLabels(third)
            )
        ));
    }

    @Test
    public void matchesAny() {
        createMetric(1, 1L, Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "kikimr-001",
            "sensor", "upTime"));

        createMetric(1, 2, Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "kikimr-002",
            "sensor", "upTime"));

        createMetric(
            1, 3,
            Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "bar",
                "host", "cluster",
                "sensor", "upTime"));

        FindResponse response = syncFind(FindRequest.newBuilder()
            .addSelectors(selector("project", "junk"))
            .addSelectors(selector(Selector.any("host")))
            .build());

        assertThat(response.getMetricsCount(), equalTo(3));
    }

    @Test
    public void matchesExact() {
        createMetric(1, 1L, Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "kikimr-001",
            "sensor", "upTime"));

        createMetric(1, 2, Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "kikimr-002",
            "sensor", "upTime"));

        Labels third = Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "cluster",
            "sensor", "upTime");

        createMetric(1, 3, third);

        FindResponse response = syncFind(FindRequest.newBuilder()
            .addSelectors(selector("project", "junk"))
            .addSelectors(selector(Selector.exact("host", "cluster")))
            .build());

        assertThat(response.getMetricsCount(), equalTo(1));

        assertThat(response.getMetricsList(), hasItem(
            allOf(
                hasMetricId(metricId(1, 3)),
                hasLabels(third)
            )
        ));
    }

    @Test
    public void matchesGlob() {
        Labels first = Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "kikimr-001",
            "sensor", "upTime");

        createMetric(1, 1L, first);

        Labels second = Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "kikimr-002",
            "sensor", "upTime");

        createMetric(1, 2, second);

        createMetric(
            1, 3,
            Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "bar",
                "host", "cluster",
                "sensor", "upTime"));

        FindResponse response = syncFind(FindRequest.newBuilder()
            .addSelectors(selector("project", "junk"))
            .addSelectors(selector(Selector.glob("host", "kikimr-*")))
            .build());

        assertThat(response.getMetricsCount(), equalTo(2));

        assertThat(response.getMetricsList(),allOf(
            hasItem(allOf(
                hasMetricId(metricId(1, 1)),
                hasLabels(first)
            )),
            hasItem(allOf(
                hasMetricId(metricId(1, 2)),
                hasLabels(second)
            ))
        ));
    }

    @Test
    public void matchesNotGlob() {
        createMetric(
            1, 1,
            Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "bar",
                "host", "kikimr-003",
                "sensor", "upTime"));

        createMetric(
            1, 2,
            Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "bar",
                "host", "kikimr-006",
                "sensor", "upTime"));

        Labels third = Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "cluster",
            "sensor", "upTime");

        createMetric(1, 3, third);

        FindResponse response = syncFind(FindRequest.newBuilder()
            .addSelectors(selector("project", "junk"))
            .addSelectors(selector(Selector.notGlob("host", "kikimr-*")))
            .build());

        assertThat(response.getMetricsCount(), equalTo(1));
        assertThat(response.getMetricsList(),
            hasItem(allOf(
                hasMetricId(metricId(1, 3)),
                hasLabels(third)))
        );
    }

    @Test
    public void matchesRegex() {
        Labels first = Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "kikimr-03",
            "sensor", "upTime");

        createMetric(1, 1L, first);

        createMetric(1, 2, Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "kikimr-006",
            "sensor", "upTime"));

        createMetric(
            1, 3,
            Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "bar",
                "host", "cluster",
                "sensor", "upTime"));

        FindResponse response = syncFind(FindRequest.newBuilder()
            .addSelectors(selector("project", "junk"))
            .addSelectors(selector(Selector.regex("host", "kikimr-\\d\\d")))
            .build());

        assertThat(response.getMetricsCount(), equalTo(1));
        assertThat(response.getMetricsList(),
            hasItem(allOf(
                hasMetricId(metricId(1, 1)),
                hasLabels(first)))
        );
    }

    @Test
    public void findWithOffset() {
        Labels first = Labels.of(
            "project", "junk",
            "cluster", "man",
            "service", "test",
            "host", "kikimr-010",
            "sensor", "upTime");

        createMetric(1, 1, first);

        Labels second = Labels.of(
            "project", "junk",
            "cluster", "man",
            "service", "test",
            "host", "kikimr-011",
            "sensor", "upTime");

        createMetric(1, 2, second);

        createMetric(
            1, 3,
            Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "bar",
                "host", "cluster",
                "sensor", "upTime"));

        FindResponse findResponse = syncFind(FindRequest.newBuilder()
            .setSliceOptions(TSliceOptions.newBuilder().setOffset(1).setLimit(-1).build())
            .addSelectors(selector("service", "test"))
            .build());

        // order is not guaranteed

        assertThat(findResponse.getMetricsCount(), equalTo(1));
        assertThat(findResponse.getMetrics(0), CoreMatchers.anyOf(
            allOf(
                hasMetricId(metricId(1, 1)),
                hasLabels(first)),
            allOf(
                hasMetricId(metricId(1, 2)),
                hasLabels(second))
        ));

        assertThat(findResponse.getTotalCount(), equalTo(2));
    }

    @Test
    public void findWithTooLongOffset() {
        Labels first = Labels.of(
            "project", "junk",
            "cluster", "man",
            "service", "test",
            "host", "kikimr-010",
            "sensor", "upTime");

        createMetric(1, 1, first);

        createMetric(
            1, 2,
            Labels.of(
                "project", "junk",
                "cluster", "man",
                "service", "test",
                "host", "kikimr-011",
                "sensor", "upTime"));

        createMetric(
            1, 3,
            Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "bar",
                "host", "cluster",
                "sensor", "upTime"));

        FindResponse findResponse = syncFind(FindRequest.newBuilder()
            .setSliceOptions(TSliceOptions.newBuilder().setOffset(10).setLimit(-1).build())
            .addSelectors(selector("service", "test"))
            .build());

        assertThat(findResponse.getMetricsCount(), equalTo(0));
        assertThat(findResponse.getTotalCount(), equalTo(2));
    }

    @Test
    public void findWithZeroLimit() {
        createMetric(
            1, 1,
            Labels.of(
                "project", "junk",
                "cluster", "sas",
                "service", "test",
                "host", "kikimr-012",
                "sensor", "upTime"));

        createMetric(
            1, 2,
            Labels.of(
                "project", "junk",
                "cluster", "man",
                "service", "test",
                "host", "kikimr-013",
                "sensor", "upTime"));

        createMetric(
            1, 3,
            Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "bar",
                "host", "cluster",
                "sensor", "upTime"));

        FindResponse findResponse = syncFind(FindRequest.newBuilder()
            .setSliceOptions(TSliceOptions.newBuilder().setOffset(15).setLimit(0).build())
            .addSelectors(selector("service", "test"))
            .build());

        assertThat(findResponse.getMetricsCount(), equalTo(0));
        assertThat(findResponse.getTotalCount(), equalTo(2));
    }

    @Test
    public void findWithLimit() {
        createMetric(
            1, 1,
            Labels.of(
                "project", "junk",
                "cluster", "sas",
                "service", "test",
                "host", "kikimr-014",
                "sensor", "upTime"));

        createMetric(
            1, 2,
            Labels.of(
                "project", "junk",
                "cluster", "man",
                "service", "test",
                "host", "kikimr-015",
                "sensor", "upTime"));

        createMetric(
            1, 3,
            Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "bar",
                "host", "cluster",
                "sensor", "upTime"));

        createMetric(
            1, 4,
            Labels.of(
                "project", "junk",
                "cluster", "man",
                "service", "test",
                "host", "kikimr-016",
                "sensor", "upTime"));

        FindResponse findResponse = syncFind(FindRequest.newBuilder()
            .setSliceOptions(TSliceOptions.newBuilder().setLimit(2).build())
            .addSelectors(selector("service", "test"))
            .build());

        assertThat(findResponse.getMetricsCount(), equalTo(2));
        assertThat(findResponse.getTotalCount(), equalTo(3));
    }

    @Test
    public void findWithTooLongLimit() {
        createMetric(
            1, 1,
            Labels.of(
                "project", "junk",
                "cluster", "sas",
                "service", "test",
                "host", "kikimr-017",
                "sensor", "upTime"));

        createMetric(
            1, 2,
            Labels.of(
                "project", "junk",
                "cluster", "man",
                "service", "test",
                "host", "kikimr-018",
                "sensor", "upTime"));

        createMetric(
            1, 3,
            Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "bar",
                "host", "cluster",
                "sensor", "upTime"));

        createMetric(
            1, 4,
            Labels.of(
                "project", "junk",
                "cluster", "man",
                "service", "test",
                "host", "kikimr-038",
                "sensor", "upTime"));

        FindResponse findResponse = syncFind(FindRequest.newBuilder()
            .setSliceOptions(TSliceOptions.newBuilder().setLimit(15).build())
            .addSelectors(selector("service", "test"))
            .build());

        assertThat(findResponse.getMetricsCount(), equalTo(3));
        assertThat(findResponse.getTotalCount(), equalTo(3));
    }

    @Test
    public void findWithOffsetAndLimit() {
        Labels first = Labels.of(
            "project", "junk",
            "cluster", "man",
            "service", "test",
            "host", "kikimr-019",
            "sensor", "upTime");

        createMetric(1, 1, first);

        Labels second = Labels.of(
            "project", "junk",
            "cluster", "man",
            "service", "test",
            "host", "kikimr-020",
            "sensor", "upTime");

        createMetric(1, 2, second);

        Labels third = Labels.of(
            "project", "junk",
            "cluster", "man",
            "service", "test",
            "host", "kikimr-021",
            "sensor", "upTime");

        createMetric(1, 3, third);

        createMetric(
            1, 4,
            Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "bar",
                "host", "cluster",
                "sensor", "upTime"));

        FindResponse findResponse = syncFind(FindRequest.newBuilder()
            .setSliceOptions(TSliceOptions.newBuilder().setOffset(1).setLimit(2).build())
            .addSelectors(selector("service", "test"))
            .build());

        // order is not guaranteed

        assertThat(findResponse.getMetricsCount(), equalTo(2));

        for (var metric : findResponse.getMetricsList()) {
            MetricId metricId = metric.getMetricId();
            if (metricId.equals(metricId(1, 1))) {
                assertThat(metric, hasLabels(first));
            } else if (metricId.equals(metricId(1, 2))) {
                assertThat(metric, hasLabels(second));
            } else if (metricId.equals(metricId(1, 3))) {
                assertThat(metric, hasLabels(third));
            } else {
                fail("found wrong sensor: " + metric);
            }
        }

        assertThat(findResponse.getTotalCount(), equalTo(3));
    }

    @Test
    public void metricNames() {
        createMetric("upTime", Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "service_with_sensor_name",
            "host", "kikimr-003"));

        Selectors protoSelectors = Selectors.newBuilder()
            .addLabelSelectors(selector("project", "junk"))
            .addLabelSelectors(selector("cluster", "foo"))
            .addLabelSelectors(selector("service", "service_with_sensor_name"))
            .build();

        MetricNamesResponse response = syncMetricNames(MetricNamesRequest.newBuilder()
            .setSelectors(protoSelectors)
            .build());

        assertThat(response.getNamesList(), allOf(iterableWithSize(1), hasItem("upTime")));
    }

    @Test
    public void metricNamesWithFolder() {
        createMetric("upTime", Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "service1",
            "host", "kikimr-003"));

        Selectors protoSelectors = Selectors.newBuilder()
            .addLabelSelectors(selector("project", "junk"))
            .addLabelSelectors(selector("cluster", "foo"))
            .addLabelSelectors(selector("service", "service1"))
            .build();

        MetricNamesResponse response = syncMetricNames(MetricNamesRequest.newBuilder()
            .setFolderId("folder1")
            .setSelectors(protoSelectors)
            .build());

        assertThat(response.getNamesList(), allOf(iterableWithSize(1), hasItem("upTime")));
    }

    @Test
    public void labelValuesInNewFormat() {
        createMetric("upTime", Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "service_with_sensor_name",
            "host", "kikimr-003"));

        Selectors protoSelectors = Selectors.newBuilder()
            .addLabelSelectors(selector("project", "junk"))
            .addLabelSelectors(selector("cluster", "foo"))
            .addLabelSelectors(selector("service", "service_with_sensor_name"))
            .setNameSelector("upTime")
            .build();

        TLabelValuesResponse response = syncLabelValues(TLabelValuesRequest.newBuilder()
            .setNewSelectors(protoSelectors)
            .build());

        Map<String, TLabelValues> groups = toLabelValuesMap(response);

        assertThat(response.getMetricCount(), equalTo(1));
        assertThat(groups.size(), equalTo(4));

        TLabelValues hostValues = groups.get("host");;
        assertThat(hostValues.getValuesList(), allOf(iterableWithSize(1), hasItem("kikimr-003")));
    }

    @Test
    public void labelValuesWithFolder() {
        createMetric("upTime", Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "service1",
            "host", "kikimr-003"));

        Selectors protoSelectors = Selectors.newBuilder()
            .addLabelSelectors(selector("project", "junk"))
            .addLabelSelectors(selector("cluster", "foo"))
            .addLabelSelectors(selector("service", "service1"))
            .setNameSelector("upTime")
            .build();

        TLabelValuesResponse response = syncLabelValues(TLabelValuesRequest.newBuilder()
            .setFolderId("folder1")
            .setNewSelectors(protoSelectors)
            .build());

        Map<String, TLabelValues> groups = toLabelValuesMap(response);

        assertThat(response.getMetricCount(), equalTo(1));
        assertThat(groups.size(), equalTo(4));

        TLabelValues hostValues = groups.get("host");;
        assertThat(hostValues.getValuesList(), allOf(iterableWithSize(1), hasItem("kikimr-003")));
    }

    @Test
    public void labelValuesByKeyInNewFormat() {
        createMetric("upTime", Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "service_with_sensor_name",
            "host", "kikimr-003"));


        Selectors protoSelectors = Selectors.newBuilder()
            .addLabelSelectors(selector("project", "junk"))
            .addLabelSelectors(selector("cluster", "foo"))
            .addLabelSelectors(selector("service", "service_with_sensor_name"))
            .setNameSelector("upTime")
            .build();

        TLabelValuesResponse response = syncLabelValues(TLabelValuesRequest.newBuilder()
            .setNewSelectors(protoSelectors)
            .addLabels("host")
            .build());

        Map<String, TLabelValues> groups = toLabelValuesMap(response);

        assertThat(response.getMetricCount(), equalTo(1));
        assertThat(groups.size(), equalTo(1));

        TLabelValues hostValues = groups.get("host");
        assertThat(hostValues.getValuesList(), allOf(iterableWithSize(1), hasItem("kikimr-003")));
    }

    @Test
    public void labelValuesAllLabels() {
        createMetric(Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "kikimr-003",
            "sensor", "upTime"));

        createMetric(Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "kikimr-006",
            "sensor", "upTime"));

        createMetric(Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "cluster",
            "sensor", "upTime"));

        TLabelValuesResponse response = syncLabelValues(TLabelValuesRequest.newBuilder()
            .addSelectors(selector("project", "junk"))
            .addSelectors(selector("cluster", "foo"))
            .addSelectors(selector("service", "bar"))
            .addSelectors(selector("sensor", "upTime"))
            .build());

        Map<String, TLabelValues> groups = toLabelValuesMap(response);

        assertThat(response.getMetricCount(), equalTo(3));
        assertThat(groups.size(), equalTo(5));
        assertThat(groups.get("project").getValuesList(), allOf(iterableWithSize(1), hasItem("junk")));
        assertThat(groups.get("cluster").getValuesList(), allOf(iterableWithSize(1), hasItem("foo")));
        assertThat(groups.get("service").getValuesList(), allOf(iterableWithSize(1), hasItem("bar")));

        TLabelValues metricValues = groups.get("sensor");
        assertThat(metricValues.getValuesList(), allOf(iterableWithSize(1), hasItem("upTime")));

        TLabelValues hostValues = groups.get("host");
        assertThat(hostValues.getValuesList(), allOf(
            iterableWithSize(3),
            hasItem("kikimr-003"),
            hasItem("kikimr-006"),
            hasItem("cluster")
        ));
    }

    @Test
    public void labelValuesForProjectSelector() {
        createMetric(Labels.of(
            "project", "junk",
            "cluster", "man",
            "service", "test",
            "host", "kikimr-003",
            "sensor", "upTime"));

        createMetric(Labels.of(
            "project", "junk",
            "cluster", "man",
            "service", "test",
            "host", "kikimr-006",
            "sensor", "upTime"));

        createMetric(Labels.of(
            "project", "junk",
            "cluster", "sas",
            "service", "test",
            "host", "cluster",
            "sensor", "upTime"));

        TLabelValuesResponse response = syncLabelValues(TLabelValuesRequest.newBuilder()
            .addSelectors(selector("project", "junk"))
            .addLabels("cluster")
            .addLabels("service")
            .addSelectors(selector("sensor", "upTime"))
            .build());

        Map<String, TLabelValues> groups = toLabelValuesMap(response);

        assertThat(response.getMetricCount(), equalTo(3));

        assertThat(groups.size(), equalTo(2));
        assertThat(groups.get("cluster").getValuesList(), allOf(iterableWithSize(2), hasItem("man"), hasItem("sas")));
        assertThat(groups.get("service").getValuesList(), allOf(iterableWithSize(1), hasItem("test")));
    }

    @Test
    public void labelValuesForProjectSelectorWithSearch() {
        createMetric(Labels.of(
            "project", "junk",
            "cluster", "man",
            "service", "test",
            "host", "kikimr-003",
            "sensor", "upTime"));

        createMetric(Labels.of(
            "project", "junk",
            "cluster", "man",
            "service", "test",
            "host", "kikimr-006",
            "sensor", "upTime"));

        createMetric(Labels.of(
            "project", "junk",
            "cluster", "sas",
            "service", "test",
            "host", "cluster",
            "sensor", "upTime"));

        TLabelValuesResponse response = syncLabelValues(TLabelValuesRequest.newBuilder()
            .addSelectors(selector("project", "junk"))
            .addLabels("cluster")
            .addLabels("service")
            .setTextSearch("MAN")
            .build());

        Map<String, TLabelValues> groups = toLabelValuesMap(response);

        assertThat(response.getMetricCount(), equalTo(3));

        assertThat(groups.size(), equalTo(2));

        TLabelValues clusterValues = groups.get("cluster");
        assertThat(clusterValues.getAbsent(), equalTo(false));
        assertThat(clusterValues.getMetricCount(), equalTo(3));
        assertThat(clusterValues.getValuesList(), allOf(iterableWithSize(1), hasItems("man")));

        TLabelValues serviceValues = groups.get("service");
        assertThat(serviceValues.getAbsent(), equalTo(false));
        assertThat(serviceValues.getMetricCount(), equalTo(3));
        assertThat(serviceValues.getValuesList(), emptyIterable());
    }

    @Test
    public void labelValuesWithLabelValidation() {
        createMetric(Labels.of(
            "project", "junk",
            "cluster", "sas",
            "service", "test",
            "stream-name", "stream",
            "sensor", "unknown\nsensor"));

        createMetric(Labels.of(
            "project", "junk",
            "cluster", "sas",
            "service", "test",
            "host", "cluster",
            "sensor", "upTime"));

        createMetric(Labels.of(
            "project", "junk",
            "cluster", "sas",
            "service", "invalid\ntest",
            "host", "cluster",
            "sensor", "upTime"));

        TLabelValuesResponse response = syncLabelValues(TLabelValuesRequest.newBuilder()
            .addSelectors(selector("project", "junk"))
            .setValidationFilter(LabelValidationFilter.INVALID_ONLY)
            .build());

        Map<String, TLabelValues> groups = toLabelValuesMap(response);

        assertThat(response.getMetricCount(), equalTo(3));

        assertThat(groups.size(), equalTo(6));

        TLabelValues clusterValues = groups.get("cluster");
        assertThat(clusterValues.getAbsent(), equalTo(false));
        assertThat(clusterValues.getMetricCount(), equalTo(3));
        assertThat(clusterValues.getValuesList(), emptyIterable());

        TLabelValues serviceValues = groups.get("service");
        assertThat(serviceValues.getAbsent(), equalTo(false));
        assertThat(serviceValues.getMetricCount(), equalTo(3));
        assertThat(serviceValues.getValuesList(), allOf(iterableWithSize(1), hasItem("invalid\ntest")));
    }

    @Test
    public void labelValuesAllLabelsWithLimit() {
        createMetric(Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "kikimr-003",
            "sensor", "upTime"));

        createMetric(Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "kikimr-006",
            "sensor", "upTime"));

        createMetric(Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "cluster",
            "sensor", "upTime"));

        TLabelValuesResponse response = syncLabelValues(TLabelValuesRequest.newBuilder()
            .addSelectors(selector("project", "junk"))
            .addSelectors(selector("cluster", "foo"))
            .addSelectors(selector("service", "bar"))
            .addSelectors(selector("sensor", "upTime"))
            .setLimit(2)
            .build());

        Map<String, TLabelValues> groups = toLabelValuesMap(response);

        assertThat(response.getMetricCount(), equalTo(3));
        assertThat(groups.size(), equalTo(5));
        assertThat(groups.get("project").getValuesList(), allOf(iterableWithSize(1), hasItem("junk")));
        assertThat(groups.get("cluster").getValuesList(), allOf(iterableWithSize(1), hasItem("foo")));
        assertThat(groups.get("service").getValuesList(), allOf(iterableWithSize(1), hasItem("bar")));

        TLabelValues metricValues = groups.get("sensor");
        assertThat(metricValues.getAbsent(), equalTo(false));
        assertThat(metricValues.getMetricCount(), equalTo(3));
        assertThat(metricValues.getValuesList(), allOf(iterableWithSize(1), hasItem("upTime")));

        TLabelValues hostValues = groups.get("host");
        assertThat(hostValues.getAbsent(), equalTo(false));
        assertThat(hostValues.getMetricCount(), equalTo(3));
        assertThat(hostValues.getValuesList(), iterableWithSize(2));
    }

    @Test
    public void labelValuesTextSearchDoesntWorkOnShardLabels() {
        createMetric(Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "kikimr-003",
            "sensor", "upTime"));

        TLabelValuesResponse response = syncLabelValues(TLabelValuesRequest.newBuilder()
            .addSelectors(selector("project", "junk"))
            .addSelectors(selector("cluster", "foo"))
            .addSelectors(selector("service", "bar"))
            .addSelectors(selector("sensor", "upTime"))
            .setTextSearch("junk")
            .build());

        assertThat(response.getMetricCount(), equalTo(1));
        assertThat(response.getValuesCount(), equalTo(5));

        Map<String, TLabelValues> groups = toLabelValuesMap(response);

        TLabelValues hostValues = groups.get("host");
        assertThat(hostValues.getAbsent(), equalTo(false));
        assertThat(hostValues.getMetricCount(), equalTo(1));
        assertThat(hostValues.getValuesList(), empty());

        TLabelValues metricValues = groups.get("sensor");
        assertThat(metricValues.getAbsent(), equalTo(false));
        assertThat(metricValues.getMetricCount(), equalTo(1));
        assertThat(metricValues.getValuesList(), empty());
    }

    @Test
    public void labelValuesValueEmptySubstringDoesntAffectSearch() {
        createMetric(Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "kikimr-003",
            "sensor", "upTime"));

        TLabelValuesResponse response = syncLabelValues(TLabelValuesRequest.newBuilder()
            .addSelectors(selector("project", "junk"))
            .addSelectors(selector("cluster", "foo"))
            .addSelectors(selector("service", "bar"))
            .addSelectors(selector("sensor", "upTime"))
            .setTextSearch("")
            .build());

        assertThat(response.getMetricCount(), equalTo(1));
        assertThat(response.getValuesList(), not(empty()));
    }

    @Test
    public void labelValuesAllLabelsWithLimitAndTextSearch() {
        createMetric(Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "kikimr-003",
            "sensor", "upTime"));

        createMetric(Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "kikimr-006",
            "sensor", "downTime"));

        createMetric(Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "kikimr-009",
            "sensor", "iowait"));

        TLabelValuesResponse response = syncLabelValues(TLabelValuesRequest.newBuilder()
            .addSelectors(selector("project", "junk"))
            .addSelectors(selector("cluster", "foo"))
            .addSelectors(selector("service", "bar"))
            .addSelectors(selector("sensor", "*"))
            .setTextSearch("KIKImr")
            .setLimit(2)
            .build());

        Map<String, TLabelValues> groups = toLabelValuesMap(response);

        assertThat(response.getMetricCount(), equalTo(3));
        assertThat(groups.size(), equalTo(5));
        assertThat(groups.get("project").getValuesList(), emptyIterable());
        assertThat(groups.get("cluster").getValuesList(), emptyIterable());
        assertThat(groups.get("service").getValuesList(), emptyIterable());

        TLabelValues metricValues = groups.get("sensor");
        assertThat(metricValues.getAbsent(), equalTo(false));
        assertThat(metricValues.getMetricCount(), equalTo(3));
        assertThat(metricValues.getValuesList(), empty());

        TLabelValues hostValues = groups.get("host");
        assertThat(hostValues.getAbsent(), equalTo(false));
        assertThat(hostValues.getMetricCount(), equalTo(3));
        assertThat(hostValues.getValuesList(), iterableWithSize(2));
    }

    @Test
    public void labelValuesAllLabelsWithTextSearchFoundInLabelKey() {
        createMetric(Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "kikimr-003",
            "sensor", "upTime"));

        createMetric(Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "kikimr-006",
            "sensor", "downTime"));

        createMetric(Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "kikimr-009",
            "sensor", "iowait"));

        TLabelValuesResponse response = syncLabelValues(TLabelValuesRequest.newBuilder()
            .addSelectors(selector("project", "junk"))
            .addSelectors(selector("cluster", "foo"))
            .addSelectors(selector("service", "bar"))
            .addSelectors(selector("sensor", "*"))
            .setTextSearch("HOS")
            .build());

        Map<String, TLabelValues> groups = toLabelValuesMap(response);

        assertThat(response.getMetricCount(), equalTo(3));
        assertThat(groups.size(), equalTo(5));
        assertThat(groups.get("project").getValuesList(), emptyIterable());
        assertThat(groups.get("cluster").getValuesList(), emptyIterable());
        assertThat(groups.get("service").getValuesList(), emptyIterable());

        TLabelValues metricValues = groups.get("sensor");
        assertThat(metricValues.getValuesList(), empty());

        TLabelValues hostValues = groups.get("host");
        assertThat(hostValues.getValuesList(), iterableWithSize(3));
    }

    @Test
    public void labelValuesParticularLabel() {
        createMetric(Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "bar",
                "host", "kikimr-003",
                "sensor", "upTime"));

        createMetric(Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "bar",
                "host", "kikimr-006",
                "sensor", "upTime"));

        // noise
        createMetric(Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "bar",
                "dc", "man",
                "sensor", "upTime"));

        TLabelValuesResponse response = syncLabelValues(TLabelValuesRequest.newBuilder()
                .addSelectors(selector("project", "junk"))
                .addSelectors(selector("cluster", "foo"))
                .addSelectors(selector("service", "bar"))
                .addSelectors(selector("sensor", "upTime"))
                .addLabels("host")
                .build());

        assertThat(response.getValuesCount(), equalTo(1));
        TLabelValues hostValus = response.getValues(0);
        assertThat(hostValus.getName(), equalTo("host"));
        assertThat(hostValus.getValuesList(),
                allOf(
                        iterableWithSize(2),
                        hasItem("kikimr-003"),
                        hasItem("kikimr-006")
                )
        );
    }

    @Test
    public void labelValuesParticularLabelWithLimit() {
        createMetric(Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "solomon-005",
            "sensor", "upTime"));

        createMetric(Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "solomon-010",
            "sensor", "upTime"));

        // noise
        createMetric(Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "dc", "man",
            "sensor", "upTime"));

        TLabelValuesResponse response = syncLabelValues(TLabelValuesRequest.newBuilder()
            .addSelectors(selector("project", "junk"))
            .addSelectors(selector("cluster", "foo"))
            .addSelectors(selector("service", "bar"))
            .addSelectors(selector("sensor", "upTime"))
            .addLabels("host")
            .setLimit(1)
            .build());

        assertThat(response.getValuesCount(), equalTo(1));

        TLabelValues hostValues = response.getValues(0);
        assertThat(hostValues.getName(), equalTo("host"));
        assertThat(hostValues.getAbsent(), equalTo(true));
        assertThat(hostValues.getMetricCount(), equalTo(2));
        assertThat(hostValues.getValuesList(), iterableWithSize(1));
    }

    @Test
    public void allLabelValuesInsideShard() {
        createMetric(Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "bar",
                "host", "kikimr-003",
                "sensor", "upTime"));

        createMetric(Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "bar",
                "host", "kikimr-006",
                "sensor", "upTime"));

        createMetric(Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "bar",
                "path", "/myPath"));

        TLabelValuesResponse response = syncLabelValues(TLabelValuesRequest.newBuilder()
                .addSelectors(selector("project", "junk"))
                .addSelectors(selector("cluster", "foo"))
                .addSelectors(selector("service", "bar"))
                .build());

        Map<String, TLabelValues> groups = toLabelValuesMap(response);

        TLabelValues metricValues = groups.get("sensor");
        assertThat(metricValues.getValuesList(), allOf(iterableWithSize(1), hasItem("upTime")));

        TLabelValues hostValues = groups.get("host");
        assertThat(hostValues.getValuesList(), allOf(
                iterableWithSize(2),
                hasItem("kikimr-003"),
                hasItem("kikimr-006")
        ));

        TLabelValues pathValues = groups.get("path");
        assertThat(pathValues.getValuesList(), allOf(iterableWithSize(1), hasItem("/myPath")));
    }

    @Test
    public void labelValuesCrossShardParticularLabel() {
        createMetric(Labels.of(
                "project", "junk",
                "cluster", "man",
                "service", "test",
                "host", "man-test-01",
                "sensor", "upTime"));

        createMetric(Labels.of(
                "project", "junk",
                "cluster", "man",
                "service", "test",
                "host", "man-test-02",
                "sensor", "upTime"));

        createMetric(Labels.of(
                "project", "junk",
                "cluster", "sas",
                "service", "test",
                "host", "sas-test-01",
                "sensor", "upTime"));

        TLabelValuesResponse response = syncLabelValues(TLabelValuesRequest.newBuilder()
                .addSelectors(selector("project", "junk"))
                .addSelectors(selector("service", "test"))
                .addSelectors(selector("sensor", "upTime"))
                .addLabels("host")
                .addLabels("cluster")
                .build());

        assertThat(response.getMetricCount(), equalTo(3));
        Map<String, TLabelValues> groups = toLabelValuesMap(response);

        TLabelValues clusterValues = groups.get("cluster");
        assertThat(clusterValues.getValuesList(), allOf(iterableWithSize(2), hasItems("man", "sas")));

        TLabelValues hostValues = groups.get("host");
        assertThat(hostValues.getValuesList(),
                allOf(
                        iterableWithSize(3),
                        hasItem("sas-test-01"),
                        hasItem("man-test-01"),
                        hasItem("man-test-02")
                )
        );
    }

    @Test
    public void labelValuesCrossShardParticularLabelWithLimit() {
        createMetric(Labels.of(
            "project", "junk",
            "cluster", "man",
            "service", "test",
            "host", "man-test-01",
            "sensor", "upTime"));

        createMetric(Labels.of(
            "project", "junk",
            "cluster", "man",
            "service", "test",
            "host", "man-test-02",
            "sensor", "upTime"));

        createMetric(Labels.of(
            "project", "junk",
            "cluster", "sas",
            "service", "test",
            "host", "sas-test-01",
            "sensor", "upTime"));

        TLabelValuesResponse response = syncLabelValues(TLabelValuesRequest.newBuilder()
            .addSelectors(selector("project", "junk"))
            .addSelectors(selector("service", "test"))
            .addSelectors(selector("sensor", "upTime"))
            .addLabels("host")
            .addLabels("cluster")
            .setLimit(1)
            .build());

        assertThat(response.getMetricCount(), equalTo(3));
        Map<String, TLabelValues> groups = toLabelValuesMap(response);

        TLabelValues clusterValues = groups.get("cluster");
        assertThat(clusterValues.getAbsent(), equalTo(false));
        assertThat(clusterValues.getMetricCount(), equalTo(3));
        assertThat(clusterValues.getValuesList(), iterableWithSize(1));

        TLabelValues hostValues = groups.get("host");
        assertThat(hostValues.getAbsent(), equalTo(false));
        assertThat(hostValues.getMetricCount(), equalTo(3));
        assertThat(hostValues.getValuesList(), iterableWithSize(1));
    }

    @Test
    public void labelValuesCrossShardParticularLabelWithTextSearch() {
        createMetric(Labels.of(
            "project", "junk",
            "cluster", "man",
            "service", "test",
            "host", "man-pre-01",
            "sensor", "upTime"));

        createMetric(Labels.of(
            "project", "junk",
            "cluster", "man",
            "service", "test",
            "host", "man-pre-02",
            "sensor", "upTime"));

        createMetric(Labels.of(
            "project", "junk",
            "cluster", "sas",
            "service", "test",
            "host", "sas-pre-01",
            "sensor", "upTime"));

        // noise
        createMetric(Labels.of(
            "project", "junk",
            "cluster", "sas",
            "service", "test",
            "host", "sas-prod-01",
            "sensor", "upTime"));

        TLabelValuesResponse response = syncLabelValues(TLabelValuesRequest.newBuilder()
            .addSelectors(selector("project", "junk"))
            .addSelectors(selector("service", "test"))
            .addSelectors(selector("sensor", "upTime"))
            .addLabels("host")
            .addLabels("cluster")
            .setTextSearch("-pre-")
            .build());

        assertThat(response.getMetricCount(), equalTo(4));

        Map<String, TLabelValues> groups = toLabelValuesMap(response);

        TLabelValues clusterValues = groups.get("cluster");
        assertThat(clusterValues.getValuesList(), emptyIterable());

        TLabelValues hostValues = groups.get("host");
        assertThat(hostValues.getValuesList(),
            allOf(
                iterableWithSize(3),
                everyItem(containsString("-pre-"))
            )
        );
    }

    @Test
    public void labelNames() {
        createMetric(Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "sensor", "upTime",
            "host", "kikimr-003"));

        createMetric(Labels.of(
            "project", "junk",
            "cluster", "sas",
            "service", "test",
            "sensor", "upTime",
            "host", "kikimr-006"));

        createMetric(Labels.of(
            "project", "junk",
            "cluster", "man",
            "service", "test",
            "path", "/myPath"));

        TLabelNamesResponse response = syncLabelNames(TLabelNamesRequest.newBuilder().build());

        assertThat(response.getNamesList(),
            allOf(
                iterableWithSize(6),
                hasItem("path"),
                hasItem("sensor"),
                hasItem("host"),
                hasItem("project"),
                hasItem("cluster"),
                hasItem("service")
            )
        );
    }

    @Test
    public void labelNamesWithFolder() {
        createMetric(Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "service1",
            "sensor", "upTime",
            "host", "kikimr-003"));

        TLabelNamesResponse response = syncLabelNames(TLabelNamesRequest.newBuilder()
                .setFolderId("folder1")
                .build());

        assertThat(response.getNamesList(),
            allOf(
                iterableWithSize(5),
                hasItem("sensor"),
                hasItem("host"),
                hasItem("project"),
                hasItem("cluster"),
                hasItem("service")
            )
        );
    }

    @Test
    public void labelNamesInsideShard() {
        createMetric(Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "kikimr-003",
            "sensor", "upTime"));

        createMetric(Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "kikimr-006",
            "sensor", "upTime"));

        createMetric(Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "path", "/myPath"));

        TLabelNamesResponse response = syncLabelNames(TLabelNamesRequest.newBuilder()
            .addSelectors(selector("project", "junk"))
            .addSelectors(selector("cluster", "foo"))
            .addSelectors(selector("service", "bar"))
            .build());

        assertThat(response.getNamesList(),
            allOf(
                iterableWithSize(3),
                hasItem("path"),
                hasItem("sensor"),
                hasItem("host")
            )
        );
    }

    @Test
    public void labelNamesCrossShard() {
        createMetric(Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "dc", "man",
            "sensor", "idleTime"));

        createMetric(Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "host", "kikimr-006",
            "sensor", "upTime"));

        createMetric(Labels.of(
            "project", "junk",
            "cluster", "sas",
            "service", "test",
            "sensor", "upTime",
            "host", "kikimr-003"));

        TLabelNamesResponse response = syncLabelNames(TLabelNamesRequest.newBuilder()
            .addSelectors(selector("sensor", "upTime"))
            .build());

        assertThat(response.getNamesList(),
            allOf(
                iterableWithSize(4),
                hasItem("host"),
                hasItem("cluster"),
                hasItem("service"),
                hasItem("project")
            )
        );
    }

    @Test
    public void labelNamesInNewFormat() {
        createMetric("upTime", Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "service_with_sensor_name",
            "host", "kikimr-003",
            "param1", "value1"));

        createMetric("upTime", Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "service_with_sensor_name",
            "host", "kikimr-006",
            "param2", "value2"));

        createMetric("upTime", Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "service_with_sensor_name",
            "param3", "value3"));


        Selectors protoSelectors = Selectors.newBuilder()
            .addLabelSelectors(selector("project", "junk"))
            .addLabelSelectors(selector("cluster", "foo"))
            .addLabelSelectors(selector("service", "service_with_sensor_name"))
            .build();

        TLabelNamesResponse response = syncLabelNames(TLabelNamesRequest.newBuilder()
            .setNewSelectors(protoSelectors)
            .build());

        assertThat(response.getNamesList(),
            allOf(
                iterableWithSize(4),
                hasItem("host"),
                hasItem("param1"),
                hasItem("param2"),
                hasItem("param3")
            )
        );
    }

    @Test
    public void uniqueLabelsForGroup() {
        Labels commonLabels = Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar",
            "sensor", "diskFreeSpace");

        createMetric(commonLabels.addAll(Labels.of("host", "test-1", "disk", "/dev/sda1", "type", "ssd")));
        createMetric(commonLabels.addAll(Labels.of("host", "test-1", "disk", "/dev/sda1", "type", "hdd")));
        createMetric(commonLabels.addAll(Labels.of("host", "test-1", "disk", "/dev/sda2", "type", "ssd")));
        createMetric(commonLabels.addAll(Labels.of("host", "test-1", "disk", "/dev/sda3", "type", "hdd")));
        createMetric(commonLabels.addAll(Labels.of("host", "test-2", "disk", "/dev/sda1", "type", "ssd")));
        createMetric(commonLabels.addAll(Labels.of("host", "noise", "disk", "/dev/sda1", "type", "ssd")));

        TUniqueLabelsRequest build = TUniqueLabelsRequest.newBuilder()
            .addAllNames(Arrays.asList("host", "disk"))
            .addSelectors(selector("project", "junk"))
            .addSelectors(selector("cluster", "foo"))
            .addSelectors(selector("service", "bar"))
            .addSelectors(selector(Selector.notGlob("host", "noise")))
            .addSelectors(selector("disk", "*"))
            .build();

        List<Labels> result = syncUniqueLabels(build)
                .getLabelListsList()
                .stream()
                .map(LabelConverter::protoToLabels)
                .collect(Collectors.toList());

        assertThat(result,
                allOf(
                        iterableWithSize(4),
                        hasItem(Labels.of("host", "test-1", "disk", "/dev/sda1")),
                        hasItem(Labels.of("host", "test-1", "disk", "/dev/sda2")),
                        hasItem(Labels.of("host", "test-1", "disk", "/dev/sda3")),
                        hasItem(Labels.of("host", "test-2", "disk", "/dev/sda1"))
                )
        );
    }

    @Test
    public void uniqueLabelsWithFolder() {
        Labels commonLabels = Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "service1",
            "sensor", "diskFreeSpace");

        createMetric(commonLabels.addAll(Labels.of("host", "test-1", "disk", "/dev/sda1", "type", "ssd")));
        createMetric(commonLabels.addAll(Labels.of("host", "test-1", "disk", "/dev/sda1", "type", "hdd")));
        createMetric(commonLabels.addAll(Labels.of("host", "test-1", "disk", "/dev/sda2", "type", "ssd")));
        createMetric(commonLabels.addAll(Labels.of("host", "test-1", "disk", "/dev/sda3", "type", "hdd")));
        createMetric(commonLabels.addAll(Labels.of("host", "test-2", "disk", "/dev/sda1", "type", "ssd")));
        createMetric(commonLabels.addAll(Labels.of("host", "noise", "disk", "/dev/sda1", "type", "ssd")));

        TUniqueLabelsRequest build = TUniqueLabelsRequest.newBuilder()
            .setFolderId("folder1")
            .addAllNames(Arrays.asList("host", "disk"))
            .addSelectors(selector("project", "junk"))
            .addSelectors(selector("cluster", "foo"))
            .addSelectors(selector("service", "service1"))
            .addSelectors(selector(Selector.notGlob("host", "noise")))
            .addSelectors(selector("disk", "*"))
            .build();

        List<Labels> result = syncUniqueLabels(build)
                .getLabelListsList()
                .stream()
                .map(LabelConverter::protoToLabels)
                .collect(Collectors.toList());

        assertThat(result,
                allOf(
                        iterableWithSize(4),
                        hasItem(Labels.of("host", "test-1", "disk", "/dev/sda1")),
                        hasItem(Labels.of("host", "test-1", "disk", "/dev/sda2")),
                        hasItem(Labels.of("host", "test-1", "disk", "/dev/sda3")),
                        hasItem(Labels.of("host", "test-2", "disk", "/dev/sda1"))
                )
        );
    }

    @Test
    public void crossShardUniqueLabels() {
        createMetric(Labels.of("project", "junk", "cluster", "foo", "service", "bar", "sensor", "a"));
        createMetric(Labels.of("project", "junk", "cluster", "foo", "service", "bar", "sensor", "b"));
        createMetric(Labels.of("project", "junk", "cluster", "man", "service", "test", "sensor", "a"));

        TUniqueLabelsRequest build = TUniqueLabelsRequest.newBuilder()
                .addAllNames(Arrays.asList("cluster", "service", "sensor"))
                .addSelectors(selector("project", "junk"))
                .addSelectors(selector("cluster", "*"))
                .addSelectors(selector("service", "*"))
                .build();

        List<Labels> result = syncUniqueLabels(build)
                .getLabelListsList()
                .stream()
                .map(LabelConverter::protoToLabels)
                .collect(Collectors.toList());

        assertThat(result,
                allOf(
                        iterableWithSize(3),
                        hasItem(Labels.of("cluster", "foo", "service", "bar", "sensor", "a")),
                        hasItem(Labels.of("cluster", "foo", "service", "bar", "sensor", "b")),
                        hasItem(Labels.of("cluster", "man", "service", "test", "sensor", "a"))
                )
        );
    }

    @Test
    public void uniqueLabelsInNewFormat() {
        Labels commonLabels = Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "service_with_sensor_name");

        createMetric("diskFreeSpace", commonLabels.addAll(Labels.of("host", "test-1", "disk", "/dev/sda1", "type", "ssd")));
        createMetric("diskFreeSpace", commonLabels.addAll(Labels.of("host", "test-1", "disk", "/dev/sda1", "type", "hdd")));
        createMetric("diskFreeSpace", commonLabels.addAll(Labels.of("host", "test-1", "disk", "/dev/sda2", "type", "ssd")));
        createMetric("diskUsedSpace", commonLabels.addAll(Labels.of("host", "test-1", "disk", "/dev/sda3", "type", "hdd")));
        createMetric("diskUsedSpace", commonLabels.addAll(Labels.of("host", "test-2", "disk", "/dev/sda1", "type", "ssd")));
        createMetric("diskUsedSpace", commonLabels.addAll(Labels.of("host", "noise", "disk", "/dev/sda1", "type", "ssd")));

        Selectors protoSelectors = Selectors.newBuilder()
            .addLabelSelectors(selector("project", "junk"))
            .addLabelSelectors(selector("cluster", "foo"))
            .addLabelSelectors(selector("service", "service_with_sensor_name"))
            .addLabelSelectors(selector(Selector.notGlob("host", "noise")))
            .addLabelSelectors(selector("disk", "*"))
            .build();

        TUniqueLabelsRequest build = TUniqueLabelsRequest.newBuilder()
            .addAllNames(List.of("disk"))
            .setNewSelectors(protoSelectors)
            .build();

        List<Labels> result = syncUniqueLabels(build)
            .getLabelListsList()
            .stream()
            .map(LabelConverter::protoToLabels)
            .collect(Collectors.toList());

        assertThat(result,
            allOf(
                iterableWithSize(3),
                hasItem(Labels.of("disk", "/dev/sda1")),
                hasItem(Labels.of("disk", "/dev/sda2")),
                hasItem(Labels.of("disk", "/dev/sda3"))
            )
        );
    }

    @Test
    public void deleteTooManyMetrics() {
        final int totalMetricsCount = 25000;
        final int batchMetricsCount = 100;

        createTooManyMetrics(totalMetricsCount);

        FindRequest findBatchRequest = FindRequest.newBuilder()
            .setSliceOptions(TSliceOptions.newBuilder().setLimit(batchMetricsCount).build())
            .build();

        int remainedCount = totalMetricsCount;

        while (remainedCount > 0) {
            FindResponse findBatchResponse = syncFind(findBatchRequest);

            assumeThat(findBatchResponse.getStatus(), equalTo(EMetabaseStatusCode.OK));

            if (findBatchResponse.getMetricsCount() == 0) {
                break;
            }

            DeleteManyRequest deleteManyRequest = DeleteManyRequest.newBuilder()
                .addAllMetrics(findBatchResponse.getMetricsList())
                .build();
            DeleteManyResponse deleteManyResponse = syncDeleteMany(deleteManyRequest);
            assumeThat(deleteManyResponse.getStatus(), equalTo(EMetabaseStatusCode.OK));

            int deletedBatchCount = deleteManyResponse.getMetricsCount();
            remainedCount -= deletedBatchCount;
        }

        FindResponse finalFindResponse = syncFind(findBatchRequest);
        assertThat(finalFindResponse.getStatus(), equalTo(EMetabaseStatusCode.OK));
        assertThat(finalFindResponse.getTotalCount(), equalTo(0));
    }

    @Test
    public void resolveLogsOneSameIds() {
        int numId = 1;
        var v1 = List.of(
            new Metric(MetricType.DGAUGE, Labels.of("name", "alice")),
            new Metric(MetricType.DGAUGE, Labels.of("name", "bob")));

        var responseOne = syncResolveLogs(TResolveLogsRequest.newBuilder()
            .setNumId(numId)
            .addUnresolvedLogMeta(toLogMeta(numId, v1))
            .build());

        assertEquals(1, responseOne.getResolvedLogMetricsCount());
        var v2 = fromResolvedMeta(responseOne.getResolvedLogMetrics(0));
        assertEquals(v1, v2);
        assertNotEquals(v1.get(0).shardId, v2.get(0).shardId);
        assertNotEquals(v1.get(0).localId, v2.get(0).localId);

        var responseTwo = syncResolveLogs(TResolveLogsRequest.newBuilder()
            .setNumId(numId)
            .addUnresolvedLogMeta(toLogMeta(numId, v1))
            .build());

        assertEquals(1, responseTwo.getResolvedLogMetricsCount());
        var v3 = fromResolvedMeta(responseTwo.getResolvedLogMetrics(0));
        assertEquals(v2, v3);
        assertEquals(v2.get(0).shardId, v3.get(0).shardId);
        assertEquals(v2.get(0).localId, v3.get(0).localId);
    }

    @Test
    public void resolveLogsChangeKind() {
        int numId = 1;
        var responseOne = syncResolveLogs(TResolveLogsRequest.newBuilder()
            .setNumId(numId)
            .addUnresolvedLogMeta(toLogMeta(numId, List.of(
                new Metric(MetricType.DGAUGE, Labels.of("name", "alice")),
                new Metric(MetricType.DGAUGE, Labels.of("name", "bob")))))
            .build());

        assertEquals(1, responseOne.getResolvedLogMetricsCount());
        var v1 = fromResolvedMeta(responseOne.getResolvedLogMetrics(0));
        var v2 = List.of(v1.get(0), v1.get(1).setType(MetricType.IGAUGE));

        var responseTwo = syncResolveLogs(TResolveLogsRequest.newBuilder()
            .setNumId(numId)
            .addUnresolvedLogMeta(toLogMeta(numId, v2))
            .build());

        assertEquals(1, responseTwo.getResolvedLogMetricsCount());
        var v3 = fromResolvedMeta(responseTwo.getResolvedLogMetrics(0));
        assertNotEquals(v1, v3);
        assertEquals(v2, v3);
    }

    @Test
    public void resolveLogsCreateNotExists() {
        int numId = 1;
        var responseOne = syncResolveLogs(TResolveLogsRequest.newBuilder()
            .setNumId(numId)
            .addUnresolvedLogMeta(toLogMeta(numId, List.of(
                new Metric(MetricType.DGAUGE, Labels.of("name", "alice")))))
            .build());

        assertEquals(1, responseOne.getResolvedLogMetricsCount());
        var v1 = fromResolvedMeta(responseOne.getResolvedLogMetrics(0));
        var v2 = List.of(v1.get(0), new Metric(MetricType.DGAUGE, Labels.of("name", "bob")));

        var responseTwo = syncResolveLogs(TResolveLogsRequest.newBuilder()
            .setNumId(numId)
            .addUnresolvedLogMeta(toLogMeta(numId, v2))
            .build());

        assertEquals(1, responseTwo.getResolvedLogMetricsCount());
        var v3 = fromResolvedMeta(responseTwo.getResolvedLogMetrics(0));
        assertEquals(v2, v3);
        assertNotEquals(0, v3.get(1).shardId);
        assertNotEquals(0, v3.get(1).localId);
        assertNotEquals(v2.get(0).shardId, v3.get(1).shardId);
        assertNotEquals(v2.get(0).localId, v3.get(1).localId);
    }

    @Test
    public void resolveLogsMultipleRequest() {
        int numId = 1;
        var one = List.of(new Metric(MetricType.DGAUGE, Labels.of("name", "alice")));
        var two = List.of(new Metric(MetricType.IGAUGE, Labels.of("name", "alice")));
        var tree = List.of(new Metric(MetricType.DGAUGE, Labels.of("name", "alice")));

        var responseOne = syncResolveLogs(TResolveLogsRequest.newBuilder()
            .setNumId(numId)
            .addUnresolvedLogMeta(toLogMeta(numId, one))
            .addUnresolvedLogMeta(toLogMeta(numId, two))
            .addUnresolvedLogMeta(toLogMeta(numId, tree))
            .build());

        assertEquals(3, responseOne.getResolvedLogMetricsCount());
        var oneV1 = fromResolvedMeta(responseOne.getResolvedLogMetrics(0));
        var twoV1 = fromResolvedMeta(responseOne.getResolvedLogMetrics(1));
        var treeV1 = fromResolvedMeta(responseOne.getResolvedLogMetrics(2));

        assertEquals(one, oneV1);
        assertEquals(two, twoV1);
        assertEquals(tree, treeV1);

        var responseTwo = syncResolveLogs(TResolveLogsRequest.newBuilder()
            .setNumId(numId)
            .addUnresolvedLogMeta(toLogMeta(numId, one))
            .addUnresolvedLogMeta(toLogMeta(numId, two))
            .addUnresolvedLogMeta(toLogMeta(numId, tree))
            .build());

        assertEquals(3, responseTwo.getResolvedLogMetricsCount());
        var oneV2 = fromResolvedMeta(responseTwo.getResolvedLogMetrics(0));
        var twoV2 = fromResolvedMeta(responseTwo.getResolvedLogMetrics(1));
        var treeV2 = fromResolvedMeta(responseTwo.getResolvedLogMetrics(2));

        assertEquals(oneV1, oneV2);
        assertEquals(twoV1, twoV2);
        assertEquals(treeV1, treeV2);
    }

    @Test
    public void resolveLogMultipleSameMetrics() {
        int numId = 1;
        var one = List.of(new Metric(MetricType.DGAUGE, Labels.of("name", "alice")));

        var responseOne = syncResolveLogs(TResolveLogsRequest.newBuilder()
            .setNumId(numId)
            .addUnresolvedLogMeta(toLogMeta(numId, one))
            .addUnresolvedLogMeta(toLogMeta(numId, one))
            .addUnresolvedLogMeta(toLogMeta(numId, one))
            .addUnresolvedLogMeta(toLogMeta(numId, one))
            .build());

        assertEquals(4, responseOne.getResolvedLogMetricsCount());
        var oneV1 = fromResolvedMeta(responseOne.getResolvedLogMetrics(0));
        var twoV1 = fromResolvedMeta(responseOne.getResolvedLogMetrics(1));

        assertEquals(one, oneV1);
        assertEquals(one, twoV1);
        assertEquals(oneV1.get(0).shardId, twoV1.get(0).shardId);
        assertEquals(oneV1.get(0).localId, twoV1.get(0).localId);

        var responseTwo = syncResolveLogs(TResolveLogsRequest.newBuilder()
            .setNumId(numId)
            .addUnresolvedLogMeta(toLogMeta(numId, one))
            .addUnresolvedLogMeta(toLogMeta(numId, one))
            .build());

        assertEquals(2, responseTwo.getResolvedLogMetricsCount());
        var oneV2 = fromResolvedMeta(responseTwo.getResolvedLogMetrics(0));
        var twoV2 = fromResolvedMeta(responseTwo.getResolvedLogMetrics(1));

        assertEquals(oneV1, oneV2);
        assertEquals(twoV1, twoV2);
        assertEquals(twoV1.get(0).shardId, twoV2.get(0).shardId);
        assertEquals(twoV1.get(0).localId, twoV2.get(0).localId);
    }

    @Test
    public void resolveLogsConcurrent() {
        int numId = 1;
        var source = List.of(
            List.of(new Metric(MetricType.DGAUGE, Labels.of("name", "alice"))),
            List.of(new Metric(MetricType.IGAUGE, Labels.of("name", "alice"))),
            List.of(new Metric(MetricType.DGAUGE, Labels.of("name", "alice"))));

        CyclicBarrier barrier = new CyclicBarrier(3);
        var results = source.stream()
            .map(metrics -> {
                return CompletableFuture.supplyAsync(() -> {
                    var req = TResolveLogsRequest.newBuilder()
                        .setNumId(numId)
                        .addUnresolvedLogMeta(toLogMeta(numId, metrics))
                        .build();

                    try {
                        barrier.await();
                        return syncResolveLogs(req);
                    } catch (Throwable e) {
                        throw new RuntimeException(e);
                    }
                });
            })
            .collect(collectingAndThen(Collectors.toList(), CompletableFutures::allOf))
            .join();

        int shardId = 0;
        long localId = 0;
        for (int index = 0; index < source.size(); index++) {
            var result = results.get(index);
            assertEquals(1, result.getResolvedLogMetricsCount());
            var metrics = fromResolvedMeta(result.getResolvedLogMetrics(0));
            assertEquals(source.get(index), metrics);
            assertNotEquals(0, metrics.get(0).shardId);
            if (shardId != 0) {
                assertEquals(shardId, metrics.get(0).shardId);
                assertEquals(localId, metrics.get(0).localId);
            } else {
                shardId = metrics.get(0).shardId;
                localId = metrics.get(0).localId;
            }
        }
    }

    @Test
    public void changeKindByCreateMany() {
        Labels commonLabels = Labels.of(
            "project", "junk",
            "cluster", "foo",
            "service", "bar");

        Labels idleTime = Labels.of("sensor", "idleTime");

        CreateManyResponse responseOne = syncCreateMany(CreateManyRequest.newBuilder()
            .addAllCommonLabels(labelsToProtoList(commonLabels))
            .addMetrics(ru.yandex.solomon.metabase.api.protobuf.Metric.newBuilder()
                .addAllLabels(labelsToProtoList(idleTime))
                .setType(ru.yandex.solomon.model.protobuf.MetricType.LOG_HISTOGRAM)
                .build())
            .build());

        assertEquals(1, responseOne.getMetricsCount());
        var v1 = responseOne.getMetrics(0);

        CreateManyResponse responseTwo = syncCreateMany(CreateManyRequest.newBuilder()
            .addAllCommonLabels(labelsToProtoList(commonLabels))
            .addMetrics(ru.yandex.solomon.metabase.api.protobuf.Metric.newBuilder()
                .addAllLabels(labelsToProtoList(idleTime))
                .setType(ru.yandex.solomon.model.protobuf.MetricType.HIST)
                .build())
            .build());

        var expected = v1.toBuilder()
            .setType(ru.yandex.solomon.model.protobuf.MetricType.HIST)
            .build();

        assertEquals(1, responseTwo.getMetricsCount());
        var v2 = responseTwo.getMetrics(0);
        assertNotEquals(v1, v2);
        assertEquals(expected, v2);
    }

    @Test
    public void findWithShardId_Success() {
        var shardKey = new ShardKey("junk", "foo", "bar");
        Labels shardKeyLabels = shardKey.toLabels();
        int shardId = resolverStub.resolveShard(shardKey).getNumId();

        createMetric(1, 1, shardKeyLabels.add("host", "kikimr-001").add("sensor", "upTime"));
        createMetric(1, 2, shardKeyLabels.add("host", "kikimr-002").add("sensor", "upTime"));
        Labels hostCluster = shardKeyLabels.add("host", "cluster").add("sensor", "upTime");
        createMetric(1, 3, hostCluster);

        FindResponse response = syncFind(FindRequest.newBuilder()
                .setShardId(shardId)
                .addSelectors(selector("host", "cluster"))
                .build());

        var metrics = response.getMetricsList();
        assertEquals(1, metrics.size());
        assertEquals(metricId(1, 3), metrics.get(0).getMetricId());
        assertEquals(hostCluster, LabelConverter.protoToLabels(metrics.get(0).getLabelsList()));
    }

    @Test
    public void findWithShardId_NotFound() {
        Labels shardKeyLabels = new ShardKey("junk", "foo", "bar").toLabels();
        createMetric(1, 1, shardKeyLabels.add("host", "kikimr-001").add("sensor", "upTime"));
        createMetric(1, 2, shardKeyLabels.add("host", "kikimr-002").add("sensor", "upTime"));

        try {
            metabaseService.find(FindRequest.newBuilder()
                    .setShardId(100_500)
                    .addSelectors(selector("host", "cluster"))
                    .build()).join();
            fail("expected ShardIsNotLocalException is not thrown");
        } catch (ShardIsNotLocalException e) {
            var status = GrpcMetabaseService.classifyError(e);
            assertEquals(EMetabaseStatusCode.SHARD_NOT_FOUND, status.code);
            assertEquals("shard 100500 is not local", status.message);
        }
    }

    @Test
    public void metricNamesWithShardId_Success() {
        var shardKey = new ShardKey("junk", "foo", "service_with_sensor_name");
        int shardId = resolverStub.resolveShard(shardKey).getNumId();

        createMetric("upTime", shardKey.toLabels().add("host", "kikimr-003"));

        MetricNamesResponse response = syncMetricNames(MetricNamesRequest.newBuilder()
                .setShardId(shardId)
                .build());

        assertThat(response.getNamesList(), allOf(iterableWithSize(1), hasItem("upTime")));
    }

    @Test
    public void metricNamesWithShardId_NotFound() {
        var shardKey = new ShardKey("junk", "foo", "service_with_sensor_name");
        createMetric("upTime", shardKey.toLabels().add("host", "kikimr-003"));

        try {
            var request = MetricNamesRequest.newBuilder()
                    .setShardId(100_500)
                    .build();
            metabaseService.metricNames(request).join();
            fail("expected ShardIsNotLocalException is not thrown");
        } catch (ShardIsNotLocalException e) {
            var status = GrpcMetabaseService.classifyError(e);
            assertEquals(EMetabaseStatusCode.SHARD_NOT_FOUND, status.code);
            assertEquals("shard 100500 is not local", status.message);
        }
    }

    @Test
    public void labelValuesWithShardId_Success() {
        var shardKey = new ShardKey("junk", "foo", "bar");
        int shardId = resolverStub.resolveShard(shardKey).getNumId();
        createMetric(shardKey.toLabels().add("host", "kikimr-003").add("sensor", "upTime"));

        TLabelValuesResponse response = syncLabelValues(TLabelValuesRequest.newBuilder()
                .setShardId(shardId)
                .addSelectors(selector("sensor", "upTime"))
                .build());

        assertThat(response.getMetricCount(), equalTo(1));
        assertThat(response.getValuesList(), not(empty()));
    }

    @Test
    public void labelValuesWithShardId_NotFound() {
        var shardKey = new ShardKey("junk", "foo", "bar");
        createMetric(shardKey.toLabels().add("host", "kikimr-003").add("sensor", "upTime"));

        try {
            var request = TLabelValuesRequest.newBuilder()
                    .setShardId(100_500)
                    .addSelectors(selector("sensor", "upTime"))
                    .build();
            metabaseService.labelValues(request).join();
            fail("expected ShardIsNotLocalException is not thrown");
        } catch (ShardIsNotLocalException e) {
            var status = GrpcMetabaseService.classifyError(e);
            assertEquals(EMetabaseStatusCode.SHARD_NOT_FOUND, status.code);
            assertEquals("shard 100500 is not local", status.message);
        }
    }

    @Test
    public void labelNamesWithShardId_Success() {
        var shardKey = new ShardKey("junk", "foo", "bar");
        int shardId = resolverStub.resolveShard(shardKey).getNumId();
        Labels shardKeyLabels = shardKey.toLabels();

        createMetric(shardKeyLabels.add("host", "kikimr-003").add("sensor", "upTime"));
        createMetric(shardKeyLabels.add("host", "kikimr-006").add("sensor", "upTime"));
        createMetric(shardKeyLabels.add("path", "/myPath"));

        TLabelNamesResponse response = syncLabelNames(TLabelNamesRequest.newBuilder()
                .setShardId(shardId)
                .build());

        var names = response.getNamesList();
        assertEquals(3, names.size());
        assertTrue(names.containsAll(List.of("path", "sensor", "host")));
    }

    @Test
    public void labelNamesWithShardId_NotFound() {
        Labels shardKeyLabels = new ShardKey("junk", "foo", "bar").toLabels();

        createMetric(shardKeyLabels.add("host", "kikimr-003").add("sensor", "upTime"));
        createMetric(shardKeyLabels.add("host", "kikimr-006").add("sensor", "upTime"));
        createMetric(shardKeyLabels.add("path", "/myPath"));

        try {
            var request = TLabelNamesRequest.newBuilder()
                    .setShardId(100_500)
                    .build();
            metabaseService.labelNames(request).join();
            fail("expected ShardIsNotLocalException is not thrown");
        } catch (ShardIsNotLocalException e) {
            var status = GrpcMetabaseService.classifyError(e);
            assertEquals(EMetabaseStatusCode.SHARD_NOT_FOUND, status.code);
            assertEquals("shard 100500 is not local", status.message);
        }
    }

    @Test
    public void uniqueLabelsWithShardId_Success() {
        var shardKey = new ShardKey("junk", "foo", "bar");
        int shardId = resolverStub.resolveShard(shardKey).getNumId();
        Labels shardKeyLabels = shardKey.toLabels();

        createMetric(shardKeyLabels.add("sensor", "a").add("host", "solomon-001"));
        createMetric(shardKeyLabels.add("sensor", "a").add("host", "solomon-002"));
        createMetric(shardKeyLabels.add("sensor", "b").add("host", "solomon-001"));
        createMetric(shardKeyLabels.add("sensor", "c").add("host", "solomon-002"));

        var request = TUniqueLabelsRequest.newBuilder()
                .setShardId(shardId)
                .addAllNames(List.of("sensor"))
                .build();

        var response = syncUniqueLabels(request);

        List<Labels> labelsList = response.getLabelListsList().stream()
                .map(LabelConverter::protoToLabels)
                .collect(Collectors.toList());

        assertEquals(3, labelsList.size());
        assertTrue(labelsList.contains(Labels.of("sensor", "a")));
        assertTrue(labelsList.contains(Labels.of("sensor", "b")));
        assertTrue(labelsList.contains(Labels.of("sensor", "c")));
    }

    @Test
    public void uniqueLabelsWithShardId_NotFound() {
        Labels shardKeyLabels = new ShardKey("junk", "foo", "bar").toLabels();

        createMetric(shardKeyLabels.add("sensor", "a").add("host", "solomon-001"));
        createMetric(shardKeyLabels.add("sensor", "a").add("host", "solomon-002"));
        createMetric(shardKeyLabels.add("sensor", "b").add("host", "solomon-001"));
        createMetric(shardKeyLabels.add("sensor", "c").add("host", "solomon-002"));

        try {
            var request = TUniqueLabelsRequest.newBuilder()
                    .setShardId(100_500)
                    .addAllNames(List.of("sensor"))
                    .build();

            metabaseService.uniqueLabels(request).join();

            fail("expected ShardIsNotLocalException is not thrown");
        } catch (ShardIsNotLocalException e) {
            var status = GrpcMetabaseService.classifyError(e);
            assertEquals(EMetabaseStatusCode.SHARD_NOT_FOUND, status.code);
            assertEquals("shard 100500 is not local", status.message);
        }
    }

    private void createTooManyMetrics(int metricsCount) {
        List<Label> commonLabels = Arrays.asList(
            Label.newBuilder().setKey("project").setValue("junk").build(),
            Label.newBuilder().setKey("cluster").setValue("foo").build(),
            Label.newBuilder().setKey("service").setValue("bar").build()
        );

        CreateManyRequest.Builder createManyRequestBuilder =
            CreateManyRequest.newBuilder()
                .addAllCommonLabels(commonLabels);
        for (int i = 0; i < metricsCount; ++i) {

            Label metricLabel =
                Label.newBuilder().setKey("sensor").setValue("sensor-" + i).build();

            MetricId metricId = MetricId.newBuilder().setShardId(StockpileShardId.random())
                .setLocalId(StockpileLocalId.random()).build();

            var metric = ru.yandex.solomon.metabase.api.protobuf.Metric.newBuilder()
                .addLabels(metricLabel)
                .setMetricId(metricId)
                .build();

            createManyRequestBuilder.addMetrics(metric);
        }

        CreateManyRequest createManyRequest = createManyRequestBuilder.build();
        CreateManyResponse createManyResponse = syncCreateMany(createManyRequest);
        assumeThat(createManyResponse.getStatus(), equalTo(EMetabaseStatusCode.OK));
    }

    private void createMetric(Labels labels) {
        createMetric(StockpileShardId.random(), StockpileLocalId.random(), labels);
    }

    private void createMetric(String name, Labels labels) {
        createMetric(StockpileShardId.random(), StockpileLocalId.random(), name, labels);
    }

    private void createMetric(int shardId, long localId, Labels labels) {
        syncCreateOne(CreateOneRequest.newBuilder()
                .setMetric(ru.yandex.solomon.metabase.api.protobuf.Metric.newBuilder()
                        .setMetricId(MetricId.newBuilder()
                                .setShardId(shardId)
                                .setLocalId(localId)
                                .build())
                        .addAllLabels(labelsToProtoList(labels))
                        .setType(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE)
                        .build())
                .build());
    }

    private void createMetric(int shardId, long localId, String name, Labels labels) {
        syncCreateOne(CreateOneRequest.newBuilder()
                .setMetric(ru.yandex.solomon.metabase.api.protobuf.Metric.newBuilder()
                        .setMetricId(MetricId.newBuilder()
                                .setShardId(shardId)
                                .setLocalId(localId)
                                .build())
                        .setName(name)
                        .addAllLabels(labelsToProtoList(labels))
                        .setType(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE)
                        .build())
                .build());
    }

    private CreateOneResponse syncCreateOne(CreateOneRequest request) {
        return syncSuccessRun(request,
                metabaseService::createOne,
                CreateOneResponse::getStatus,
                CreateOneResponse::getStatusMessage);
    }

    private CreateManyResponse syncCreateMany(CreateManyRequest request) {
        return syncSuccessRun(request,
                metabaseService::createMany,
                CreateManyResponse::getStatus,
                CreateManyResponse::getStatusMessage);
    }

    private ResolveOneResponse syncResolveOne(ResolveOneRequest request) {
        return syncSuccessRun(request,
                metabaseService::resolveOne,
                ResolveOneResponse::getStatus,
                ResolveOneResponse::getStatusMessage);
    }

    private ResolveManyResponse syncResolveMany(ResolveManyRequest request) {
        return syncSuccessRun(request,
                metabaseService::resolveMany,
                ResolveManyResponse::getStatus,
                ResolveManyResponse::getStatusMessage);
    }

    private FindResponse syncFind(FindRequest request) {
        return syncSuccessRun(request,
                metabaseService::find,
                FindResponse::getStatus,
                FindResponse::getStatusMessage);
    }

    private MetricNamesResponse syncMetricNames(MetricNamesRequest request) {
        return syncSuccessRun(request,
            metabaseService::metricNames,
            MetricNamesResponse::getStatus,
            MetricNamesResponse::getStatusMessage);
    }

    private TLabelValuesResponse syncLabelValues(TLabelValuesRequest request) {
        return syncSuccessRun(request,
                metabaseService::labelValues,
                TLabelValuesResponse::getStatus,
                TLabelValuesResponse::getStatusMessage);
    }

    private TLabelNamesResponse syncLabelNames(TLabelNamesRequest request) {
        return syncSuccessRun(request,
            metabaseService::labelNames,
            TLabelNamesResponse::getStatus,
            TLabelNamesResponse::getStatusMessage);
    }

    private TUniqueLabelsResponse syncUniqueLabels(TUniqueLabelsRequest request) {
        return syncSuccessRun(request,
                metabaseService::uniqueLabels,
                TUniqueLabelsResponse::getStatus,
                TUniqueLabelsResponse::getStatusMessage);
    }

    private DeleteManyResponse syncDeleteMany(DeleteManyRequest request) {
        return syncSuccessRun(request,
                metabaseService::deleteMany,
                DeleteManyResponse::getStatus,
                DeleteManyResponse::getStatusMessage);
    }

    private TResolveLogsResponse syncResolveLogs(TResolveLogsRequest request) {
        return syncSuccessRun(request,
            metabaseService::resolveLogs,
            TResolveLogsResponse::getStatus,
            TResolveLogsResponse::getStatusMessage);
    }

    private <Request, Response> Response syncSuccessRun(
        Request request,
        Function<Request, CompletableFuture<Response>> fn,
        Function<Response, EMetabaseStatusCode> status,
        Function<Response, String> message)
    {
        Response response = fn.apply(request).join();

        if (status.apply(response) != EMetabaseStatusCode.OK) {
            throw new IllegalStateException(status.apply(response) + ": " + message.apply(response));
        }

        return response;
    }

    private Map<String, TLabelValues> toLabelValuesMap(TLabelValuesResponse response) {
        return response
            .getValuesList()
            .stream()
            .collect(Collectors.toMap(TLabelValues::getName, Function.identity()));
    }

    private static ByteString toLogMeta(int numId, List<Metric> metrics) {
        try (UnresolvedLogMetaBuilderImpl builder = new UnresolvedLogMetaBuilderImpl(numId, CompressionAlg.LZ4, UnpooledByteBufAllocator.DEFAULT)) {
            builder.onCommonLabels(Labels.of());
            for (var metric : metrics) {
                builder.onMetric(metric.type, metric.labels, 1, 14);
            }
            var buffer = builder.build();
            var result = ByteString.copyFrom(ByteBufUtil.getBytes(buffer));
            buffer.release();
            return result;
        }
    }

    private static List<Metric> fromResolvedMeta(ByteString meta) {
        var result = new ArrayList<Metric>();
        var record = new ResolvedLogMetricsRecord();
        try (var it = new ResolvedLogMetricsIteratorImpl(ByteStrings.toByteBuf(meta))) {
            while (it.next(record)) {
                result.add(new Metric(record.type, record.labels, record.shardId, record.localId));
            }
        }
        return result;
    }

    private static class Metric extends DefaultToString {
        final MetricType type;
        final Labels labels;
        final int shardId;
        final long localId;
        final boolean full;

        public Metric(MetricType type, Labels labels) {
            this.type = type;
            this.labels = labels;
            this.shardId = 0;
            this.localId = 0;
            this.full = false;
        }

        public Metric(MetricType type, Labels labels, int shardId, long localId) {
            this.type = type;
            this.labels = labels;
            this.shardId = shardId;
            this.localId = localId;
            this.full = shardId != 0 && localId != 0;
        }

        public Metric setType(MetricType type) {
            return new Metric(type, labels, shardId, localId);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (!(o instanceof Metric)) return false;

            Metric metric = (Metric) o;
            if (full && metric.full) {
                if (shardId != metric.shardId) return false;
                if (localId != metric.localId) return false;
            }

            if (type != metric.type) return false;
            return labels.equals(metric.labels);
        }

        @Override
        public int hashCode() {
            int result = type.hashCode();
            result = 31 * result + labels.hashCode();
            result = 31 * result + shardId;
            result = 31 * result + (int) (localId ^ (localId >>> 32));
            result = 31 * result + (full ? 1 : 0);
            return result;
        }
    }
}
