package ru.yandex.solomon.coremon.meta.service;

import java.util.Collections;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.net.HostAndPort;
import io.grpc.Server;
import io.grpc.inprocess.InProcessServerBuilder;
import io.netty.util.concurrent.DefaultThreadFactory;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.grpc.utils.DefaultClientOptions;
import ru.yandex.grpc.utils.InProcessChannelFactory;
import ru.yandex.metabase.client.MetabaseClient;
import ru.yandex.metabase.client.MetabaseClientOptions;
import ru.yandex.metabase.client.MetabaseClients;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.FileCoremonMetric;
import ru.yandex.solomon.labels.protobuf.LabelConverter;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.metabase.api.protobuf.CreateOneRequest;
import ru.yandex.solomon.metabase.api.protobuf.CreateOneResponse;
import ru.yandex.solomon.metabase.api.protobuf.DeleteManyRequest;
import ru.yandex.solomon.metabase.api.protobuf.DeleteManyResponse;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metabase.api.protobuf.FindRequest;
import ru.yandex.solomon.metabase.api.protobuf.FindResponse;
import ru.yandex.solomon.metabase.api.protobuf.Metric;
import ru.yandex.solomon.metabase.api.protobuf.ResolveManyRequest;
import ru.yandex.solomon.metabase.api.protobuf.ResolveManyResponse;
import ru.yandex.solomon.metabase.api.protobuf.ResolveOneRequest;
import ru.yandex.solomon.metabase.api.protobuf.ResolveOneResponse;
import ru.yandex.solomon.metabase.api.protobuf.TLabelNamesRequest;
import ru.yandex.solomon.metabase.api.protobuf.TLabelNamesResponse;
import ru.yandex.solomon.metabase.api.protobuf.TLabelValues;
import ru.yandex.solomon.metabase.api.protobuf.TLabelValuesRequest;
import ru.yandex.solomon.metabase.api.protobuf.TLabelValuesResponse;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.model.protobuf.MetricTypeConverter;
import ru.yandex.solomon.model.protobuf.Selector;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.shard.StockpileShardId;

import static org.hamcrest.CoreMatchers.allOf;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.hasItem;
import static org.hamcrest.Matchers.iterableWithSize;
import static org.junit.Assert.assertThat;
import static ru.yandex.solomon.coremon.meta.service.MetricMatchers.hasLabels;
import static ru.yandex.solomon.coremon.meta.service.MetricMatchers.hasMetricId;

/**
 * @author Vladimir Gordiychuk
 */
public class GrpcMetabaseServerTest {

    private ExecutorService grpcServerExecutorService;
    private ExecutorService grpcClientExecutorService;
    private Server server;
    private MetabaseServiceTestStub stub;
    private MetabaseClient client;

    private static Selector selector(String key, String pattern) {
        return Selector.newBuilder().setKey(key).setPattern(pattern).build();
    }

    @Before
    public void setUp() throws Exception {
        grpcServerExecutorService = Executors.newFixedThreadPool(Runtime.getRuntime().availableProcessors(),
                new DefaultThreadFactory("metabase-server-grpc"));
        grpcClientExecutorService = Executors.newFixedThreadPool(Runtime.getRuntime().availableProcessors(),
                new DefaultThreadFactory("metabase-client-grpc"));

        stub = new MetabaseServiceTestStub(
                new ShardKey("junk", "foo", "bar")
        );
        String serverName = UUID.randomUUID().toString();
        var service = new GrpcMetabaseService(stub, new MetricRegistry());
        server = InProcessServerBuilder.forName(serverName)
                .addService(service)
                .addService(service)
                .executor(grpcServerExecutorService)
                .build()
                .start();

        var clientOptions = MetabaseClientOptions.newBuilder(
                DefaultClientOptions.newBuilder()
                        .setRequestTimeOut(10, TimeUnit.SECONDS)
                        .setChannelFactory(new InProcessChannelFactory())
                        .setRpcExecutor(grpcClientExecutorService))
                .build();

        client = MetabaseClients.create(
                Collections.singletonList(HostAndPort.fromHost(serverName)),
                clientOptions
        );
        forceUpdateClusterMetadata();
    }

    @After
    public void tearDown() throws Exception {
        stub.close();

        if (client != null) {
            client.close();
        }

        if (server != null) {
            server.shutdownNow();
        }

        if (grpcClientExecutorService != null) {
            grpcClientExecutorService.shutdown();
        }

        if (grpcServerExecutorService != null) {
            grpcServerExecutorService.shutdown();
        }
    }

    @Test
    public void createOne() throws Exception {
        CoremonMetric metric = metric(
                Labels.of(
                        "project", "junk",
                        "cluster", "foo",
                        "service", "bar",
                        "host", "kikimr-003",
                        "sensor", "upTime"));

        CreateOneResponse response = syncCreateOne(CreateOneRequest.newBuilder()
                .setMetric(metricToProto(metric))
                .build());

        assertThat(response.getMetric(), allOf(
                hasMetricId(metricId(metric.getShardId(), metric.getLocalId())),
                hasLabels(metric.getLabels())
        ));
    }

    @Test
    public void createOneNotValidLabelName() throws Exception {
        CoremonMetric metric = metric(
                Labels.of(
                        "project", "junk",
                        "cluster", "foo",
                        "service", "bar",
                        "!@#$%^&*()_-", "kikimr-003",
                        "sensor", "upTime"));

        CreateOneResponse response = client.createOne(CreateOneRequest.newBuilder()
                .setMetric(metricToProto(metric))
                .build())
                .join();

        assertThat(response.getStatus(), equalTo(EMetabaseStatusCode.INVALID_REQUEST));
    }

    @Test
    public void createOneNotValidLabelValue() throws Exception {
        CoremonMetric metric = metric(
                Labels.of(
                        "project", "junk",
                        "cluster", "foo",
                        "service", "bar",
                        "host", ThreadLocalRandom.current()
                                .longs(100)
                                .mapToObj(String::valueOf)
                                .collect(Collectors.joining()),
                        "sensor", "upTime"));

        CreateOneResponse response = client.createOne(CreateOneRequest.newBuilder()
                .setMetric(metricToProto(metric))
                .build())
                .join();

        assertThat(response.getStatus(), equalTo(EMetabaseStatusCode.INVALID_REQUEST));
    }

    @Test
    public void exactlySearch() throws Exception {
        CoremonMetric first = metric(
                1, 1,
                Labels.of(
                        "project", "junk",
                        "cluster", "foo",
                        "service", "bar",
                        "host", "kikimr-003",
                        "sensor", "upTime"));

        CoremonMetric noise = metric(
                1, 2,
                Labels.of(
                        "project", "junk",
                        "cluster", "foo",
                        "service", "bar",
                        "host", "kikimr-006",
                        "sensor", "upTime"));

        addMetrics(noise, first);

        FindResponse response = syncFind(FindRequest.newBuilder()
                .addSelectors(selector("project", "junk"))
                .addSelectors(selector("cluster", "foo"))
                .addSelectors(selector("service", "bar"))
                .addSelectors(selector("host", "kikimr-003"))
                .addSelectors(selector("sensor", "upTime"))
                .build());

        assertThat(response.getMetricsCount(), equalTo(1));
        assertThat(response.getTotalCount(), equalTo(1));
        assertThat(response.getMetrics(0),
                allOf(
                        hasMetricId(metricId(1, 1)),
                        hasLabels(first.getLabels())
                )
        );
    }

    @Test
    public void deleteMetric() throws Exception {
        CoremonMetric first = metric(
                1, 1,
                Labels.of(
                        "project", "junk",
                        "cluster", "foo",
                        "service", "bar",
                        "host", "kikimr-003",
                        "sensor", "upTime"));

        CoremonMetric second = metric(
                1, 2,
                Labels.of(
                        "project", "junk",
                        "cluster", "foo",
                        "service", "bar",
                        "host", "kikimr-006",
                        "sensor", "upTime"));

        CoremonMetric noise = metric(
                1, 3,
                Labels.of(
                        "project", "junk",
                        "cluster", "foo",
                        "service", "bar",
                        "host", "cluster",
                        "sensor", "upTime"));

        addMetrics(first, second, noise);

        {
            FindResponse response = syncFind(FindRequest.newBuilder()
                .addSelectors(selector("project", "junk"))
                .addSelectors(selector("host", "kikimr-006"))
                .build());

            assertThat(response.getMetricsCount(), equalTo(1));
            assertThat(response.getMetricsList(),
                hasItem(allOf(
                    hasMetricId(metricId(1, 2)),
                    hasLabels(second.getLabels()))));
        }

        syncFindAndDelete(FindRequest.newBuilder()
                .addSelectors(selector("project", "junk"))
                .addSelectors(selector("host", "kikimr-006"))
                .build());

        {
            FindResponse response = syncFind(FindRequest.newBuilder()
                .addSelectors(selector("project", "junk"))
                .addSelectors(selector("host", "kikimr-006"))
                .build());

            assertThat(response.getMetricsCount(), equalTo(0));
        }
    }

    @Test
    public void resolveOne() throws Exception {
        CoremonMetric metric = metric(
                Labels.of(
                        "project", "junk",
                        "cluster", "foo",
                        "service", "bar",
                        "host", "kikimr-003",
                        "sensor", "upTime"));

        CoremonMetric noise = metric(
                Labels.of(
                        "project", "junk",
                        "cluster", "foo",
                        "service", "bar",
                        "host", "cluster",
                        "sensor", "upTime"));

        addMetrics(metric, noise);

        ResolveOneResponse response = syncResolveOne(ResolveOneRequest.newBuilder()
                .addAllLabels(LabelConverter.labelsToProtoList(metric.getLabels()))
                .build());

        assertThat(response.getMetric(), allOf(
                hasMetricId(metricId(metric.getShardId(), metric.getLocalId())),
                hasLabels(metric.getLabels())
        ));
    }

    private void addMetrics(CoremonMetric... metrics) {
        for (CoremonMetric metric : metrics) {
            syncCreateOne(CreateOneRequest.newBuilder()
                    .setMetric(metricToProto(metric))
                    .build());
        }
    }

    @Test
    public void resolveOneNotFound() throws Exception {
        CoremonMetric metric = metric(
                Labels.of(
                        "project", "junk",
                        "cluster", "foo",
                        "service", "bar",
                        "host", "kikimr-003",
                        "sensor", "upTime"));

        addMetrics(metric);

        ResolveOneResponse response = client.resolveOne(ResolveOneRequest.newBuilder()
                .addAllLabels(LabelConverter.labelsToProtoList(Labels.of(
                        "project", "junk",
                        "cluster", "foo",
                        "service", "bar",
                        "sensor", "upTime")
                ))
                .build())
                .join();

        assertThat(response.getStatus(), equalTo(EMetabaseStatusCode.NOT_FOUND));
    }

    @Test
    public void resolveMany() throws Exception {
        CoremonMetric first = metric(
                Labels.of(
                        "project", "junk",
                        "cluster", "foo",
                        "service", "bar",
                        "host", "kikimr-003",
                        "sensor", "upTime"));

        CoremonMetric second = metric(
                Labels.of(
                        "project", "junk",
                        "cluster", "foo",
                        "service", "bar",
                        "host", "kikimr-006",
                        "sensor", "upTime"));

        CoremonMetric noise = metric(
                Labels.of(
                        "project", "junk",
                        "cluster", "foo",
                        "service", "bar",
                        "host", "cluster",
                        "sensor", "upTime"));

        addMetrics(first, second, noise);

        ResolveManyRequest.Builder requestBuilder = ResolveManyRequest.newBuilder();
        LabelConverter.addLabels(requestBuilder.addListLabelsBuilder(), first.getLabels());
        LabelConverter.addLabels(requestBuilder.addListLabelsBuilder(), second.getLabels());
        ResolveManyResponse response = syncResolveMany(requestBuilder.build());

        assertThat(response.getMetricsCount(), equalTo(2));
        assertThat(response.getMetricsList(), allOf(
                hasItem(
                        allOf(
                                hasMetricId(metricId(first.getShardId(), first.getLocalId())),
                                hasLabels(first.getLabels())
                        )
                ),
                hasItem(
                        allOf(
                                hasMetricId(metricId(second.getShardId(), second.getLocalId())),
                                hasLabels(second.getLabels())
                        )
                )
        ));
    }

    @Test
    public void labelValues() throws Exception {
        CoremonMetric first = metric(
                Labels.of(
                        "project", "junk",
                        "cluster", "foo",
                        "service", "bar",
                        "host", "kikimr-003",
                        "sensor", "upTime")
        );

        CoremonMetric second = metric(
                Labels.of(
                        "project", "junk",
                        "cluster", "foo",
                        "service", "bar",
                        "host", "kikimr-006",
                        "sensor", "upTime")
        );

        CoremonMetric noise = metric(
                Labels.of(
                        "project", "junk",
                        "cluster", "foo",
                        "service", "bar",
                        "dc", "MAN",
                        "sensor", "upTime")
        );

        addMetrics(first, second, noise);

        TLabelValuesResponse response = syncLabelValues(TLabelValuesRequest.newBuilder()
                .addSelectors(selector("project", "junk"))
                .addSelectors(selector("cluster", "foo"))
                .addSelectors(selector("service", "bar"))
                .addSelectors(selector("sensor", "upTime"))
                .addLabels("host")
                .build());

        assertThat(response.getValuesCount(), equalTo(1));
        TLabelValues values = response.getValues(0);
        assertThat(values.getName(), equalTo("host"));
        assertThat(values.getAbsent(), equalTo(true));
        assertThat(values.getMetricCount(), equalTo(2));
        assertThat(values.getValuesList(),
                allOf(
                        iterableWithSize(2),
                        hasItem("kikimr-006"),
                        hasItem("kikimr-003")
                )
        );
    }

    @Test
    public void labelNames() throws Exception {
        CoremonMetric first = metric(
            Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "bar",
                "host", "kikimr-003",
                "sensor", "upTime"));

        CoremonMetric second = metric(
            Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "bar",
                "host", "kikimr-006",
                "sensor", "upTime"));

        CoremonMetric third = metric(
            Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "bar",
                "host", "kikimr-008",
                "sensor", "upTime"));

        CoremonMetric noise1 = metric(
            Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "bar",
                "dc", "MAN",
                "sensor", "upTime"));

        CoremonMetric noise2 = metric(
            Labels.of(
                "project", "junk",
                "cluster", "foo",
                "service", "bar",
                "type", "threshold",
                "sensor", "upTime"));

        addMetrics(first, second, third, noise1, noise2);

        TLabelNamesResponse response = syncLabelNames(TLabelNamesRequest.newBuilder()
            .addSelectors(selector("sensor", "upTime"))
            .build());

        assertThat(response.getNamesList(),
            allOf(
                iterableWithSize(6),
                hasItem("project"),
                hasItem("cluster"),
                hasItem("service"),
                hasItem("type"),
                hasItem("host"),
                hasItem("dc")
            )
        );
    }

    private CreateOneResponse syncCreateOne(CreateOneRequest request) {
        return syncSuccessRun(request,
                client::createOne,
                CreateOneResponse::getStatus,
                CreateOneResponse::getStatusMessage);
    }

    private ResolveOneResponse syncResolveOne(ResolveOneRequest request) {
        return syncSuccessRun(request,
                client::resolveOne,
                ResolveOneResponse::getStatus,
                ResolveOneResponse::getStatusMessage);
    }

    private ResolveManyResponse syncResolveMany(ResolveManyRequest request) {
        return syncSuccessRun(request,
                client::resolveMany,
                ResolveManyResponse::getStatus,
                ResolveManyResponse::getStatusMessage);
    }

    private FindResponse syncFind(FindRequest request) {
        return syncSuccessRun(request,
                client::find,
                FindResponse::getStatus,
                FindResponse::getStatusMessage);
    }

    private TLabelValuesResponse syncLabelValues(TLabelValuesRequest request) {
        return syncSuccessRun(request,
                client::labelValues,
                TLabelValuesResponse::getStatus,
                TLabelValuesResponse::getStatusMessage);
    }

    private TLabelNamesResponse syncLabelNames(TLabelNamesRequest request) {
        return syncSuccessRun(request,
                client::labelNames,
                TLabelNamesResponse::getStatus,
                TLabelNamesResponse::getStatusMessage);
    }

    private DeleteManyResponse syncDeleteMany(DeleteManyRequest request) {
        return syncSuccessRun(request,
                client::deleteMany,
                DeleteManyResponse::getStatus,
                DeleteManyResponse::getStatusMessage);
    }

    private DeleteManyResponse syncFindAndDelete(FindRequest request) {
        FindResponse findResponse = syncFind(request);
        return syncDeleteMany(DeleteManyRequest.newBuilder()
                .setDeadlineMillis(request.getDeadlineMillis())
                .addAllMetrics(findResponse.getMetricsList())
                .build());
    }

    private <Request, Response> Response syncSuccessRun(Request request,
                                                        Function<Request, CompletableFuture<Response>> fn,
                                                        Function<Response, EMetabaseStatusCode> status,
                                                        Function<Response, String> message) {
        Response response = fn.apply(request).join();

        if (status.apply(response) != EMetabaseStatusCode.OK) {
            throw new IllegalStateException(status.apply(response) + ": " + message.apply(response));
        }

        return response;
    }

    private CoremonMetric metric(Labels labels) {
        ThreadLocalRandom random = ThreadLocalRandom.current();
        int shardId = StockpileShardId.random(random);
        long localId = StockpileLocalId.random(random);
        return metric(shardId, localId, labels);
    }

    private CoremonMetric metric(int shardId, long localId, Labels labels) {
        return new FileCoremonMetric(shardId, localId, labels, MetricType.DGAUGE);
    }

    private MetricId metricId(int shardId, long localId) {
        return MetricId.newBuilder().setShardId(shardId).setLocalId(localId).build();
    }

    private Metric metricToProto(CoremonMetric metric) {
        return Metric.newBuilder()
                .setMetricId(MetricId.newBuilder()
                        .setShardId(metric.getShardId())
                        .setLocalId(metric.getLocalId())
                        .build())
                .addAllLabels(LabelConverter.labelsToProtoList(metric.getLabels()))
                .setCreatedAtMillis(System.currentTimeMillis())
                .setType(MetricTypeConverter.toProto(metric.getType()))
                .build();
    }

    private void forceUpdateClusterMetadata() {
        client.forceUpdateClusterMetaData().join();
    }
}
