package ru.yandex.solomon.coremon.meta.service;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.labels.LabelsBuilder;
import ru.yandex.monlib.metrics.labels.validate.LabelValidationFilter;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.CoremonMetricArray;
import ru.yandex.solomon.coremon.meta.FileCoremonMetric;
import ru.yandex.solomon.coremon.meta.file.FileMetricsCollection;
import ru.yandex.solomon.labels.protobuf.LabelConverter;
import ru.yandex.solomon.labels.protobuf.LabelSelectorConverter;
import ru.yandex.solomon.labels.protobuf.LabelValidationFilterConverter;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.metabase.api.protobuf.CreateManyRequest;
import ru.yandex.solomon.metabase.api.protobuf.CreateManyResponse;
import ru.yandex.solomon.metabase.api.protobuf.CreateOneRequest;
import ru.yandex.solomon.metabase.api.protobuf.CreateOneResponse;
import ru.yandex.solomon.metabase.api.protobuf.DeleteManyRequest;
import ru.yandex.solomon.metabase.api.protobuf.DeleteManyResponse;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metabase.api.protobuf.FindRequest;
import ru.yandex.solomon.metabase.api.protobuf.FindResponse;
import ru.yandex.solomon.metabase.api.protobuf.Metric;
import ru.yandex.solomon.metabase.api.protobuf.MetricNamesRequest;
import ru.yandex.solomon.metabase.api.protobuf.MetricNamesResponse;
import ru.yandex.solomon.metabase.api.protobuf.ResolveManyRequest;
import ru.yandex.solomon.metabase.api.protobuf.ResolveManyResponse;
import ru.yandex.solomon.metabase.api.protobuf.ResolveOneRequest;
import ru.yandex.solomon.metabase.api.protobuf.ResolveOneResponse;
import ru.yandex.solomon.metabase.api.protobuf.TLabelNamesRequest;
import ru.yandex.solomon.metabase.api.protobuf.TLabelNamesResponse;
import ru.yandex.solomon.metabase.api.protobuf.TLabelValuesRequest;
import ru.yandex.solomon.metabase.api.protobuf.TLabelValuesResponse;
import ru.yandex.solomon.metabase.api.protobuf.TResolveLogsRequest;
import ru.yandex.solomon.metabase.api.protobuf.TResolveLogsResponse;
import ru.yandex.solomon.metabase.api.protobuf.TServerStatusRequest;
import ru.yandex.solomon.metabase.api.protobuf.TServerStatusResponse;
import ru.yandex.solomon.metabase.api.protobuf.TShardStatus;
import ru.yandex.solomon.metabase.api.protobuf.TSliceOptions;
import ru.yandex.solomon.metabase.api.protobuf.TUniqueLabelsRequest;
import ru.yandex.solomon.metabase.api.protobuf.TUniqueLabelsResponse;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.model.protobuf.MetricTypeConverter;
import ru.yandex.solomon.util.collection.Slicer;
import ru.yandex.solomon.util.labelStats.LabelStatsConverter;
import ru.yandex.solomon.util.labelStats.LabelValuesStats;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.shard.StockpileShardId;


/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class MetabaseServiceTestStub implements MetabaseService, AutoCloseable {
    private final FileMetricsCollection metrics;
    private List<ShardKey> shards;

    public MetabaseServiceTestStub(ShardKey... shards) {
        int[] levelsSizes = { 10, 100, 1000 };
        this.metrics = new FileMetricsCollection("test", ForkJoinPool.commonPool(), levelsSizes, new CoremonMetricArray(0));
        this.shards = Arrays.asList(shards);
    }

    private List<Metric> searchMetrics(Selectors selectors) {
        List<Metric> result = new ArrayList<>();
        metrics.searchMetrics(selectors, s -> result.add(convertMetric(s)));
        return result;
    }

    private Stream<Metric> resolve(Collection<Labels> labels) {
        return labels.stream()
                .map(metrics::getOrNull)
                .filter(Objects::nonNull)
                .map(this::convertMetric);
    }

    @Override
    public CompletableFuture<TServerStatusResponse> serverStatus(TServerStatusRequest request) {
        List<TShardStatus> statuses = shards.stream()
                .map(shard -> TShardStatus.newBuilder()
                        .setShardId(shard.toUniqueId())
                        .setReady(true)
                        .addAllLabels(LabelConverter.labelsToProtoList(shard.toLabels()))
                        .build())
                .collect(Collectors.toList());

        TServerStatusResponse response = TServerStatusResponse.newBuilder()
                .setStatus(EMetabaseStatusCode.OK)
                .addAllPartitionStatus(statuses)
                .setTotalPartitionCountKnown(true)
                .setTotalPartitionCount(statuses.size())
                .build();

        return CompletableFuture.completedFuture(response);
    }

    @Override
    public CompletableFuture<CreateOneResponse> createOne(CreateOneRequest request) {
        ThreadLocalRandom random = ThreadLocalRandom.current();
        Metric metric = request.getMetric();
        int shardId = metric.getMetricId().getShardId();
        long localId = metric.getMetricId().getLocalId();

        if (shardId == 0) {
            shardId = StockpileShardId.random(random);
        }

        if (localId == 0) {
            localId = StockpileLocalId.random(random);
        }

        Labels labels = LabelConverter.protoToLabels(metric.getLabelsList());
        CoremonMetric coremonMetric = new FileCoremonMetric(
                shardId,
                localId,
                labels,
                MetricType.DGAUGE);

        CompletableFuture<Void> future = metrics.put(coremonMetric);

        // TODO: drop it because unnecessary
        return future.thenApply(aVoid -> CreateOneResponse.newBuilder()
                .setStatus(EMetabaseStatusCode.OK)
                .setMetric(convertMetric(coremonMetric))
                .build());
    }

    @Override
    public CompletableFuture<CreateManyResponse> createMany(CreateManyRequest request) {
        ThreadLocalRandom random = ThreadLocalRandom.current();
        int commonShardId = StockpileShardId.random(random);
        Labels commonLabels = LabelConverter.protoToLabels(request.getCommonLabelsList());

        List<Metric> result = new ArrayList<>(request.getMetricsCount());
        CompletableFuture[] futures = new CompletableFuture[request.getMetricsCount()];

        int i = 0;
        for (Metric metric : request.getMetricsList()) {
            int shardId = metric.getMetricId().getShardId();
            long localId = metric.getMetricId().getLocalId();

            if (shardId == 0) {
                shardId = commonShardId;
            }

            if (localId == 0) {
                localId = StockpileLocalId.random(random);
            }

            Labels labels = LabelConverter.protoToLabels(metric.getLabelsList());

            if (!commonLabels.isEmpty()) {
                labels = labels.toBuilder()
                    .addAll(commonLabels)
                    .build();
            }

            CoremonMetric coremonMetric = new FileCoremonMetric(
                    shardId,
                    localId,
                    labels,
                    MetricType.DGAUGE);

            futures[i++] = metrics.put(coremonMetric);
            result.add(convertMetric(coremonMetric));
        }

        return CompletableFuture.allOf(futures)
            .thenApply(aVoid -> CreateManyResponse.newBuilder()
                .setStatus(EMetabaseStatusCode.OK)
                .addAllMetrics(result)
                .build());
    }

    @Override
    public CompletableFuture<ResolveOneResponse> resolveOne(ResolveOneRequest request) {
        ResolveOneResponse response = resolve(Collections.singleton(LabelConverter.protoToLabels(request.getLabelsList())))
                .findFirst()
                .map(metric -> ResolveOneResponse.newBuilder()
                        .setStatus(EMetabaseStatusCode.OK)
                        .setMetric(metric)
                        .build())
                .orElseGet(() -> ResolveOneResponse.newBuilder()
                        .setStatus(EMetabaseStatusCode.NOT_FOUND)
                        .build());

        return CompletableFuture.completedFuture(response);
    }

    @Override
    public CompletableFuture<ResolveManyResponse> resolveMany(ResolveManyRequest request) {
        Labels commonLabels = LabelConverter.protoToLabels(request.getCommonLabelsList());

        List<Labels> requestedLabels = request.getListLabelsList().stream()
            .map(labelList -> {
                Labels labels = LabelConverter.protoToLabels(labelList);
                if (!commonLabels.isEmpty()) {
                    labels = labels.toBuilder()
                        .addAll(commonLabels)
                        .build();
                }
                return labels;
            })
            .collect(Collectors.toList());

        List<Metric> resolvedMetrics = resolve(requestedLabels).collect(Collectors.toList());

        ResolveManyResponse response = ResolveManyResponse.newBuilder()
                .setStatus(EMetabaseStatusCode.OK)
                .addAllMetrics(resolvedMetrics)
                .build();

        return CompletableFuture.completedFuture(response);
    }

    @Override
    public CompletableFuture<DeleteManyResponse> deleteMany(DeleteManyRequest request) {
        List<Labels> lables = request.getMetricsList().stream()
            .map(metric -> LabelConverter.protoToLabels(metric.getLabelsList()))
            .collect(Collectors.toList());

        CompletableFuture<Void> future = deleteMetrics(lables);
        return future.thenApply(aVoid -> DeleteManyResponse.newBuilder()
                .setStatus(EMetabaseStatusCode.OK)
                .build());
    }

    @Override
    public CompletableFuture<FindResponse> find(FindRequest request) {
        Selectors selector = LabelSelectorConverter.protoToSelectors(request.getSelectorsList());

        List<Metric> rawMetrics = searchMetrics(selector);
        int totalCount = rawMetrics.size();
        TSliceOptions sliceOptions = request.getSliceOptions();

        List<Metric> result = Slicer.slice(rawMetrics, sliceOptions.getOffset(), sliceOptions.getLimit());

        return CompletableFuture.completedFuture(FindResponse.newBuilder()
            .setStatus(EMetabaseStatusCode.OK)
            .addAllMetrics(result)
            .setTotalCount(totalCount)
            .build());
    }

    @Override
    public CompletableFuture<MetricNamesResponse> metricNames(MetricNamesRequest request) {
        throw new UnsupportedOperationException("Not implemented");
    }

    @Override
    public CompletableFuture<TLabelValuesResponse> labelValues(TLabelValuesRequest request) {
        Selectors selector = LabelSelectorConverter.protoToSelectors(request.getSelectorsList());
        LabelValidationFilter validationFilter =
            LabelValidationFilterConverter.protoToFilter(request.getValidationFilter());

        Set<String> requestedLabels = ImmutableSet.copyOf(request.getLabelsList());
        LabelValuesStats result = new LabelValuesStats();
        metrics.searchLabels(selector, l -> result.add(l, requestedLabels));

        result.filter(request.getTextSearch());
        result.filter(validationFilter);
        result.limit(request.getLimit());
        return CompletableFuture.completedFuture(LabelStatsConverter.toProto(result));
    }

    @Override
    public CompletableFuture<TLabelNamesResponse> labelNames(TLabelNamesRequest request) {
        Selectors selector =
            LabelSelectorConverter.protoToSelectors(request.getSelectorsList());

        Set<String> names = new HashSet<>();
        metrics.searchLabels(selector, labels -> {
            labels.forEach(label -> {
                names.add(label.getKey());
            });
        });

        List<String> filteredNames = names.stream()
            .filter(name -> !selector.hasKey(name))
            .collect(Collectors.toList());

        return CompletableFuture.completedFuture(TLabelNamesResponse.newBuilder()
            .setStatus(EMetabaseStatusCode.OK)
            .addAllNames(filteredNames)
            .build());
    }

    @Override
    public CompletableFuture<TUniqueLabelsResponse> uniqueLabels(TUniqueLabelsRequest request) {
        Selectors selector =
                LabelSelectorConverter.protoToSelectors(request.getSelectorsList());

        Set<String> requestedNames = ImmutableSet.copyOf(request.getNamesList());
        Set<Labels> uniqueLabels = new HashSet<>();
        LabelsBuilder labelsBuilder = Labels.builder(Labels.MAX_LABELS_COUNT);

        metrics.searchLabels(selector, labels -> {
            labelsBuilder.clear();
            labels.forEach(label -> {
                if (requestedNames.contains(label.getKey())) {
                    labelsBuilder.add(label);
                }
            });

            uniqueLabels.add(labelsBuilder.build());
        });

        TUniqueLabelsResponse.Builder responseBuilder = TUniqueLabelsResponse.newBuilder()
            .setStatus(EMetabaseStatusCode.OK);

        for (Labels labels : uniqueLabels) {
            LabelConverter.addLabels(responseBuilder.addLabelListsBuilder(), labels);
        }
        return CompletableFuture.completedFuture(responseBuilder.build());
    }

    @Override
    public CompletableFuture<TResolveLogsResponse> resolveLogs(TResolveLogsRequest request) {
        return CompletableFuture.failedFuture(new UnsupportedOperationException("Not implemented"));
    }

    private CompletableFuture<Void> deleteMetrics(Collection<Labels> labels) {
        return metrics.removeAll(labels);
    }

    private Metric convertMetric(CoremonMetric metric) {
        return Metric.newBuilder()
            .setMetricId(MetricId.newBuilder()
                    .setShardId(metric.getShardId())
                    .setLocalId(metric.getLocalId())
                    .build())
            .addAllLabels(LabelConverter.labelsToProtoList(metric.getLabels()))
            .setType(MetricTypeConverter.toProto(metric.getType()))
            .setCreatedAtMillis(TimeUnit.SECONDS.toMillis(metric.getCreatedAtSeconds()))
            .build();
    }

    @Override
    public void close() throws Exception {
        metrics.close();
    }
}
