package ru.yandex.solomon.coremon.meta.service.cloud;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.junit.Test;

import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.name.resolver.client.Resource;

import static org.hamcrest.CoreMatchers.anyOf;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThat;
import static ru.yandex.solomon.name.resolver.client.ResourcesTestSupport.staticResource;

/**
 * @author Vladimir Gordiychuk
 */
public class SelectorsTransformTest {

    @Test
    public void resourceSelector() {
        var reference = new ReferenceLabel("resource_id", Set.of(), Set.of(), Set.of());
        {
            var original = selectors("name=cpu_usage, resource_type=vm, resource_id=stockpile-01");
            var expected = selectors("resource=stockpile-01");
            assertEquals(expected, resource(original, reference));
        }
        {
            var original = selectors("name=cpu_usage, resource_type=vm, resource_id!=stockpile-01");
            var expected = selectors("resource!=stockpile-01");
            assertEquals(expected, resource(original, reference));
        }
        {
            var original = selectors("name=cpu_usage, resource_type=vm, resource_id==stockpile-01");
            var expected = selectors("resource==stockpile-01");
            assertEquals(expected, resource(original, reference));
        }
        {
            var original = selectors("name=cpu_usage, resource_type=vm, resource_id=stockpile-*");
            var expected = selectors("resource=stockpile-*");
            assertEquals(expected, resource(original, reference));
        }
        {
            var original = selectors("name=cpu_usage, resource_type=vm, resource_id!=cluster|sas|man|vla");
            var expected = selectors("resource!=cluster|sas|man|vla");
            assertEquals(expected, resource(original, reference));
        }
    }

    @Test
    public void resourceSelectorManyReferenceUse() {
        var reference = new ReferenceLabel("certificate", Set.of(), Set.of() , Set.of());
        var original = selectors("name=timeToLive, certificate=stockpile-*, certificate!=*sas*");
        var expected = selectors("resource=stockpile-*, resource!=*sas*");
        assertEquals(expected, resource(original, reference));
    }

    @Test
    public void resourceSelectorTypeMatch() {
        var reference = new ReferenceLabel("resource_id", Set.of("vm"), Set.of(), Set.of());
        {
            var original = selectors("name=cpu_usage, resource_id=stockpile-01");
            var expected = selectors("resource=stockpile-01, type=vm");
            assertEquals(expected, resource(original, reference));
        }
        {
            var original = selectors("name=cpu_usage, resource_id!=stockpile-01");
            var expected = selectors("resource!=stockpile-01, type=vm");
            assertEquals(expected, resource(original, reference));
        }
        {
            var original = selectors("name=cpu_usage, resource_id==stockpile-01");
            var expected = selectors("resource==stockpile-01, type=vm");
            assertEquals(expected, resource(original, reference));
        }
        {
            var original = selectors("name=cpu_usage, resource_id=stockpile-*");
            var expected = selectors("resource=stockpile-*, type=vm");
            assertEquals(expected, resource(original, reference));
        }
        {
            var original = selectors("name=cpu_usage, resource_id!=cluster|sas|man|vla");
            var expected = selectors("resource!=cluster|sas|man|vla, type=vm");
            assertEquals(expected, resource(original, reference));
        }
    }

    @Test
    public void resourceSelectorMultipleTypes() {
        var reference = new ReferenceLabel("resource_id", Set.of("vm", "disk"), Set.of(), Set.of());
        {
            var original = selectors("name=cpu_usage, resource_id=stockpile-01");
            assertThat(resource(original, reference), anyOf(
                    equalTo(selectors("resource=stockpile-01, type=vm|disk")),
                    equalTo(selectors("resource=stockpile-01, type=disk|vm"))
            ));
        }
        {
            var original = selectors("name=cpu_usage, resource_id!=stockpile-01");
            assertThat(resource(original, reference), anyOf(
                    equalTo(selectors("resource!=stockpile-01, type=vm|disk")),
                    equalTo(selectors("resource!=stockpile-01, type=disk|vm"))
            ));
        }
        {
            var original = selectors("name=cpu_usage, resource_id==stockpile-01");
            assertThat(resource(original, reference), anyOf(
                    equalTo(selectors("resource==stockpile-01, type=vm|disk")),
                    equalTo(selectors("resource==stockpile-01, type=disk|vm"))
            ));
        }
        {
            var original = selectors("name=cpu_usage, resource_id=stockpile-*");
            assertThat(resource(original, reference), anyOf(
                    equalTo(selectors("resource=stockpile-*, type=vm|disk")),
                    equalTo(selectors("resource=stockpile-*, type=disk|vm"))
            ));
        }
        {
            var original = selectors("name=cpu_usage, resource_type=vm, resource_id!=cluster|sas|man|vla");
            assertThat(resource(original, reference), anyOf(
                    equalTo(selectors("resource!=cluster|sas|man|vla, type=vm|disk")),
                    equalTo(selectors("resource!=cluster|sas|man|vla, type=disk|vm"))
            ));
        }
    }

    @Test
    public void resourceSelectorCrossFolder() {
        var reference = new ReferenceLabel("resource_id", Set.of(), Set.of(), Set.of());
        var original = selectors("name=cpu_usage, cluster=my-folder, resource_id=stockpile-01");
        var expected = selectors("resource=stockpile-01");
        assertEquals(expected, resource(original, reference));
    }

    @Test
    public void resourceSelectorFolder() {
        var reference = new ReferenceLabel("resource_id", Set.of(), Set.of(), Set.of("my-folder"));
        var original = selectors("name=cpu_usage, cluster=my-folder, resource_id=stockpile-01");
        var expected = selectors("resource=stockpile-01, folderId=my-folder");
        assertEquals(expected, resource(original, reference));
    }

    @Test
    public void resourceSelectorAbsentReference() {
        {
            var reference = new ReferenceLabel("vm", Set.of("vm"), Set.of(), Set.of());
            var original = selectors("name=cpu_usage");
            var expected = selectors("type=vm");
            assertEquals(expected, resource(original, reference));
        }
        {
            var reference = new ReferenceLabel("resource_id", Set.of(), Set.of(), Set.of());
            var original = selectors("name=cpu_usage");
            var expected = Selectors.of();
            assertEquals(expected, resource(original, reference));
        }
    }

    @Test
    public void toMetricSelectorAbsentResource() {
        var selector = Selector.glob("resource_id", "my-id");
        assertEquals(selector, metric(selector, List.of()));
    }

    @Test
    public void toMetricsAnySelector() {
        {
            var selector = Selector.any("certificate");
            assertEquals(selector, metric(selector, List.of(resource("one"), resource("two"))));
        }
        {
            var selector = selectors("vm=*").at(0);
            assertEquals(selector, metric(selector, List.of(resource("one"), resource("two"))));
        }
    }

    @Test
    public void toMetricsAbsentSelector() {
        {
            var selector = Selector.absent("certificate");
            assertEquals(selector, metric(selector, List.of(resource("one"), resource("two"))));
        }
        {
            var selector = selectors("vm=-").at(0);
            assertEquals(selector, metric(selector, List.of(resource("one"), resource("two"))));
        }
    }

    @Test
    public void toMetricsCombineOriginalAndReference() {
        {
            var selector = Selector.glob("certificate", "my-name");
            var actual = metric(selector, List.of(resource("my-id")));
            assertEquals(actual, Selector.glob("certificate", "my-name|my-id"));
        }
        {
            var selector = Selector.exact("certificate", "my-name");
            var actual = metric(selector, List.of(resource("my-id")));
            assertEquals(actual, Selector.glob("certificate", "my-name|my-id"));
        }
        {
            var selector = Selector.exact("certificate", "my-name");
            var actual = metric(selector, List.of(resource("one"), resource("two")));
            assertThat(actual, anyOf(
                    equalTo(Selector.glob("certificate", "my-name|one|two")),
                    equalTo(Selector.glob("certificate", "my-name|two|one"))
            ));
        }
    }

    @Test
    public void toMetricsUpdateReplaceSelector() {
        var selector = Selector.notGlob("resource_id", "non-expect-name");
        {
            var actual = metric(selector, List.of(resource("one")));
            assertEquals(actual, Selector.glob("resource_id", "one"));
        }
        {
            var actual = metric(selector, List.of(resource("one"), resource("two")));
            assertThat(actual, anyOf(
                    equalTo(Selector.glob("resource_id", "one|two")),
                    equalTo(Selector.glob("resource_id", "two|one"))
            ));
        }
    }

    @Test
    public void toResourceSelectorsSkip() {
        assertEquals(Map.of(), SelectorsTransform.toResourceSelectors(selectors(""), Map.of()));
        assertEquals(Map.of(), SelectorsTransform.toResourceSelectors(selectors("name=test"), Map.of()));
        {
            // absent reference in selector
            var reference = new ReferenceLabel("resource_id", Set.of(), Set.of(), Set.of());
            assertEquals(Map.of(), SelectorsTransform.toResourceSelectors(selectors("name=test"), Map.of(reference.label(), reference)));
        }
        {
            // any match equal to absent selector at all
            var reference = new ReferenceLabel("resource_id", Set.of(), Set.of(), Set.of());
            assertEquals(Map.of(), SelectorsTransform.toResourceSelectors(selectors("resource_id=*"), Map.of(reference.label(), reference)));
        }
        {
            // absent match anyway return zero resources
            var reference = new ReferenceLabel("resource_id", Set.of(), Set.of(), Set.of());
            assertEquals(Map.of(), SelectorsTransform.toResourceSelectors(selectors("resource_id=-"), Map.of(reference.label(), reference)));
        }
    }

    private Resource resource(String id) {
        return staticResource().setCloudId("").setResourceId(id);
    }

    private Selectors selectors(String text) {
        return Selectors.parse(text);
    }

    private Selectors resource(Selectors selectors, ReferenceLabel reference) {
        return SelectorsTransform.toResourceSelectors(selectors, reference);
    }

    private Selector metric(Selector selector, Collection<Resource> resources) {
        return SelectorsTransform.toMetricSelector(selector, resources);
    }
}
