package ru.yandex.solomon.coremon.meta.service.handler;

import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.base.Strings;
import org.junit.Test;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.coremon.meta.service.ShardKeyAndMetricKey;
import ru.yandex.solomon.labels.LabelsFormat;
import ru.yandex.solomon.labels.protobuf.LabelConverter;
import ru.yandex.solomon.labels.protobuf.LabelSelectorConverter;
import ru.yandex.solomon.labels.query.SelectorsFormat;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metabase.api.protobuf.FindRequest;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;

/**
 * @author Vladimir Gordiychuk
 */
public class FindHandlerTest extends HandlerTestBase {
    @Test
    public void findEmpty() {
        assertEquals(List.of(), find(SHARD_CERTIFICATE_MANAGER, "certificate=*"));
        assertEquals(List.of(), find(SHARD_COMPUTE, ""));
        assertEquals(List.of(), find(SHARD_CERTIFICATE_MANAGER, "host=*"));
    }

    @Test
    public void findMetricWithoutResourceReference() {
        var shard = shardResolver.resolveShard(SHARD_CERTIFICATE_MANAGER);
        var usage = LabelsFormat.parse("name=quota.certificates_count.usage");
        var limit = LabelsFormat.parse("name=quota.certificates_count.limit");
        addMetrics(shard, usage, limit);
        assertEquals(List.of(usage), find(SHARD_CERTIFICATE_MANAGER, "name=quota.certificates_count.usage"));
        assertEquals(List.of(usage), find(SHARD_CERTIFICATE_MANAGER, "name=quota.certificates_count.us*"));
        assertEquals(List.of(limit), find(SHARD_CERTIFICATE_MANAGER, "name=quota.certificates_count.l*"));
        assertEquals(List.of(limit), find(SHARD_CERTIFICATE_MANAGER, "'quota.certificates_count.l*'{}"));
    }

    @Test
    public void findWhenNameUnknown() {
        var shard = shardResolver.resolveShard(SHARD_CERTIFICATE_MANAGER);
        var certificate = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=fd3cdv65706qam9l4cob");
        addMetrics(shard, certificate);

        assertEquals(List.of(certificate), find(SHARD_CERTIFICATE_MANAGER, "name=certificate.days_until_expiration, certificate=fd3cdv65706qam9l4cob"));
        assertEquals(List.of(certificate), find(SHARD_CERTIFICATE_MANAGER, "certificate.days_until_expiration{certificate=fd3cdv65706qam9l4cob}"));
        assertEquals(List.of(certificate), find(SHARD_CERTIFICATE_MANAGER, "certificate=fd3cdv65706qam9l4cob"));
        assertEquals(List.of(certificate), find(SHARD_CERTIFICATE_MANAGER, "name=certificate.days_until_expiration, certificate=*"));
        assertEquals(List.of(certificate), find(SHARD_CERTIFICATE_MANAGER, "certificate.days_until_expiration{certificate=*}"));
        assertEquals(List.of(certificate), find(SHARD_CERTIFICATE_MANAGER, "name=certificate.days_until_expiration"));
        assertEquals(List.of(certificate), find(SHARD_CERTIFICATE_MANAGER, "certificate.days_until_expiration{}"));
        assertEquals(List.of(certificate), find(SHARD_CERTIFICATE_MANAGER, ""));
        assertEquals(List.of(), find(SHARD_CERTIFICATE_MANAGER, "certificate!=fd3cdv65706qam9l4cob"));
    }

    @Test
    public void findByNameBeforeMetabase() {
        var shard = shardResolver.resolveShard(SHARD_CERTIFICATE_MANAGER);
        var certificate = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=fd3cdv65706qam9l4cob");
        var certificateName = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=my");
        addMetrics(shard, certificate);
        addResource(shard, "certificate", "fd3cdv65706qam9l4cob", "my");
        assertEquals(List.of(certificateName), find(SHARD_CERTIFICATE_MANAGER, "certificate=my"));
        assertEquals(List.of(certificateName), find(SHARD_CERTIFICATE_MANAGER, "certificate=*"));
        assertEquals(List.of(certificateName), find(SHARD_CERTIFICATE_MANAGER, "name=certificate.days_until_expiration, certificate=fd3cdv65706qam9l4cob"));
        assertEquals(List.of(certificateName), find(SHARD_CERTIFICATE_MANAGER, "certificate.days_until_expiration{certificate=fd3cdv65706qam9l4cob}"));
    }

    @Test
    public void findByNameAfterMetabase() {
        var shard = shardResolver.resolveShard(SHARD_CERTIFICATE_MANAGER);
        var certificate = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=fd3cdv65706qam9l4cob");
        var certificateName = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=mon-cert");
        addMetrics(shard, certificate);
        addResource(shard, "certificate", "fd3cdv65706qam9l4cob", "mon-cert");
        assertEquals(List.of(certificateName), find(SHARD_CERTIFICATE_MANAGER, "name=certificate.days_until_expiration"));
        assertEquals(List.of(certificateName), find(SHARD_CERTIFICATE_MANAGER, ""));
        assertEquals(List.of(), find(SHARD_CERTIFICATE_MANAGER, "name=quota.certificates_count.usage"));
    }

    @Test
    public void findMultipleReference() {
        var shard = shardResolver.resolveShard(SHARD_COMPUTE);
        var diskLatency = LabelsFormat.parse("name='disk_write_latency', resource_id='epdn6uas6ujoonvd99ck', bin='100', device='epdplhd148habqj0np92'");
        var diskLatencyName = LabelsFormat.parse("name='disk_write_latency', resource_id='solomon-stockpile-sas-01', bin='100', device='hdd-01'");

        addMetrics(shard, diskLatency);
        addResource(shard, "vm", "epdn6uas6ujoonvd99ck", "solomon-stockpile-sas-01");
        addResource(shard, "disk", "epdplhd148habqj0np92", "hdd-01");
        assertEquals(List.of(diskLatencyName), find(SHARD_COMPUTE, ""));
        assertEquals(List.of(diskLatencyName), find(SHARD_COMPUTE, "name=disk_write_latency, resource_id=solomon-stockpile-sas-01"));
        assertEquals(List.of(diskLatencyName), find(SHARD_COMPUTE, "name=disk_write_latency, resource_id=*, device=hdd-01"));
        assertEquals(List.of(diskLatencyName), find(SHARD_COMPUTE, "name=disk_write_latency, device=hdd-01"));
        assertEquals(List.of(diskLatencyName), find(SHARD_COMPUTE, "name=disk_write_latency, resource_id=*, device=epdplhd148habqj0np92"));
        assertEquals(List.of(diskLatencyName), find(SHARD_COMPUTE, "name=disk_write_latency, device=epdplhd148habqj0np92"));
    }

    @Test // TODO: decide what todo with negative match by name or id, maybe need limit match by resource_id to only exact match (gordiychuk@)
    public void findByNotName() {
        var shard = shardResolver.resolveShard(SHARD_COMPUTE);
        var diskLatencyOne = LabelsFormat.parse("name='disk_write_latency', resource_id='epdn6uas6ujoonvd99ck', bin='100', device='epdplhd148habqj0np92'");
        var diskLatencyOneName = LabelsFormat.parse("name='disk_write_latency', resource_id='solomon-stockpile-sas-01', bin='100', device='hdd-01'");
        var diskLatencyTwo = LabelsFormat.parse("name='disk_write_latency', resource_id='epdhh61pg5vnviq7sfkd', bin='100', device='epd9629qg0o6offba1m7'");
        var diskLatencyTwoName = LabelsFormat.parse("name='disk_write_latency', resource_id='solomon-stockpile-sas-16', bin='100', device='hdd-16'");

        addMetrics(shard, diskLatencyOne);
        addMetrics(shard, diskLatencyTwo);
        addResource(shard, "vm", "epdn6uas6ujoonvd99ck", "solomon-stockpile-sas-01");
        addResource(shard, "vm", "epdhh61pg5vnviq7sfkd", "solomon-stockpile-sas-16");
        addResource(shard, "disk", "epdplhd148habqj0np92", "hdd-01");
        addResource(shard, "disk", "epd9629qg0o6offba1m7", "hdd-16");
        assertEquals(List.of(diskLatencyTwoName), find(SHARD_COMPUTE, "name=disk_write_latency, device!=hdd-01"));
        assertEquals(List.of(diskLatencyTwoName), find(SHARD_COMPUTE, "name=disk_write_latency, resource_id=*, device!=hdd-01"));
        assertEquals(List.of(diskLatencyOneName), find(SHARD_COMPUTE, "name=disk_write_latency, device!=hdd-16"));
        assertEquals(List.of(diskLatencyOneName), find(SHARD_COMPUTE, "name=disk_write_latency, device!=hdd-16"));
        assertEquals(List.of(diskLatencyOneName), find(SHARD_COMPUTE, "name=disk_write_latency, resource_id!=solomon-stockpile-sas-16, device!=hdd-16"));
        assertEquals(List.of(diskLatencyOneName), find(SHARD_COMPUTE, "name=disk_write_latency, resource_id!=solomon-stockpile-sas-16"));
        assertEquals(List.of(diskLatencyTwoName), find(SHARD_COMPUTE, "name=disk_write_latency, resource_id!=solomon-stockpile-sas-0*"));
    }

    @Test
    public void doNotLostSelectorName() {
        var shard = shardResolver.resolveShard(SHARD_COMPUTE);
        var diskLatency = LabelsFormat.parse("name='disk_write_latency', resource_id='epdn6uas6ujoonvd99ck', bin='100', device='epdplhd148habqj0np92'");
        var diskLatencyName = LabelsFormat.parse("name='disk_write_latency', resource_id='solomon-stockpile-sas-01', bin='100', device='hdd-01'");
        var cpuUsage = LabelsFormat.parse("name='cpu_usage', resource_id='epdn6uas6ujoonvd99ck'");
        var cpuUsageName = LabelsFormat.parse("name='cpu_usage', resource_id='solomon-stockpile-sas-01'");

        addMetrics(shard, diskLatency);
        addMetrics(shard, cpuUsage);
        addResource(shard, "vm", "epdn6uas6ujoonvd99ck", "solomon-stockpile-sas-01");
        addResource(shard, "disk", "epdplhd148habqj0np92", "hdd-01");
        assertEquals(List.of(cpuUsageName), find(SHARD_COMPUTE, "cpu_usage{resource_id=solomon-stockpile-sas-01}"));
        assertEquals(List.of(cpuUsageName), find(SHARD_COMPUTE, "cpu_usage{}"));
        assertEquals(List.of(diskLatencyName), find(SHARD_COMPUTE, "disk_write_latency{}"));
        assertEquals(List.of(diskLatencyName), find(SHARD_COMPUTE, "disk_write_latency{device=hdd-01}"));
    }

    @Test
    public void findIgnoreReplacedMetrics() {
        var shard = shardResolver.resolveShard(SHARD_CERTIFICATE_MANAGER);
        var certificateReplaced = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=fd3ae6fi6fc00en0qkm9");
        var certificate = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=fd3cdv65706qam9l4cob");
        var certificateName = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=my");
        addMetrics(shard, certificateReplaced);
        addMetrics(shard, certificate);
        addResource(shard, "certificate", "fd3ae6fi6fc00en0qkm9", "my", true);
        addResource(shard, "certificate", "fd3cdv65706qam9l4cob", "my", false);
        assertEquals(List.of(certificateName), find(SHARD_CERTIFICATE_MANAGER, "certificate=my"));
        assertEquals(List.of(certificateName), find(SHARD_CERTIFICATE_MANAGER, "certificate=m*"));
        assertEquals(List.of(certificateName), find(SHARD_CERTIFICATE_MANAGER, "certificate=*"));
        assertEquals(List.of(certificateName), find(SHARD_CERTIFICATE_MANAGER, ""));
    }

    @Test
    public void findMultipleValues() {
        var shard = shardResolver.resolveShard(SHARD_COMPUTE);
        var diskLatencyOne = LabelsFormat.parse("name='disk_write_latency', resource_id='epdn6uas6ujoonvd99ck', bin='100', device='epdplhd148habqj0np92'");
        var diskLatencyOneName = LabelsFormat.parse("name='disk_write_latency', resource_id='solomon-stockpile-sas-01', bin='100', device='hdd-01'");
        var diskLatencyTwo = LabelsFormat.parse("name='disk_write_latency', resource_id='epdhh61pg5vnviq7sfkd', bin='100', device='epd9629qg0o6offba1m7'");
        var diskLatencyTwoName = LabelsFormat.parse("name='disk_write_latency', resource_id='solomon-stockpile-sas-16', bin='100', device='hdd-16'");

        addMetrics(shard, diskLatencyOne);
        addMetrics(shard, diskLatencyTwo);
        addResource(shard, "vm", "epdn6uas6ujoonvd99ck", "solomon-stockpile-sas-01");
        addResource(shard, "vm", "epdhh61pg5vnviq7sfkd", "solomon-stockpile-sas-16");
        addResource(shard, "disk", "epdplhd148habqj0np92", "hdd-01");
        addResource(shard, "disk", "epd9629qg0o6offba1m7", "hdd-16");

        assertEquals(Set.of(diskLatencyOneName, diskLatencyTwoName), Set.copyOf(find(SHARD_COMPUTE, "name=disk_write_latency, device=hdd-01|hdd-16")));
        assertEquals(Set.of(diskLatencyOneName, diskLatencyTwoName), Set.copyOf(find(SHARD_COMPUTE, "name=disk_write_latency, resource_id=epdn6uas6ujoonvd99ck|epdhh61pg5vnviq7sfkd")));
    }

    @Test
    public void ableToFindDeviceById() {
        // CLOUDSUPPORT-104910
        var shard = shardResolver.resolveShard(SHARD_COMPUTE);
        var diskLatencyOne = LabelsFormat.parse("bin='1', name='disk_read_latency', resource_type='vm', resource_id='fhms9icnngpg88jesvo7', device='fhmhgrkddjeie9crrq9m'");
        var diskLatencyOneName = LabelsFormat.parse("bin='1', name='disk_read_latency', resource_type='vm', resource_id='masteralgous', device='disk1'");

        addMetrics(shard);
        addMetrics(shard, diskLatencyOne);
        addResource(shard, "vm", "fhms9icnngpg88jesvo7", "masteralgous");
        addResource(shard, "disk", "fhmhgrkddjeie9crrq9m", "disk1");

        assertEquals(List.of(diskLatencyOneName), find(SHARD_COMPUTE, "name='disk_read_latency', bin='1', resource_type='vm', device='fhmhgrkddjeie9crrq9m'"));
        assertEquals(List.of(diskLatencyOneName), find(SHARD_COMPUTE, "name='disk_read_latency', bin='1', resource_type='vm', device='disk1'"));
    }

    private List<Labels> find(int numId, String selectors) {
        var handler = new FindHandler(shardResolver, referenceResolver, resourceFinder);
        var parsedSelectors = SelectorsFormat.parse(selectors);
        var req = FindRequest.newBuilder()
                .setShardId(numId);
        if (parsedSelectors.getNameSelector().isEmpty()) {
            req.addAllSelectors(LabelSelectorConverter.selectorsToProto(SelectorsFormat.parse(selectors)));
        } else {
            req.setNewSelectors(LabelSelectorConverter.selectorsToNewProto(parsedSelectors));
        }

        var response = handler.find(req.build()).join();
        assertEquals(response.getStatusMessage(), EMetabaseStatusCode.OK, response.getStatus());
        return response.getMetricsList()
                .stream()
                .peek(metric -> {
                    assertNotEquals(metric.toString(), 0, metric.getMetricId().getShardId());
                    assertNotEquals(metric.toString(),0, metric.getMetricId().getLocalId());
                })
                .map(metric -> {
                    var labels = LabelConverter.protoToLabels(metric.getLabelsList());
                    if (!Strings.isNullOrEmpty(metric.getName())) {
                        labels = labels.add("name", metric.getName());
                    }

                    return labels;
                })
                .map(labels -> ShardKeyAndMetricKey.of(labels).getMetricKey())
                .collect(Collectors.toList());
    }
}
