package ru.yandex.solomon.coremon.meta.service.handler;

import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ForkJoinPool;
import java.util.stream.Collectors;

import org.junit.After;
import org.junit.Before;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.coremon.meta.db.memory.InMemoryMetricsDaoFactory;
import ru.yandex.solomon.coremon.meta.service.MetabaseShard;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardConf;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardImpl;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardResolverStub;
import ru.yandex.solomon.coremon.meta.service.cloud.NameResolverResourceFinder;
import ru.yandex.solomon.coremon.meta.service.cloud.ReferenceLabel;
import ru.yandex.solomon.coremon.meta.service.cloud.ReferenceResolverStub;
import ru.yandex.solomon.coremon.meta.service.cloud.ResourceFinder;
import ru.yandex.solomon.labels.protobuf.LabelConverter;
import ru.yandex.solomon.metabase.api.protobuf.Metric;
import ru.yandex.solomon.metrics.client.StockpileClientStub;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.name.resolver.client.NameResolverClientStub;
import ru.yandex.solomon.name.resolver.client.Resource;

public class HandlerTestBase {
    protected static final int SHARD_COMPUTE = 1;
    protected static final int SHARD_CERTIFICATE_MANAGER = 2;
    protected static final List<MetabaseShardConf> SHARDS = List.of(
            MetabaseShardConf.of("alice", "", "pre", "compute", "name", SHARD_COMPUTE, 1),
            MetabaseShardConf.of("bob", "folder", "prod", "certificate-manager", "name", SHARD_CERTIFICATE_MANAGER, 1));

    protected MetabaseShardResolverStub shardResolver;
    protected NameResolverClientStub nameResolverClient;
    protected ReferenceResolverStub referenceResolver;
    protected ResourceFinder resourceFinder;

    @Before
    public void setUp() {
        var stockpileClient = new StockpileClientStub(ForkJoinPool.commonPool());
        var metricDao = new InMemoryMetricsDaoFactory();

        shardResolver = new MetabaseShardResolverStub(SHARDS, metricDao, stockpileClient);
        shardResolver.awaitShardsReady();
        nameResolverClient = new NameResolverClientStub();
        referenceResolver = new ReferenceResolverStub(shardResolver);
        referenceResolver.addReference("compute", new ReferenceLabel("resource_id", Set.of("vm"), Set.of("compute"), Set.of("pre")));
        referenceResolver.addReference("compute", new ReferenceLabel("device", Set.of("disk"),  Set.of("compute"), Set.of("pre")));
        referenceResolver.addReference("certificate-manager", new ReferenceLabel("certificate", Set.of("certificate"),  Set.of("certificate-manager"), Set.of("prod")));
        resourceFinder = new NameResolverResourceFinder(nameResolverClient);
    }

    @After
    public void tearDown() throws Exception {
        if (shardResolver != null) {
            shardResolver.close();
        }
    }

    protected void addMetrics(MetabaseShardImpl shard, Labels... metrics) {
        var toCreate = Arrays.stream(metrics)
                .map(labels -> Metric.newBuilder()
                        .setType(MetricType.DGAUGE)
                        .addAllLabels(LabelConverter.labelsToProtoList(labels))
                        .build())
                .collect(Collectors.toList());
        shard.createMetrics(Labels.empty(), toCreate).join();
    }

    protected void addResource(MetabaseShard shard, String type, String resourceId, String name) {
        addResource(shard, type, resourceId, name, false);
    }

    protected void addResource(MetabaseShard shard, String type, String resourceId, String name, boolean replaced) {
        nameResolverClient.addResource(shard.getShardKey().getProject(), new Resource()
                .setCloudId(shard.getShardKey().getProject())
                .setService(shard.getShardKey().getService())
                .setFolderId(shard.getShardKey().getCluster())
                .setType(type)
                .setResourceId(resourceId)
                .setName(name)
                .setUpdatedAt(System.nanoTime())
                .setReplaced(replaced));
    }
}
