package ru.yandex.solomon.coremon.meta.service.handler;

import java.util.Set;

import org.junit.Test;

import ru.yandex.solomon.labels.LabelsFormat;
import ru.yandex.solomon.labels.protobuf.LabelSelectorConverter;
import ru.yandex.solomon.labels.query.SelectorsFormat;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metabase.api.protobuf.TLabelNamesRequest;

import static org.junit.Assert.assertEquals;

/**
 * @author Vladimir Gordiychuk
 */
public class LabelNamesHandlerTest extends HandlerTestBase {

    @Test
    public void labelNamesEmpty() {
        assertEquals(Set.of(), labelNames(SHARD_CERTIFICATE_MANAGER, "certificate=*"));
        assertEquals(Set.of(), labelNames(SHARD_CERTIFICATE_MANAGER, "certificate=abc"));
        assertEquals(Set.of(), labelNames(SHARD_COMPUTE, ""));
        assertEquals(Set.of(), labelNames(SHARD_CERTIFICATE_MANAGER, "host=*"));
    }

    @Test
    public void labelNamesMetricWithoutResourceReference() {
        var shard = shardResolver.resolveShard(SHARD_CERTIFICATE_MANAGER);
        var usage = LabelsFormat.parse("name=quota.certificates_count.usage, host=cluster");
        var limit = LabelsFormat.parse("name=quota.certificates_count.limit, host=cluster");
        addMetrics(shard, usage, limit);
        assertEquals(Set.of("host"), labelNames(SHARD_CERTIFICATE_MANAGER, "name=quota.certificates_count.usage"));
        assertEquals(Set.of("host"), labelNames(SHARD_CERTIFICATE_MANAGER, "name=quota.certificates_count.us*"));
        assertEquals(Set.of("host"), labelNames(SHARD_CERTIFICATE_MANAGER, "name=quota.certificates_count.l*"));
        assertEquals(Set.of("name", "host"), labelNames(SHARD_CERTIFICATE_MANAGER, ""));
    }

    @Test
    public void labelNamesWhenResourceNameUnknown() {
        var shard = shardResolver.resolveShard(SHARD_CERTIFICATE_MANAGER);
        var certificate = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=fd3cdv65706qam9l4cob");
        addMetrics(shard, certificate);

        assertEquals(Set.of(), labelNames(SHARD_CERTIFICATE_MANAGER, "name=certificate.days_until_expiration, certificate=fd3cdv65706qam9l4cob"));
        assertEquals(Set.of("name"), labelNames(SHARD_CERTIFICATE_MANAGER, "certificate=fd3cdv65706qam9l4cob"));
        assertEquals(Set.of(), labelNames(SHARD_CERTIFICATE_MANAGER, "name=certificate.days_until_expiration, certificate=*"));
        assertEquals(Set.of("certificate"), labelNames(SHARD_CERTIFICATE_MANAGER, "name=certificate.days_until_expiration"));
        assertEquals(Set.of("name", "certificate"), labelNames(SHARD_CERTIFICATE_MANAGER, ""));
        assertEquals(Set.of(), labelNames(SHARD_CERTIFICATE_MANAGER, "certificate!=fd3cdv65706qam9l4cob"));
    }

    @Test
    public void labelNamesByNameBeforeMetabase() {
        var shard = shardResolver.resolveShard(SHARD_CERTIFICATE_MANAGER);
        var certificate = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=fd3cdv65706qam9l4cob");
        addMetrics(shard, certificate);
        addResource(shard, "certificate", "fd3cdv65706qam9l4cob", "my");
        assertEquals(Set.of("name"), labelNames(SHARD_CERTIFICATE_MANAGER, "certificate=my"));
        assertEquals(Set.of("name"), labelNames(SHARD_CERTIFICATE_MANAGER, "certificate=*"));
        assertEquals(Set.of(), labelNames(SHARD_CERTIFICATE_MANAGER, "name=certificate.days_until_expiration, certificate=fd3cdv65706qam9l4cob"));
    }

    @Test
    public void labelNamesByNameAfterMetabase() {
        var shard = shardResolver.resolveShard(SHARD_CERTIFICATE_MANAGER);
        var certificate = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=fd3cdv65706qam9l4cob");
        addMetrics(shard, certificate);
        addResource(shard, "certificate", "fd3cdv65706qam9l4cob", "mon-cert");
        assertEquals(Set.of("certificate"), labelNames(SHARD_CERTIFICATE_MANAGER, "name=certificate.days_until_expiration"));
        assertEquals(Set.of("name", "certificate"), labelNames(SHARD_CERTIFICATE_MANAGER, ""));
        assertEquals(Set.of(), labelNames(SHARD_CERTIFICATE_MANAGER, "name=quota.certificates_count.usage"));
    }

    @Test
    public void labelNameMultipleReference() {
        var shard = shardResolver.resolveShard(SHARD_COMPUTE);
        var diskLatency = LabelsFormat.parse("name='disk_write_latency', resource_id='epdn6uas6ujoonvd99ck', bin='100', device='epdplhd148habqj0np92'");

        addMetrics(shard, diskLatency);
        addResource(shard, "vm", "epdn6uas6ujoonvd99ck", "solomon-stockpile-sas-01");
        addResource(shard, "disk", "epdplhd148habqj0np92", "hdd-01");
        assertEquals(Set.of("name", "resource_id", "bin", "device"), labelNames(SHARD_COMPUTE, ""));
        assertEquals(Set.of("bin", "device"), labelNames(SHARD_COMPUTE, "name=disk_write_latency, resource_id=solomon-stockpile-sas-01"));
        assertEquals(Set.of("bin"), labelNames(SHARD_COMPUTE, "name=disk_write_latency, resource_id=*, device=hdd-01"));
        assertEquals(Set.of("resource_id", "bin"), labelNames(SHARD_COMPUTE, "name=disk_write_latency, device=hdd-01"));
        assertEquals(Set.of("bin"), labelNames(SHARD_COMPUTE, "name=disk_write_latency, resource_id=*, device=epdplhd148habqj0np92"));
        assertEquals(Set.of("resource_id", "bin"), labelNames(SHARD_COMPUTE, "name=disk_write_latency, device=epdplhd148habqj0np92"));
    }

    @Test
    public void labelNamesByNotName() {
        var shard = shardResolver.resolveShard(SHARD_COMPUTE);
        var diskLatencyOne = LabelsFormat.parse("name='disk_write_latency', resource_id='epdn6uas6ujoonvd99ck', bin='100', device='epdplhd148habqj0np92'");
        var diskLatencyTwo = LabelsFormat.parse("name='disk_write_latency', resource_id='epdhh61pg5vnviq7sfkd', bin='100', device='epd9629qg0o6offba1m7'");

        addMetrics(shard, diskLatencyOne);
        addMetrics(shard, diskLatencyTwo);
        addResource(shard, "vm", "epdn6uas6ujoonvd99ck", "solomon-stockpile-sas-01");
        addResource(shard, "vm", "epdhh61pg5vnviq7sfkd", "solomon-stockpile-sas-16");
        addResource(shard, "disk", "epdplhd148habqj0np92", "hdd-01");
        addResource(shard, "disk", "epd9629qg0o6offba1m7", "hdd-16");
        assertEquals(Set.of("resource_id", "bin"), labelNames(SHARD_COMPUTE, "name=disk_write_latency, device!=hdd-01"));
        assertEquals(Set.of("bin"), labelNames(SHARD_COMPUTE, "name=disk_write_latency, resource_id=*, device!=hdd-01"));
        assertEquals(Set.of("resource_id", "bin"), labelNames(SHARD_COMPUTE, "name=disk_write_latency, device!=hdd-16"));
        assertEquals(Set.of("resource_id", "bin"), labelNames(SHARD_COMPUTE, "name=disk_write_latency, device!=hdd-16"));
        assertEquals(Set.of("bin"), labelNames(SHARD_COMPUTE, "name=disk_write_latency, resource_id!=solomon-stockpile-sas-16, device!=hdd-16"));
        assertEquals(Set.of("bin", "device"), labelNames(SHARD_COMPUTE, "name=disk_write_latency, resource_id!=solomon-stockpile-sas-16"));
        assertEquals(Set.of("bin", "device"), labelNames(SHARD_COMPUTE, "name=disk_write_latency, resource_id!=solomon-stockpile-sas-0*"));
    }

    @Test
    public void labelNamesIgnoreReplacedMetrics() {
        var shard = shardResolver.resolveShard(SHARD_CERTIFICATE_MANAGER);
        var certificateReplaced = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=fd3ae6fi6fc00en0qkm9");
        var certificate = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=fd3cdv65706qam9l4cob");
        addMetrics(shard, certificateReplaced);
        addMetrics(shard, certificate);
        addResource(shard, "certificate", "fd3ae6fi6fc00en0qkm9", "my", true);
        addResource(shard, "certificate", "fd3cdv65706qam9l4cob", "my", false);
        assertEquals(Set.of("certificate", "name"), labelNames(SHARD_CERTIFICATE_MANAGER, ""));
        assertEquals(Set.of("certificate"), labelNames(SHARD_CERTIFICATE_MANAGER, "name=certificate.days_until_expiration"));
    }

    private Set<String> labelNames(int numId, String selectors) {
        var handler = new LabelNamesHandler(shardResolver, referenceResolver, resourceFinder);
        var response = handler.labelNames(TLabelNamesRequest.newBuilder()
                .setShardId(numId)
                .addAllSelectors(LabelSelectorConverter.selectorsToProto(SelectorsFormat.parse(selectors)))
                .build())
                .join();
        assertEquals(response.getStatusMessage(), EMetabaseStatusCode.OK, response.getStatus());
        return Set.copyOf(response.getNamesList());
    }
}
