package ru.yandex.solomon.coremon.meta.service.handler;

import java.util.Map;
import java.util.Set;

import org.junit.Test;

import ru.yandex.solomon.labels.LabelsFormat;
import ru.yandex.solomon.labels.protobuf.LabelSelectorConverter;
import ru.yandex.solomon.labels.query.SelectorsFormat;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metabase.api.protobuf.TLabelValuesRequest;
import ru.yandex.solomon.util.labelStats.LabelStatsConverter;
import ru.yandex.solomon.util.labelStats.LabelValuesStats;

import static org.junit.Assert.assertEquals;

/**
 * @author Vladimir Gordiychuk
 */
public class LabelValuesHandlerTest extends HandlerTestBase {

    @Test
    public void labelValuesEmpty() {
        {
            var stats = labelValues(SHARD_CERTIFICATE_MANAGER, "certificate=*");
            assertEquals(Map.of(), stats.getStatsByLabelKey());
        }
        {
            var stats = labelValues(SHARD_CERTIFICATE_MANAGER, "certificate=abc");
            assertEquals(Map.of(), stats.getStatsByLabelKey());
        }
        {
            var stats = labelValues(SHARD_COMPUTE, "");
            assertEquals(Map.of(), stats.getStatsByLabelKey());
        }
        {
            var stats = labelValues(SHARD_CERTIFICATE_MANAGER, "host=*");
            assertEquals(Map.of(), stats.getStatsByLabelKey());
        }
        {
            var stats = labelValues(SHARD_CERTIFICATE_MANAGER, "", "hdd-00");
            assertEquals(Map.of(), stats.getStatsByLabelKey());
        }
    }

    @Test
    public void labelValuesMetricWithoutResourceReference() {
        var shard = shardResolver.resolveShard(SHARD_CERTIFICATE_MANAGER);
        var usage = LabelsFormat.parse("name=quota.certificates_count.usage, host=cluster");
        var limit = LabelsFormat.parse("name=quota.certificates_count.limit, host=cluster");
        addMetrics(shard, usage, limit);
        {
            var stats = labelValues(SHARD_CERTIFICATE_MANAGER, "name=quota.certificates_count.usage");
            assertEquals(Set.of("cluster"), values("host", stats));
            assertEquals(Set.of("quota.certificates_count.usage"), values("name", stats));
        }
        {
            var stats = labelValues(SHARD_CERTIFICATE_MANAGER, "name=quota.certificates_count.us*");
            assertEquals(Set.of("cluster"), values("host", stats));
            assertEquals(Set.of("quota.certificates_count.usage"), values("name", stats));
        }
        {
            var stats = labelValues(SHARD_CERTIFICATE_MANAGER, "");
            assertEquals(Set.of("cluster"), values("host", stats));
            assertEquals(Set.of("quota.certificates_count.usage", "quota.certificates_count.limit"), values("name", stats));
        }
        {
            var stats = labelValues(SHARD_CERTIFICATE_MANAGER, "", "limit");
            assertEquals(Set.of(), values("host", stats));
            assertEquals(Set.of("quota.certificates_count.limit"), values("name", stats));
        }
    }

    @Test
    public void labelValuesWhenResourceNameUnknown() {
        var shard = shardResolver.resolveShard(SHARD_CERTIFICATE_MANAGER);
        var certificate = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=fd3cdv65706qam9l4cob");
        addMetrics(shard, certificate);
        {
            var stats = labelValues(SHARD_CERTIFICATE_MANAGER, "name=certificate.days_until_expiration, certificate=fd3cdv65706qam9l4cob");
            assertEquals(Set.of("fd3cdv65706qam9l4cob"), values("certificate", stats));
            assertEquals(Set.of("certificate.days_until_expiration"), values("name", stats));
        }
        {
            var stats = labelValues(SHARD_CERTIFICATE_MANAGER, "", "expiration");
            assertEquals(Set.of(), values("certificate", stats));
            assertEquals(Set.of("certificate.days_until_expiration"), values("name", stats));
        }
        {
            var stats = labelValues(SHARD_CERTIFICATE_MANAGER, "", "fd3cdv65706q");
            assertEquals(Set.of("fd3cdv65706qam9l4cob"), values("certificate", stats));
            assertEquals(Set.of(), values("name", stats));
        }
        {
            var stats = labelValues(SHARD_CERTIFICATE_MANAGER, "certificate=fd3cdv65706qam9l4cob");
            assertEquals(Set.of("fd3cdv65706qam9l4cob"), values("certificate", stats));
            assertEquals(Set.of("certificate.days_until_expiration"), values("name", stats));
        }
        {
            var stats = labelValues(SHARD_CERTIFICATE_MANAGER, "name=certificate.days_until_expiration, certificate=*");
            assertEquals(Set.of("fd3cdv65706qam9l4cob"), values("certificate", stats));
            assertEquals(Set.of("certificate.days_until_expiration"), values("name", stats));
        }
        {
            var stats = labelValues(SHARD_CERTIFICATE_MANAGER, "name=certificate.days_until_expiration");
            assertEquals(Set.of("fd3cdv65706qam9l4cob"), values("certificate", stats));
            assertEquals(Set.of("certificate.days_until_expiration"), values("name", stats));
        }
        {
            var stats = labelValues(SHARD_CERTIFICATE_MANAGER, "");
            assertEquals(Set.of("fd3cdv65706qam9l4cob"), values("certificate", stats));
            assertEquals(Set.of("certificate.days_until_expiration"), values("name", stats));
        }
        {
            var stats = labelValues(SHARD_CERTIFICATE_MANAGER, "certificate!=fd3cdv65706qam9l4cob");
            assertEquals(Set.of(), values("certificate", stats));
            assertEquals(Set.of(), values("name", stats));
        }
    }

    @Test
    public void labelValuesByNameBeforeMetabase() {
        var shard = shardResolver.resolveShard(SHARD_CERTIFICATE_MANAGER);
        var certificate = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=fd3cdv65706qam9l4cob");
        addMetrics(shard, certificate);
        addResource(shard, "certificate", "fd3cdv65706qam9l4cob", "my");
        {
            var stats = labelValues(SHARD_CERTIFICATE_MANAGER, "certificate=my");
            assertEquals(Set.of("certificate.days_until_expiration"), values("name", stats));
            assertEquals(Set.of("my"), values("certificate", stats));
        }
        {
            var stats = labelValues(SHARD_CERTIFICATE_MANAGER, "certificate=*");
            assertEquals(Set.of("certificate.days_until_expiration"), values("name", stats));
            assertEquals(Set.of("my"), values("certificate", stats));
        }
        {
            var stats = labelValues(SHARD_CERTIFICATE_MANAGER, "name=certificate.days_until_expiration, certificate=fd3cdv65706qam9l4cob");
            assertEquals(Set.of("certificate.days_until_expiration"), values("name", stats));
            assertEquals(Set.of("my"), values("certificate", stats));
        }
    }

    @Test
    public void labelValuesByNameAfterMetabase() {
        var shard = shardResolver.resolveShard(SHARD_CERTIFICATE_MANAGER);
        var certificate = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=fd3cdv65706qam9l4cob");
        addMetrics(shard, certificate);
        addResource(shard, "certificate", "fd3cdv65706qam9l4cob", "mon-cert");
        {
            var stats = labelValues(SHARD_CERTIFICATE_MANAGER, "name=certificate.days_until_expiration");
            assertEquals(Set.of("certificate.days_until_expiration"), values("name", stats));
            assertEquals(Set.of("mon-cert"), values("certificate", stats));
        }
        {
            var stats = labelValues(SHARD_CERTIFICATE_MANAGER, "");
            assertEquals(Set.of("certificate.days_until_expiration"), values("name", stats));
            assertEquals(Set.of("mon-cert"), values("certificate", stats));
        }
        {
            var stats = labelValues(SHARD_CERTIFICATE_MANAGER, "name=quota.certificates_count.usage");
            assertEquals(Set.of(), values("name", stats));
            assertEquals(Set.of(), values("certificate", stats));
        }
        {
            var stats = labelValues(SHARD_CERTIFICATE_MANAGER, "", "mon");
            assertEquals(Set.of(), values("name", stats));
            assertEquals(Set.of("mon-cert"), values("certificate", stats));
        }
    }

    @Test
    public void labelValuesMultipleReference() {
        var shard = shardResolver.resolveShard(SHARD_COMPUTE);
        var diskLatency = LabelsFormat.parse("name='disk_write_latency', resource_id='epdn6uas6ujoonvd99ck', bin='100', device='epdplhd148habqj0np92'");

        addMetrics(shard, diskLatency);
        addResource(shard, "vm", "epdn6uas6ujoonvd99ck", "solomon-stockpile-sas-01");
        addResource(shard, "disk", "epdplhd148habqj0np92", "hdd-01");
        {
            var selectors = Set.of(
                    "",
                    "name=disk_write_latency, resource_id=solomon-stockpile-sas-01",
                    "name=disk_write_latency, resource_id=*, device=hdd-01",
                    "name=disk_write_latency, device=hdd-01",
                    "name=disk_write_latency, resource_id=*, device=epdplhd148habqj0np92",
                    "name=disk_write_latency, device=epdplhd148habqj0np92",
                    "name=disk_write_latency, resource_id=epdn6uas6ujoonvd99ck"
            );
            for (String selector : selectors) {
                System.out.println(selector);
                var stats = labelValues(SHARD_COMPUTE, selector);
                assertEquals(Set.of("disk_write_latency"), values("name", stats));
                assertEquals(Set.of("solomon-stockpile-sas-01"), values("resource_id", stats));
                assertEquals(Set.of("hdd-01"), values("device", stats));
            }
        }

        {
            var stats = labelValues(SHARD_COMPUTE, "", "hdd");
            assertEquals(Set.of(), values("name", stats));
            assertEquals(Set.of(), values("resource_id", stats));
            assertEquals(Set.of("hdd-01"), values("device", stats));
        }
        {
            var stats = labelValues(SHARD_COMPUTE, "name=*", "stockpile");
            assertEquals(Set.of(), values("name", stats));
            assertEquals(Set.of("solomon-stockpile-sas-01"), values("resource_id", stats));
            assertEquals(Set.of(), values("device", stats));
        }
        {
            var stats = labelValues(SHARD_COMPUTE, "name=*", "not_exist");
            assertEquals(Set.of(), values("name", stats));
            assertEquals(Set.of(), values("resource_id", stats));
            assertEquals(Set.of(), values("device", stats));
        }
        {
            var stats = labelValues(SHARD_COMPUTE, "name=*", "latency");
            assertEquals(Set.of("disk_write_latency"), values("name", stats));
            assertEquals(Set.of(), values("resource_id", stats));
            assertEquals(Set.of(), values("device", stats));
        }
        {
            var stats = labelValues(SHARD_COMPUTE, "device=hdd-01", "latency");
            assertEquals(Set.of("disk_write_latency"), values("name", stats));
            assertEquals(Set.of(), values("resource_id", stats));
            assertEquals(Set.of(), values("device", stats));
        }
    }

    @Test
    public void labelValuesByNotName() {
        var shard = shardResolver.resolveShard(SHARD_COMPUTE);
        var diskLatencyOne = LabelsFormat.parse("name='disk_write_latency', resource_id='epdn6uas6ujoonvd99ck', bin='100', device='epdplhd148habqj0np92'");
        var diskLatencyTwo = LabelsFormat.parse("name='disk_write_latency', resource_id='epdhh61pg5vnviq7sfkd', bin='100', device='epd9629qg0o6offba1m7'");

        addMetrics(shard, diskLatencyOne);
        addMetrics(shard, diskLatencyTwo);
        addResource(shard, "vm", "epdn6uas6ujoonvd99ck", "solomon-stockpile-sas-01");
        addResource(shard, "vm", "epdhh61pg5vnviq7sfkd", "solomon-stockpile-sas-16");
        addResource(shard, "disk", "epdplhd148habqj0np92", "hdd-01");
        addResource(shard, "disk", "epd9629qg0o6offba1m7", "hdd-16");

        {
            var stats = labelValues(SHARD_COMPUTE, "name=disk_write_latency, device!=hdd-01");
            assertEquals(Set.of("hdd-16"), values("device", stats));
        }
        {
            var stats = labelValues(SHARD_COMPUTE, "name=disk_write_latency, resource_id=*, device!=hdd-01");
            assertEquals(Set.of("hdd-16"), values("device", stats));
        }
        {
            var stats = labelValues(SHARD_COMPUTE, "name=disk_write_latency, device!=hdd-16");
            assertEquals(Set.of("hdd-01"), values("device", stats));
        }
        {
            var stats = labelValues(SHARD_COMPUTE, "name=disk_write_latency, resource_id!=solomon-stockpile-sas-16, device!=hdd-16");
            assertEquals(Set.of("hdd-01"), values("device", stats));
            assertEquals(Set.of("solomon-stockpile-sas-01"), values("resource_id", stats));
        }
        {
            var stats = labelValues(SHARD_COMPUTE, "name=disk_write_latency, resource_id!=solomon-stockpile-sas-0*");
            assertEquals(Set.of("solomon-stockpile-sas-16"), values("resource_id", stats));
            assertEquals(Set.of("hdd-16"), values("device", stats));
        }
    }

    @Test
    public void labelValuesIgnoreReplacedMetrics() {
        var shard = shardResolver.resolveShard(SHARD_CERTIFICATE_MANAGER);
        var certificateReplaced = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=fd3ae6fi6fc00en0qkm9");
        var certificate = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=fd3cdv65706qam9l4cob");
        addMetrics(shard, certificateReplaced);

        addResource(shard, "certificate", "fd3ae6fi6fc00en0qkm9", "my", true);
        addResource(shard, "certificate", "fd3cdv65706qam9l4cob", "my", false);

        {
            var stats = labelValues(SHARD_CERTIFICATE_MANAGER, "certificate=my");
            assertEquals(Set.of(), values("certificate", stats));
        }
        {
            var stats = labelValues(SHARD_CERTIFICATE_MANAGER, "");
            assertEquals(Set.of(), values("certificate", stats));
        }
        {
            var stats = labelValues(SHARD_CERTIFICATE_MANAGER, "certificate=*");
            assertEquals(Set.of(), values("certificate", stats));
        }

        addMetrics(shard, certificate);
        {
            var stats = labelValues(SHARD_CERTIFICATE_MANAGER, "certificate=my");
            assertEquals(Set.of("my"), values("certificate", stats));
        }
        {
            var stats = labelValues(SHARD_CERTIFICATE_MANAGER, "");
            assertEquals(Set.of("my"), values("certificate", stats));
        }
        {
            var stats = labelValues(SHARD_CERTIFICATE_MANAGER, "certificate=*");
            assertEquals(Set.of("my"), values("certificate", stats));
        }
        {
            var stats = labelValues(SHARD_CERTIFICATE_MANAGER, "certificate!=my");
            assertEquals(Set.of(), values("certificate", stats));
        }
    }

    private static Set<String> values(String name, LabelValuesStats stats) {
        var valueStats = stats.getStatsByLabelKey().get(name);
        if (valueStats == null) {
            return Set.of();
        }
        return valueStats.getValues();
    }

    private LabelValuesStats labelValues(int numId, String selectors) {
        return labelValues(numId, selectors, "");
    }

    private LabelValuesStats labelValues(int numId, String selectors, String textSearch) {
        var handler = new LabelValuesHandler(shardResolver, referenceResolver, resourceFinder);
        var response = handler.labelValues(TLabelValuesRequest.newBuilder()
                .setShardId(numId)
                .addAllSelectors(LabelSelectorConverter.selectorsToProto(SelectorsFormat.parse(selectors)))
                .setTextSearch(textSearch)
                .build())
                .join();
        assertEquals(response.getStatusMessage(), EMetabaseStatusCode.OK, response.getStatus());
        return LabelStatsConverter.fromProto(response);
    }
}
